/***************************************************************************
                          system.cpp  -  description
                             -------------------
    begin                :   1 31 2003
    copyright            : (C) 2003 by k-kobayashi
    email                : k-kobayashi@isis.ocn.ne.jp
 ***************************************************************************/

/***************************************************************************
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 ***************************************************************************/
//using namespace std;
//#include <iostream>
#include <pwd.h>
#include <grp.h>
#include <stdlib.h>       // for getenv()
#include <time.h>         // for clock()

// include files for Qt
#include <qfileinfo.h>
#include <qdir.h>
#include <qregexp.h>

// include files for KDE
#include <klocale.h>


// application specific includes
#include "kroneconfig.h"
#include "option.h"
#include "system.h"

#define   KDVER "v0.3/"

extern Option*    g_Option;


/////////////////////////////////////////////////////////////////////////////
//
// KdeDir : Constructor
//
KdeDir::KdeDir( KFTYPE kftype )
{

  // initialize 'kroneko' application data directory
  // if local directory not found. create directory and copy data
  QString s;
  int dtype;
  switch( kftype ){
    case KFILE_DETAIL :      s = KDVER "detail";            dtype=2;  break;
    case KFILE_TMPL :        s = KDVER "templ";             dtype=1;  break;
    case KFILE_ICON :        s = KDVER "image/icon";        dtype=2;  break;
    case KFILE_CLOCK_CONF :
    case KFILE_CLOCK_IMAGE : s = KDVER "image/clock";       dtype=2;  break;
    default:                 s = KDVER "sound";             dtype=2;  break;
  }
  int cflag = 0;
  if ( dtype == 1 ){
    dirName = localkdedir() + kde_default( "data" ) + "kroneko/" + s;
    if ( ! QFile( dirName ).exists() ) cflag = 1;
  }
  if ( dtype == 2 || cflag == 1 ){
    // Add Install Position
    addPrefix( "/usr/local/kde/" );
    addPrefix( System::uHome() + "/" );
    QString hdir = getenv( "HOME" );
    addPrefix( hdir + "/" );
    QStringList sl = findDirs( "data", "kroneko/" + s );
    if ( cflag ){
      makeDir ( dirName );
      s = "cp -f " + sl[0] + "* " + dirName;
      Command cmd( s );
    }else{
      dirName = sl[0];
    }
  }
}

//
// KdeDir : Get file list
//
QStringList KdeDir::getEntry( const QString& filter )
{
  // getn file entry list filered by parameter.
  //
  QStringList get;
  get = QDir( dirName ).entryList( filter );

  QStringList ret;
  for( QStringList::Iterator i=get.begin(); i != get.end(); i++ ){
    ret.append( QString(*i).section( '.', 0, 0 ) );
  }
  return ret;
}

/////////////////////////////////////////////////////////////////////////////
//
// KdeFile : Constructor
//
KdeFile::KdeFile( const QString &name, KFTYPE kftype )
 : KdeDir( kftype )
{
  QString s;
  switch( kftype ){
    case KFILE_DETAIL :
    case KFILE_TMPL :
    case KFILE_CLOCK_CONF :  s = ".txt";  break;
    case KFILE_ICON :
    case KFILE_CLOCK_IMAGE : s = ".png";  break;
    default:                 s = ".wav";  break;
  }

  fullName = dirName + "/" + name + s;

}

//
// KdeFile : Read text data
//
QString KdeFile::readText()
{
  QString s("");
  QFile f( fullName );
  if ( f.open(IO_ReadOnly) ){
      QTextStream t( &f );        // use a text stream
      s = t.read();
      f.close();
  }else{
  }
  return s;
}

//
// KdeFile : Write text data
//
void KdeFile::writeText(const QString& str)
{
  QFile f( fullName );
  if ( f.open(IO_WriteOnly) ){
      QTextStream t( &f );        // use a text stream
      t << str;
      f.close();
  }else{
  }
}

//
// DetailFile : Read Detail Text and Convert
//
QString DetailFile::readDetail()
{
  QString s = KdeFile::readText();
  s.replace( QRegExp("\\|FILE\\|"), i18n( "File" ) );
  s.replace( QRegExp("\\|TASK\\|"), i18n( "Task" ) );
  s.replace( QRegExp("\\|TIME\\|"), i18n( "Time" ) );
  s.replace( QRegExp("\\|USER\\|"), i18n( "User" ) );
  s.replace( QRegExp("\\|COMMAND\\|"), i18n( "Command" ) );
  s.replace( QRegExp("\\|COMMENT\\|"), i18n( "Comment" ) );
  s.replace( QRegExp("\\|VARIABLE\\|"), i18n( "Variable" ) );
  s.replace( QRegExp("\\|NAME\\|"), i18n( "Name" ) );
  s.replace( QRegExp("\\|VALUE\\|"), i18n( "Value" ) );
  s.replace( QRegExp("\\|EXECUTE\\|"), i18n( "Execute Time" ) );
  s.replace( QRegExp("\\|PERIOD\\|"), i18n( "Period(Day)" ) );
  s.replace( QRegExp("\\|DELAY\\|"), i18n( "Delay(Minute)" ) );
  s.replace( QRegExp("\\|IDENT\\|"), i18n( "Identifier" ) );
  s.replace( QRegExp("\\|LASTEXE\\|"), i18n( "Last Execute" ) );
  s.replace( QRegExp("\\|JOBNO\\|"), i18n( "Job No" ) );
  s.replace( QRegExp("\\|QUEUE\\|"), i18n( "Queue Level" ) );
  s.replace( QRegExp("\\|MAIL\\|"), i18n( "Mail" ) );
  s.replace( QRegExp("\\|PROGRAM\\|"), i18n( "Program" ) );
  s.replace( QRegExp("\\|HOST\\|"), i18n( "Host" ) );
  s.replace( QRegExp("\\|EVENT\\|"), i18n( "Event" ) );

  s.replace( QRegExp("\\|STATINFO\\|"), i18n( "Statistic Information" ) );
  s.replace( QRegExp("\\|TEXE\\|"), i18n( "Total Execute:" ) );
  s.replace( QRegExp("\\|EXECNT\\|"), i18n( "Execute Count" ) );  
  s.replace( QRegExp("\\|TASKS\\|"), i18n( "Tasks:" ) );
  s.replace( QRegExp("\\|SYSCRON\\|"), i18n( "System Cron" ) );
  s.replace( QRegExp("\\|USRCRON\\|"), i18n( "User Cron" ) );
  s.replace( QRegExp("\\|JNO\\|"), "Job No" );
  s.replace( QRegExp("\\|QNO\\|"), "Queue No" );
  s.replace( QRegExp("\\|LOGINFO\\|"), i18n( "Statistic Information (LOG)" ) );
  s.replace( QRegExp("\\|TYPE\\|"), i18n( "Type" ) );
  s.replace( QRegExp("\\|CRONEXE\\|"), i18n( "Cron Executed Task" ) );
  s.replace( QRegExp("\\|ANACEXE\\|"), i18n( "Anacron Executed Task" ) );

  return s;
}


/////////////////////////////////////////////////////////////////////////////
//
// Command : Execute Command
//
Command::Command( const QString& str, bool detach )
{
  errStr = "";
  outStr = "";

  KShellProcess proc( System::uShell() );
  proc << str;
  if ( detach ){
    proc.start( KProcess::DontCare );
  }else{
    connect( &proc, SIGNAL(receivedStderr(KProcess*,char*,int)), this, SLOT(slotStderr(KProcess*,char*,int)));
    connect( &proc, SIGNAL(receivedStdout(KProcess*,char*,int)), this, SLOT(slotStdout(KProcess*,char*,int)));
    proc.start( KProcess::Block, KProcess::AllOutput );
    exitStatus = proc.exitStatus();
  }
}

//
// Command : slot-stdout
//
void Command::slotStdout(KProcess*,char* s, int n )
{
  outStr = outStr + QString().fromLocal8Bit((const char*)s,n) ;
}

//
// Command : slot-stderr
//
void Command::slotStderr(KProcess*,char* s, int n )
{
  errStr = errStr + QString().fromLocal8Bit((const char*)s,n) ;
}

/////////////////////////////////////////////////////////////////////////////
//
// System : get file accessible mode
//
int System::access( const QString& fname )
{
  int flg = 0;

  QFileInfo file( fname );
  if ( file.exists() ){
    flg = FILE_EXIST;
    if ( file.isReadable() ){
      flg = flg | CAN_READ;
    }
    if ( file.isWritable() ){
      flg = flg | CAN_WRITE;
    }
  }else{
    QFileInfo path( file.dirPath() );
    if ( path.exists() ){
      flg = PATH_EXIST;
      if ( path.isReadable() ){
        flg = flg | CAN_READ_DIR;
      }
      if ( path.isWritable() ){
        flg = flg | CAN_WRITE_DIR;
      }
    }
  }

  return flg;
}

//
// System : Get all users from 'passwd'
//
QStringList System::allUsers()
{

  QStringList ulist;
  struct passwd *pw;
  while( ( pw = getpwent() ) != NULL ) ulist.append( pw->pw_name );
  endpwent();
  return ulist;
}

//
// System : Get 'cron' allow users
//
QStringList System::cronUsers()
{

  QStringList ulist;
  if ( access( g_Option->cronFileAllow ) & FILE_EXIST ){
    if ( access( g_Option->cronFileAllow ) & CAN_READ ){
      ulist = getUsers( g_Option->cronFileAllow );
    }else{
      ulist = allUsers();
    }
  }else if ( access( g_Option->cronFileDeny ) & FILE_EXIST ){
    if ( access( g_Option->cronFileDeny ) & CAN_READ ){
      QStringList dlist;
      dlist = getUsers( g_Option->cronFileDeny );
      struct passwd *pw;
      while( ( pw = getpwent() ) != NULL ){
        if ( dlist.find( pw->pw_name  ) == dlist.end() ){
          ulist << pw->pw_name;
        }
      }
      endpwent();
    }else{
      ulist = allUsers();
    }
  }else{
#ifdef CRONONLYROOT
    ulist << "root";
#else
    ulist = allUsers();
#endif
  }
  return ulist;
}


//
// System : Get at allow users
//
QStringList System::atUsers()
{

  QStringList ulist;
  if ( access( g_Option->atFileAllow ) & FILE_EXIST ){
    if ( access( g_Option->atFileAllow ) & CAN_READ ){
      ulist = getUsers( g_Option->atFileAllow );
    }else{
      ulist = allUsers();
    }
  }else if ( access( g_Option->atFileDeny ) & FILE_EXIST ){
    if ( access( g_Option->atFileDeny ) & CAN_READ ){
      QStringList dlist;
      dlist = getUsers( g_Option->atFileDeny );
      struct passwd *pw;
      while( ( pw = getpwent() ) != NULL ){
        if ( dlist.find( pw->pw_name  ) == dlist.end() ){
          ulist << pw->pw_name;
        }
      }
      endpwent();
    }else{
      ulist = allUsers();
    }
  }else{
    ulist << "root";
  }
  return ulist;
}

bool System::cronEnable()
{
  bool ret = true;
  QStringList s=cronUsers();
  if ( s.find( uName() ) == s.end() ) ret = false;
  return ret;
}

bool System::atEnable()
{
  bool ret = true;
  if ( g_Option->useAt ){
    if ( uid() ){
      QStringList s=atUsers();
      if ( s.find( uName() ) == s.end() ) ret = false;
    }
  }else{
    ret = false;
  }
  return ret;
}

bool System::isCronWritable()
{
  bool ret = false;
  if ( access( g_Option->cronFileSys ) & ( CAN_WRITE | CAN_WRITE_DIR ) ) ret = true;
  return ret;    
}

int System::getAnacStatus()
{
  int ret = 0;
  if ( getCommand( "anacron" ) != "" ) {
    ret = ANAC_INSTALLED;
    if ( System::uid() == 0 ){
      ret = ret | ANAC_WRITABLE | ANAC_READABLE;
    }else{
      if ( access( g_Option->anacFileSys ) &  CAN_READ ) ret = ret | ANAC_READABLE;
    }
  }
  return ret;
}
//
// System : Get users list from file
//
QStringList System::getUsers( const QString& fname )
{
  QStringList list;

  QFile f( fname );

  if ( f.open(IO_ReadOnly) ) {   // file opened successfully
    QTextStream t( &f );        // use a text stream
    QString s = t.read();
    f.close();
    list = QStringList::split( '\n', s, false );
  }
  return list;
}

//
// Save Users
bool System::saveUsers( int flg, QStringList* user, const QString& afile, const QString& dfile )
{
  bool ret = true;
  QFile af( afile );
  QFile df( dfile );
  if ( access( afile ) & FILE_EXIST ){
    af.remove();
  }
  if ( access( dfile ) & FILE_EXIST ){
    df.remove();
  }
  if ( flg == ALLOW_USER ){
    if ( af.open(IO_WriteOnly) ){
      QTextStream t( &af );        // use a text stream
      for ( QStringList::Iterator it = user->begin(); it != user->end(); ++it ) {
        t << *it << '\n';
      }
      af.close();
    }else{
      ret = false;
    }
  }else if ( flg == DENY_USER ){
    QStringList al = allUsers();
    if ( df.open(IO_WriteOnly) ){
      QTextStream t( &df );        // use a text stream
      for ( QStringList::Iterator it = al.begin(); it != al.end(); ++it ) {
        if ( user->find( *it  ) == user->end() ){
          t << *it << '\n';
        }
      }
      df.close();
    }else{
      ret = false;
    }
  }

  return ret;

}

//
// System : Get command absolute path
//
QString System::getCommand( const QString& name )
{

  QString ret = "";

  for( int flg=0; flg<2; flg++ ){

    QString path;
    if ( !flg ) path = getenv( "PATH" );
    else        path = PSEARCH_PATH;

    QStringList plist = QStringList::split( ':', path, false );
    for ( QStringList::Iterator i=plist.begin(); i != plist.end(); i++ ){
      QString s = *i + "/" + name;
      if ( QFile::exists( s ) ){
        ret = s;
        break;
      }
    }
    if ( ret != "" ) break;
  }

  return ret;
}

//
// System : Get 'cron' version
//
QString System::cronVersion()
{

  QString ret( "" );
  
  Command cmd( CRON_VER + getCommand(  "crontab" ) + " -l" );

  if ( cmd.exitStatus == 0 ){
    ret = QString(cmd.outStr.section( "\n", CRON_VERLINE_START, CRON_VERLINE_END )).mid(2);
  }

  return ret; 
}

//
// System : Get 'anacron' version
//
QString System::anacVersion()
{

  QString ret( "" );

  Command cmd( getCommand(  "anacron" ) + " -V" );
  if ( cmd.exitStatus == 0 ){
    ret = cmd.outStr.section( "\n", ANAC_VERLINE_START, ANAC_VERLINE_END );
  }
  return ret;
}

//
// System : Get 'at' version
//
QString System::atVersion()
{

  QString ret( "" );

  Command cmd( getCommand(  "at" ) + " -V" );
  if ( cmd.exitStatus == 1 ){
    ret = cmd.errStr.section( "\n", AT_VERLINE_START, AT_VERLINE_END );
  }
  return ret;
}


static clock_t fClock;

//
// System : Set user id
//
void System::setUid( const QString& uname )
{
  struct passwd *pw;
  pw = getpwnam( uname );
  seteuid( pw->pw_uid );
  setegid( pw->pw_gid );
}

//
// System : Init Process Clock
//
void System::iniClock()
{
  fClock = clock();
}


//
// System : Get Process Clock
//
int System::getClock()
{
  return ( ( clock() - fClock ) / ( CLOCKS_PER_SEC / 1000 ) );
}

