/*
 kstrax
 Copyright (c) 2005,2006 Hitachi,Ltd.,
 Created by Satoru Moriya <s-moriya@sdl.hitachi.co.jp>
 
 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <limits.h>
#include <string.h>
#include <time.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include "kstrax.h"
#include "kstrax_buf.h"
#include "kstrax_ioc.h"

#define DEFAULT_NR_ENTRY 4000
#define DEFAULT_LIMIT_TIME 0 
#define NAME       "kstrax"
#define COPYRIGHT  "COPYRIGHT (C) HITACHI,LTD. 2005,2006"
#define INPUT_FILE "/dev/kstrax"

/*
 *  global variable
 */
extern char *optarg;
int kstrax_cflag = 0, kstrax_kflag = 0, kstrax_rflag = 0; 
static int kstrax_bflag = 0, kstrax_eflag = 0, kstrax_oflag = 0;
static int kstrax_pflag = 0, kstrax_sflag = 0, kstrax_tflag = 0;
static int kstrax_Tflag = 0;

/*
 *  function declaration
 */
static void kstrax_set_bin_filename(char *, const char *);
static void kstrax_set_text_filename(char *, char *, const char *);
static void kstrax_set_trace_syscall(char *);
static void kstrax_set_trace_pid(char *);
static void kstrax_usage(void);
static void kstrax_version(void);
static void kstrax_status(void);
static int kstrax_strtol(const char *);

int main(int argc, char *argv[])
{
	
	char c;
	char input_file[1024] = INPUT_FILE, output_file[1024] = "\0";
	int nr_entry = DEFAULT_NR_ENTRY, limit_time = DEFAULT_LIMIT_TIME; /* sec */

	while ((c = getopt(argc, argv, "+b::c:e:hko::p:r:s:St:T:v")) != EOF) {
		switch (c) {
		case 'b': /* output binary */
			kstrax_set_bin_filename(output_file, optarg);
			kstrax_bflag = 1;
			break;
		case 'c': /* count calls, times... print statistics information */
			if (strlen(optarg) > (KSTRAX_FNAME_MAX - 5)) {
				printf("Too long filename\n");
				exit(1);
			}
			strncpy(input_file, optarg, (KSTRAX_FNAME_MAX - 5));
			kstrax_cflag = 1;
			break;
		case 'e': /* system call specification in kernel space */
			kstrax_set_trace_syscall(optarg);
			kstrax_status();
			kstrax_eflag = 1;
			break;
		case 'h': /* help */
			kstrax_usage();
			goto out;
		case 'k': /* keep read index in kernel space */
			kstrax_kflag = 1;
			break;
		case 'o': /* set print file name */
			kstrax_set_text_filename(input_file, output_file, optarg);
			kstrax_oflag = 1;
			break;
		case 'p': /* pid specification in kernel space */
			kstrax_set_trace_pid(optarg);
			kstrax_status();
			kstrax_pflag = 1;
			break;
		case 'r': /* print text at RAW mode */
			if (strlen(optarg) > (KSTRAX_FNAME_MAX - 5)) {
				printf("Too long filename\n");
				exit(1);
			}
			strncpy(input_file, optarg, (KSTRAX_FNAME_MAX - 5));
			kstrax_rflag = 1;
			break;
		case 's': /* set buffer size in user space */
			nr_entry = kstrax_strtol(optarg);
			if (nr_entry < 0) {
				fprintf(stderr, 
					"-s has bad argument\n"
					"Invalid buffer size(%d)\n", nr_entry);
				exit(1);
			}
			kstrax_sflag = 1;
			break;
		case 'S': /* get kstrax status */
			kstrax_status();
			goto out;
		case 't': /* print text (nomal mode)*/
			if (strlen(optarg) > (KSTRAX_FNAME_MAX - 5)) {
				printf("Too long filename\n");
				exit(1);
			}
			strncpy(input_file, optarg, (KSTRAX_FNAME_MAX - 5));
			kstrax_tflag = 1;
			break;
		case 'T': /* timer */
			limit_time = kstrax_strtol(optarg);
			if (limit_time < 0) {
				fprintf(stderr, 
					"-T has bad argument\n"
					"Invalid limit time(%d)\n", limit_time);
				exit(1);
			}
			kstrax_Tflag = 1;
			break;
		case 'v': /* print version information*/
			kstrax_version();
			goto out;
		case '?':
			kstrax_usage();
			goto out;
		}
	}

	/* exclusion check */
	if (((kstrax_bflag || kstrax_kflag || kstrax_sflag || kstrax_Tflag) &&
	     (kstrax_cflag || kstrax_rflag || kstrax_tflag))) {
		fprintf(stderr, "option confliction\n");
		exit(1);
	}
	
	if(kstrax_bflag == 1) {
		kstrax_read(nr_entry, limit_time, input_file, output_file);
	} else if (kstrax_cflag == 1 || kstrax_rflag == 1 || kstrax_tflag == 1) {
		kstrax_print(input_file, output_file);
	} else if (kstrax_pflag != 1 && kstrax_eflag != 1) {
			kstrax_usage();
	}
 out:
	return 0;
}

static void kstrax_set_bin_filename(char *filename, const char *optarg)
{
       	if (optarg == NULL) {
		char host[HOST_NAME_MAX];
		time_t now;
		struct tm *my_time;
		
		gethostname(host, HOST_NAME_MAX);
		if (strlen(host) > (KSTRAX_FNAME_MAX - 17)) {
			printf("Too long filename\n");
			exit(1);
		}
		time(&now);
		my_time = localtime(&now);
		sprintf(filename, "%02d%02d%02d.%02d%02d_%s",
			my_time->tm_year-100, 
			my_time->tm_mon+1, 
			my_time->tm_mday, 
			my_time->tm_hour,
			my_time->tm_min, host);
	} else {
		/* real file name "filename_cpuid" */
		if (strlen(optarg) > (KSTRAX_FNAME_MAX - 5)) {
			printf("Too long filename\n");
			exit(1);
		}
		strncpy(filename, optarg, (KSTRAX_FNAME_MAX - 5)); 
	}
}

static void kstrax_set_text_filename(char *ifile, char *ofile, const char *optarg)
{
	if (kstrax_cflag == 0 && kstrax_rflag == 0 && 
	    kstrax_tflag == 0) {
		fprintf(stderr, 
			"Select First!!"
			"\"-c\", \"-r\" or \"-t\"\n");
		exit(1);
	}

	if (optarg == NULL) {
		sprintf(ofile, "%s.txt", ifile);
	} else {
		if (strlen(optarg) >= KSTRAX_FNAME_MAX) {
			printf("Too long filename\n");
			exit(1);
		}		
		strncpy(ofile, optarg, KSTRAX_FNAME_MAX);
	}
}

static void kstrax_set_trace_syscall(char *optarg)
{
	int i, fd;
	char type[6][8] = {"all", "file", "network", "ipc", "process", "signal"};

	if ((fd = open(INPUT_FILE, O_WRONLY)) < 0) {
		perror("kstrax_set_trace_syscall(open)");
		exit(1);
	}

	for (i = 0; i < 6; i++) {
		if(strcmp(optarg, type[i]) != 0)
			continue;

		if (ioctl(fd, KSTRAX_IOC_SYSCALL_SPEC, &i) < 0) {
			perror("kstrax_set_trace_syscall(ioctl)");
			exit(1);
		}
		break;
	}
	if (i == 6) {
		fprintf(stderr, "error:unknown syscall list\n");
		exit(1);
	}
	close(fd);
}

static void kstrax_set_trace_pid(char *optarg)
{
	int fd;
	pid_t pid;

	pid = kstrax_strtol(optarg);
	if ((fd = open(INPUT_FILE, O_WRONLY)) < 0) {
		perror("kstrax_set_trace_pid(open)");
		exit(1);
	}
	if (ioctl(fd, KSTRAX_IOC_PID_SPEC, &pid) < 0) {
		perror("kstrax_set_trace_pic(ioctl)");
		exit(1);		
	}
	close(fd);
}

static void kstrax_usage(void)
{
	printf("\nThis controls status of KSTRAX.\n");
	printf("Usage:\n\t%s [option(s)]\n\n", NAME);
	printf("<OPTION>\n");
	printf("-b [filename]\t\tOutput buffer data to binary file[name]\n");
	printf("-c [filename]\t\tPrint statistics information from binary file[name]\n");
	printf("-e [syscall type]\tSet tracing syscall type"
	       "[all/file/network/ipc/process/signal]\n");
	printf("-h\t\t\tPrint this message.\n");
	printf("-k\t\t\tNot clear kernel buffer\n");
	printf("-o [filename]\t\tSet output file name\n");
	printf("-p [pid]\t\tSet tracing pid[pid]\n");
	printf("-r\t\t\tPrint system call information on console"
	       " from binary file[name] at RAW mode\n");
	printf("-s [number of entry]\tSet buffer size to [number of entry]\n");
	printf("-S\t\t\tPrint trace status\n");
	printf("-t [filename]\t\tPrint system call information"
	       " from binary file[name] at text\n");
	printf("-T [sec]\t\tPrint to binary file for [sec]seconds\n");
	printf("-v\t\t\tPrint Version information\n\n");
	exit(0);
}

static void kstrax_version(void)
{
        printf("%s command version %s\n%s\n", NAME, KSTRAX_VERSION, COPYRIGHT);
	exit(0);
}

static void kstrax_status(void)
{
	int i;
	int fd;
	struct kstrax_status status;

	if ((fd = open(INPUT_FILE, O_WRONLY)) < 0) {
		perror("kstrax_status(open)");
		exit(1);
	}
	if (ioctl(fd, KSTRAX_IOC_STATUS, &status) < 0) {
		perror("kstrax_status(ioctl)");
		exit(1);		
	}
	printf("kernel buf entry :%d\n", status.nr_buf_entry);
	printf("tracing pid      :");
	for (i = 0; i < NR_TRACE_PROCESS_MAX; i++) {
		if (status.trace_pid[i] == -1) {
			if (i == 0) 
				printf("ALL");
			break;
		}
		printf("%d ", status.trace_pid[i]);
	}
	printf("\ntracing syscall  :");
	switch (status.trace_type) {
	case TRACE_ALL :
		printf("ALL\n");
		break;
	case TRACE_FILE :
		printf("file\n");
		break;
	case TRACE_NETWORK :
		printf("network\n");
		break;
	case TRACE_IPC :
		printf("ipc\n");
		break;
	case TRACE_PROCESS :
		printf("process\n");
		break;
	case TRACE_SIGNAL :
		printf("signal\n");
		break;
	default:
		fprintf(stderr, "unknown\n");
		exit(1);
	}
	close(fd);
}

static int kstrax_strtol(const char *string)
{
	int retval; 
	char *ptr;
	
	retval = strtol(string, &ptr, 10);
	if (*ptr == 0 && *string != 0) 
		return retval; /* success */
	
	fprintf(stderr, "bad argument(%s)\n", string);  
	exit(1);
}
