/*
 kstrax_buffer header for i386
 Copyright (c) 2006 Hitachi,Ltd.,
 Created by Satoru Moriya <s-moriya@sdl.hitachi.co.jp>
 
 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#ifndef __KSTRAX_BUF_ARCH_H__
#define __KSTRAX_BUF_ARCH_H__

#include <stdio.h>
#include <time.h>

#include "kstrax_ioc.h"

#define __IS_NO_PAIR if (buffer->sys_call_number == KS_EXIT || \
			 buffer->sys_call_number == KS_EXIT_GROUP)
#define __SET_UTIME(var) (var->utime)
/* max value of long long int */
#define PROC_TIME_MAX    ((unsigned long long)(1LL << 63) - 1)
#define PRINT_END        -1    

typedef long long proc_time_t;
typedef long long syscall_count_t;

typedef struct sys_call_info{
	pid_t pid;
	short sys_call_number;
	unsigned long time;
	unsigned long utime;
	long arg_1;
	union {
		long arg_2;
		long return_value;
	};
	long arg_3;
	long arg_4;
	long arg_5;
	long arg_6;
	unsigned long serial; /* unsigned long */
	long cpu;
} buffer_t;

typedef struct kstrax_list_entry {
	struct kstrax_list_entry *next;
	buffer_t buf;
} kst_list_entry_t;

typedef struct kstrax_basetime {
	long long g_t_d;         /* time of gettimeofday */
	long long t_s_c;         /* time of tsc */
	long hz;                /* cpu freq (hz) */
} basetime_t;

typedef struct kstrax_per_cpu {
	buffer_t buf;
	basetime_t basetime;
	int fd;
} per_cpu_t;

typedef struct kstrax_statistics {
	unsigned long count;
	unsigned long nr_error;
	unsigned long nr_no_entry;
	unsigned long nr_no_return;
	proc_time_t max_time;
	proc_time_t min_time;
	proc_time_t total_time;
} statistics_t;

typedef struct kstrax_print_data {
	buffer_t *buf;
	kst_list_entry_t **hash_table;
	kst_list_entry_t *free_list;
	statistics_t *stats;
	FILE *w_fp;
} print_data_t;

static void print_return_entry(const buffer_t *, FILE *);
static void print_pair_entry(const buffer_t *, const buffer_t *, FILE *);
static void add_pair_entry(const buffer_t *, const buffer_t *, statistics_t *);
static void add_no_pair_entry(const buffer_t *, statistics_t *);
static void list_add(kst_list_entry_t *, kst_list_entry_t **);
static kst_list_entry_t *list_del(kst_list_entry_t **);

extern long long kstrax_first_time;
extern long kstrax_sum_proc_time[];
extern long kstrax_sum_count[];

static inline basetime_t __get_basetime(const buffer_t *buf)
{
	unsigned long *high, *low;
	basetime_t retval;

	if (buf->pid != KS_BASETIME) {
		fprintf(stderr, "error:invalid header\n");
		exit(1);
	}
	
	retval.g_t_d = (long long)buf->arg_1 * 1000000 + (long long)buf->arg_2; /* mu sec */
	high = (unsigned long *)&buf->time;
	low  = (unsigned long *)&buf->utime;
	retval.t_s_c = ((long long)(*high) << 32) | (long long)(*low);
	retval.hz = buf->arg_3;
	
	return retval;
}

static inline void __tsc_to_gtd(per_cpu_t *kst_data)
{
	long long tmp_tsc, tmp_gtd;
	unsigned long *tmp_h, *tmp_l;

	tmp_h = (unsigned long *)&(kst_data->buf.time);
	tmp_l = (unsigned long *)&(kst_data->buf.utime);
	tmp_tsc = ((long long)(*tmp_h) << 32) | (long long)(*tmp_l);

	tmp_tsc -= kst_data->basetime.t_s_c;	
	tmp_tsc = (long long)(tmp_tsc * 10000 / kst_data->basetime.hz + 5) / 10;

	tmp_gtd = kst_data->basetime.g_t_d;
	tmp_gtd += tmp_tsc;

	/* gtd (gettimeofday)*/
	kst_data->buf.time  = (long)(tmp_gtd / (long long)1000000);
	kst_data->buf.utime = (long)(tmp_gtd % (long long)1000000);
}

static inline struct tm __set_time(const buffer_t *arg)
{
	return (*localtime((const time_t *)&arg->time));
}

static inline void __print_pair_entry_graph(const buffer_t *call,
					    const buffer_t *ret,
					    FILE *w_fp)
{
	int i;

	long long call_time = (long long)call->time;
	long long ret_time = (long long)ret->time;

	call_time *= 1000000;
	ret_time *= 1000000;
	
	call_time += (long long)call->utime;
	ret_time += (long long)ret->utime;
	
	if (kstrax_first_time == 0)
		kstrax_first_time = ret_time;	

	fprintf(w_fp, "%lld ", ret_time - kstrax_first_time);	
//#if 0
	for (i = 0; i < NR_syscalls; i++) {
		if (call->sys_call_number == i) {
			fprintf(w_fp, "%lld ", ret_time - call_time);
			kstrax_sum_proc_time[i] += ret_time - call_time;
			kstrax_sum_count[i]++;
		}
		else 
			fprintf(w_fp, "?0 ");
	}
	fprintf(w_fp, "\n");
//#endif
}

static inline void __print_syscall_count(FILE *fp,
					 syscall_count_t total,
					 syscall_count_t lost)
{
	fprintf(fp, "\n\ntotal syscalls      :: %lld\n"
		"total lost syscalls :: %lld\n",
		total / 2, lost / 2);
}

static inline void __print_syscall_name(FILE *fp, const buffer_t *buf)
{
	if (abs(buf->sys_call_number) == KS_IPC) {
		/* ipc */
		if (buf->arg_1 < 1 ||
		    buf->arg_1 > IPC_MAX || 
		    (buf->arg_1 > 4 && buf->arg_1 < 11) ||
		    (buf->arg_1 > 14 && buf->arg_1 < 21)) {
			fprintf(fp, "ipc");
		} else 
			fprintf(fp, "%s", ipc_syscall_name[abs(buf->arg_1)].name);
	} else 	if (abs(buf->sys_call_number) == KS_SOCKETCALL) {
		/* socketcall */
		if (buf->arg_1 < 1 || 
		    buf->arg_1 > SOCKETCALL_MAX) {
			fprintf(fp, "socketcall");
		} else 
			fprintf(fp, "%s", 
				socketcall_syscall_name[abs(buf->arg_1)].name);
	} else if (buf->sys_call_number == -KS_SIGRETURN) {
		/* sigreturn (return entry) */
		fprintf(fp, "sigreturn/rt_sigreturn");
	} else {
		/* default */
		fprintf(fp, "%s", syscall_name[abs(buf->sys_call_number)].name);
	}
	return;
}

static inline void __print_syscall_name_stats(FILE *fp, int index)
{
	if (index < NR_syscalls) {
		fprintf(fp, "%3d  ", index);
		fprintf(fp, "%-24s  ", syscall_name[index].name);
	} else if (NR_syscalls -1 < index && 
		   index < NR_syscalls + NR_SOCKETCALL - 1) {
		fprintf(fp, "%3d  ", KS_SOCKETCALL);
		fprintf(fp, "%-24s  ", 
			socketcall_syscall_name[index - NR_syscalls + 1].name);
	} else if (NR_syscalls + NR_SOCKETCALL  - 2 < index) {
		fprintf(fp, "%3d  ", KS_IPC);
		fprintf(fp, "%-24s  ", 
			ipc_syscall_name[index - NR_syscalls - NR_SOCKETCALL + 2].name);		
	}
}

static inline void __print_syscall_name_stats_graph(FILE *fp, int index)
{
	if (index < NR_syscalls) {
		fprintf(fp, "%d ", index);
		fprintf(fp, "%s ", syscall_name[index].name);
	} else if (NR_syscalls -1 < index && 
		   index < NR_syscalls + NR_SOCKETCALL - 1) {
		fprintf(fp, "%d ", KS_SOCKETCALL);
		fprintf(fp, "%s ", 
			socketcall_syscall_name[index - NR_syscalls + 1].name);
	} else if (NR_syscalls + NR_SOCKETCALL  - 2 < index) {
		fprintf(fp, "%d ", KS_IPC);
		fprintf(fp, "%s ", 
			ipc_syscall_name[index - NR_syscalls - NR_SOCKETCALL + 2].name);		

	}
}

static inline void __print_ipc_arg(FILE *fp, const buffer_t *buf)
{
	switch (buf->arg_1) {
	case KS_SEMOP:
		fprintf(fp, "0x%lx, 0x%lx, 0x%lx, 0x%lx",
			buf->arg_2, buf->arg_5, buf->arg_3, (unsigned long)0);
		break;
	case KS_SEMGET:
		fprintf(fp, "0x%lx, 0x%lx, 0x%lx",
			buf->arg_2, buf->arg_3, buf->arg_4);
		break;
	case KS_SEMCTL:
		fprintf(fp, "0x%lx, 0x%lx, 0x%lx", // necessary to modify
			buf->arg_2, buf->arg_3, buf->arg_4);
		break;
	case KS_SEMTIMEDOP:
		fprintf(fp, "0x%lx, 0x%lx, 0x%lx, 0x%lx",
			buf->arg_2,  buf->arg_5, buf->arg_3, buf->arg_6);
		break;
	case KS_MSGSND:
		fprintf(fp, "0x%lx, 0x%lx, 0x%lx, 0x%lx",
			buf->arg_2, buf->arg_5, buf->arg_3, buf->arg_4);
		break;
	case KS_MSGRCV:
		fprintf(fp, "0x%lx, 0x%lx, 0x%lx, 0x%lx, 0x%lx",
			buf->arg_2, buf->arg_5, buf->arg_3,
			buf->arg_6, buf->arg_4);
		break;
	case KS_MSGGET:
		fprintf(fp, "0x%lx, 0x%lx", buf->arg_2, buf->arg_3);
		break;
	case KS_MSGCTL:
		fprintf(fp, "0x%lx, 0x%lx, 0x%lx",
			buf->arg_2, buf->arg_3, buf->arg_5);
		break;
	case KS_SHMAT:
		fprintf(fp, "0x%lx, 0x%lx, 0x%lx, 0x%lx",
			buf->arg_2, buf->arg_5, buf->arg_3, buf->arg_4);
		break;
	case KS_SHMDT:
		fprintf(fp, "0x%lx", buf->arg_5);
		break;
	case KS_SHMGET:
		fprintf(fp, "0x%lx, 0x%lx, 0x%lx",
			buf->arg_2, buf->arg_3, buf->arg_4);
		break;
	case KS_SHMCTL:
		fprintf(fp, "0x%lx, 0x%lx",
			buf->arg_2, buf->arg_5);
		break;
	default:
		fprintf(fp, "0x%lx, 0x%lx, 0x%lx, 0x%lx, 0x%lx, 0x%lx", 
			buf->arg_1, buf->arg_2, buf->arg_3, buf->arg_4,
			buf->arg_5, buf->arg_6);
	}
}

static inline void __print_syscall_arg(FILE *fp, const buffer_t *buf)
{
	int i;
	long *ptr;
	int num = buf->sys_call_number;
	
	fprintf(fp, "(");
	if (num < 0) {
		fprintf(fp, " --- ");
	} else if (num == KS_SOCKETCALL) {
		if (buf->arg_1 < 1 || buf-> arg_1 > SOCKETCALL_MAX)
			fprintf(fp, "0x%lx, 0x%lx", buf->arg_1, buf->arg_2);
		else 
			fprintf(fp, "0x%lx", buf->arg_2);
	} else if (num == KS_IPC) {
		__print_ipc_arg(fp, buf);
	} else {
		for(i = 0; i < syscall_name[num].args_number; i++) {
			if (i == 0) 
				ptr = (long *)&buf->arg_1;
			else
				fprintf(fp, ", ");
			
			fprintf(fp, "0x%lx", *ptr);
			ptr++;
		}
	}

	fprintf(fp, ")");
}

static inline kst_list_entry_t **__get_sigreturn_call_entry(kst_list_entry_t **entry1,
							    kst_list_entry_t **entry2,
							    buffer_t *buf)
{
	while (*entry1) {
		if ((*entry1)->buf.pid == buf->pid)
			break;
		else
			entry1 = &(*entry1)->next;
	}
	while (*entry2) {
		if ((*entry2)->buf.pid == buf->pid)
			break;
		else
			entry2 = &(*entry2)->next;
	}
	if (*entry1 == NULL && *entry2 == NULL)
		return NULL;
	if (*entry1 == NULL)
		return entry2;
	if (*entry2 == NULL)
		return entry1;
	/* *entry1 != NULL && *entry2 != NULL */
	if ((*entry1)->buf.serial < (*entry2)->buf.serial) {
		return entry2;
	}
	else {
		return entry1;
	}
}

static inline int __print_sigreturn_normal(kst_list_entry_t **entry1,
					   buffer_t *buf,
					   print_data_t *p_data)
{
	int hash = (-buf->sys_call_number) % NR_syscalls;
	kst_list_entry_t *del;
	kst_list_entry_t **call_entry;
	kst_list_entry_t **entry2 = &p_data->hash_table[KS_RT_SIGRETURN];
	kst_list_entry_t **free_list_head = &p_data->free_list;

	if (hash == KS_SIGRETURN) {
		call_entry = __get_sigreturn_call_entry(entry1, entry2, buf);

		if (call_entry == NULL) {
			if (kstrax_gflag == 0)
				print_return_entry(buf, p_data->w_fp);
			return PRINT_END;
		}
		if (kstrax_gflag == 0)
			print_pair_entry(&(*call_entry)->buf, buf, p_data->w_fp);
		else
			__print_pair_entry_graph(&(*call_entry)->buf, buf, 
						 p_data->w_fp);
		del = list_del(call_entry);
		list_add(del, free_list_head);
		return PRINT_END;
	}
	return 0;
}

static inline void __print_sigreturn_raw(buffer_t *buf,
					 print_data_t *p_data)
{
	if (buf->sys_call_number == -KS_SIGRETURN)
		fprintf(p_data->w_fp, "/%s", syscall_name[KS_RT_SIGRETURN].name);
}
static inline int __print_sigreturn_stats(kst_list_entry_t **entry1,
					  buffer_t *buf,
					  print_data_t *p_data)
{
	int hash = (-buf->sys_call_number) % NR_syscalls;
	kst_list_entry_t *del;
	kst_list_entry_t **call_entry;
	kst_list_entry_t **entry2 = &p_data->hash_table[KS_RT_SIGRETURN];
	kst_list_entry_t **free_list_head = &p_data->free_list;

	if (hash == KS_SIGRETURN) {
		call_entry = __get_sigreturn_call_entry(entry1, entry2, buf);

		if (call_entry == NULL) {
			add_no_pair_entry(buf, p_data->stats);
			return PRINT_END;
		}
		add_pair_entry(&(*call_entry)->buf, buf, p_data->stats);
		del = list_del(call_entry);
		list_add(del, free_list_head);
		return PRINT_END;
	}
	return 0;
}

static inline int __get_nr_stats(void)
{
	return (NR_syscalls + NR_SOCKETCALL + NR_IPC);
}

static inline int __get_stats_array_index(const buffer_t *buf)
{
	int number =  buf->sys_call_number;

	if (NR_syscalls < abs(number)) {
		fprintf(stderr, "Unknown syscall number : %d\n", number);
		return -1;
	}

	if (number == KS_SOCKETCALL) {
		/* socketcall call */
		if (0 < buf->arg_1 && buf->arg_1 < 18)
			number = NR_syscalls + buf->arg_1 - 1;
	} else if (number == -KS_SOCKETCALL) {
		/* socketcall return */
		if (0 < buf->arg_1 && buf->arg_1 < 18)
			number = -(NR_syscalls + buf->arg_1 - 1);
	} 
	else if (number == KS_IPC) {
		/* ipc call */
		if ((0 < buf->arg_1 && buf->arg_1 < 5) ||
		    (10 < buf->arg_1 && buf->arg_1 < 15) ||
		    (20 < buf->arg_1 && buf->arg_1 < 25))
			number = NR_syscalls + NR_SOCKETCALL + buf->arg_1 - 2;
	} else if (number == -KS_IPC) {
		/* ipc return */
		if ((0 < buf->arg_1 && buf->arg_1 < 5) ||
		    (10 < buf->arg_1 && buf->arg_1 < 15) ||
		    (20 < buf->arg_1 && buf->arg_1 < 25))
			number = -(NR_syscalls + NR_SOCKETCALL + buf->arg_1 - 2);
	}
	return number;
}

static inline proc_time_t __calc_processing_time(const buffer_t *call, 
						 const buffer_t *ret)
{
	proc_time_t call_time, ret_time;

	call_time = (proc_time_t)call->time * 1000000 + (proc_time_t)call->utime;
	ret_time  = (proc_time_t)ret->time * 1000000 + (proc_time_t)ret->utime;

	return (ret_time - call_time);
}

static inline void __print_proc_time(FILE *fp, proc_time_t p_time)
{
	fprintf(fp, "%6lld.%06u  ", p_time / 1000000,
		(unsigned int)p_time % 1000000);
}

static inline void __print_proc_time_graph(FILE *fp, proc_time_t p_time)
{
	fprintf(fp, "%lld.%06u ", p_time / 1000000,
		(unsigned int)p_time % 1000000);
	
}
#endif /* __KSTRAX_BUF_ARCH_H__ */
