/*
 kstrax_buffer
 Copyright (c) 2005,2007 Hitachi,Ltd.,
 Created by Satoru Moriya <satoru.moriya.br@hitachi.com>
 
 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <string.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/mman.h>
#include <pthread.h>
#include <limits.h>
#include <sys/time.h>

#include "kstrax.h"
#include "kstrax_buf.h"
#include "kstrax_ioc.h"
#include "syscall_name.h"
#include "kstrax_syscall_list.h"

#include "kstrax_buf_arch.h"

#define READ_FILE     1
#define WRITE_FILE    2
#define KS_FILE_END  -2
#define KS_FINISHED  -3
#define BIN_HEAD_ID 893
#define ERR_LIMIT   530
#define KS_FILENAME -11

#define kstrax_for_each_online_cpu(cpu, max_nr_cpu) \
for (cpu = 0; cpu < max_nr_cpu; cpu++)

typedef struct kstrax_thread_param {
	pthread_t th;
	int cpu_id;
	int input_fd;
	int output_fd;
	int nr_entry;
	int limit;
} tparam_t;

typedef struct kstrax_list_head {
	kst_list_entry_t *next;
} kst_list_head_t;

typedef struct kstrax_binary_header {
	int id;
	int nr_syscalls;
	char version[10];
	buffer_t buf_basetime;
} bin_header_t;

/*
 *  global variable
 */
static int module_file_handle;
static tparam_t *kstrax_tparam;
proc_time_t *kstrax_sum_proc_time;
syscall_count_t *kstrax_sum_count;

#ifdef _IA64_KSTRAX_
long kstrax_first_time = 0;
#else  /* _IA64_KSTRAX_ */
long long kstrax_first_time = 0;
#endif /* _IA64_KSTRAX_ */

/*---------------------------------------------------------------------------------
 *  signal handler 
 */
static void kstrax_sig_handler(int signum)
{
	int cpu;
	int flags;
	struct kstrax_status sparam;

	if (ioctl(module_file_handle, KSTRAX_IOC_SET_FORCE_WAKEUP)) {
		perror("kstrax_sig_handler(ioctl:force wakeup)");
		exit(1);
	}
	flags = fcntl(module_file_handle, F_GETFL);
	fcntl(module_file_handle, F_SETFL, flags|O_NONBLOCK);
	
	if (ioctl(module_file_handle, KSTRAX_IOC_STATUS, &sparam) < 0) {
		perror("kstrax_sig_handler(ioctl:get status)");
		exit(1);
	}
	kstrax_for_each_online_cpu(cpu, sparam.nr_cpu) {
		pthread_kill(kstrax_tparam[cpu].th, SIGUSR1);
	}

	fprintf(stdout, "terminating...\n");
	return;
}

static void kstrax_dummy_handler(int signum)
{
	return;
}

/*-------------------------------------------------------------------------------
 *  file operation
 */
/* open */
static int kstrax_file_open(const char *filename, int flag)
{
	int fd;

	if (flag == READ_FILE) {
		fd = open(filename, O_RDONLY);
	} else if (flag == WRITE_FILE) {
		fd = open(filename, O_WRONLY | O_CREAT | O_TRUNC, S_IRWXU);
	} else {
		fprintf(stderr, "Bad Argument : kstrax_file_open\n");
		exit(1);
	}
	
	if (fd < 0) {
		perror("kstrax_file_open(open)");
		exit(1);
	}
	return fd; /* file handler */
}

/*-------------------------------------------------------------------------------
 *  read system call information
 *  "-b" option
 */
static void read_syscall_per_cpu(tparam_t *);
static void output_to_bin_file(int, int, int, int);
static void format_bin_file(int, int);

void kstrax_read(int nr_entry, int limit_time, 
		  const char *input_f, const char *output_f)
{
	int cpu, pid;
	struct kstrax_status sparam;

	/* register signal action */
	signal(SIGINT,  &kstrax_sig_handler);
	signal(SIGALRM, &kstrax_sig_handler);
	signal(SIGUSR1, &kstrax_dummy_handler);

	module_file_handle = kstrax_file_open(input_f, READ_FILE); 

	/* set owner process */
	pid = getpid();
	if (ioctl(module_file_handle, KSTRAX_IOC_SET_OWNER, &pid) < 0) {
		perror("kstrax_read(ioctl:set owner)");
		exit(1);
	}
	
	/* reset read index */
	if (kstrax_kflag != 1) {
		if (ioctl(module_file_handle, KSTRAX_IOC_INIT_INDEX) < 0) {
			perror("kstrax_read(ioctl:init index)");
			exit(1);
		}
	}
	
 	/* read system status */
	if (ioctl(module_file_handle, KSTRAX_IOC_STATUS, &sparam) < 0) {
		perror("kstrax_read(ioctl:read status)");
		exit(1);
	}
	
	kstrax_tparam = (tparam_t *)malloc(sizeof(tparam_t) * sparam.nr_cpu);
	if (kstrax_tparam == NULL) {
		perror("kstrax_read(malloc)");
		exit(1);
	}

	/* fork per processor */
	/* create threads */
	kstrax_for_each_online_cpu(cpu, sparam.nr_cpu) {
		char s[1024];
		sprintf(s, "%s_%02d", output_f, cpu);
		kstrax_tparam[cpu].cpu_id    = cpu;
		kstrax_tparam[cpu].input_fd  = module_file_handle;
		kstrax_tparam[cpu].output_fd = kstrax_file_open(s, WRITE_FILE);
		kstrax_tparam[cpu].nr_entry  = nr_entry; 
		kstrax_tparam[cpu].limit     = limit_time;

		pthread_create(&kstrax_tparam[cpu].th, NULL, 
			       (void *)read_syscall_per_cpu, 
			       (void *)&kstrax_tparam[cpu]);
	}

	if (kstrax_Rflag == 0)
		pause();

	/* wait threads */
	kstrax_for_each_online_cpu(cpu, sparam.nr_cpu) {
		pthread_join(kstrax_tparam[cpu].th, NULL);
		close(kstrax_tparam[cpu].output_fd);
	}		

	/* clear owner process */
	if (ioctl(module_file_handle, KSTRAX_IOC_CLR_OWNER) < 0) {
		perror("kstrax_read(ioctl:clear owner)");
		exit(1);
	}

	close(module_file_handle);
}

/*
 *  this function is executed by each thread
 */
static void read_syscall_per_cpu(tparam_t *tparam)
{
	sigset_t mask_set;

	if (ioctl(tparam->input_fd, KSTRAX_IOC_SET_CPU, &tparam->cpu_id)) {
		perror("read_syscall_per_cpu(ioctl:set cpu affinity)");
		exit(1);
	}

	/* block signals */
	if (sigfillset(&mask_set) < 0) {
		perror("read_syscall_per_cpu(sigfillset)");
		exit(1);
	}
	if (sigdelset(&mask_set, SIGUSR1) < 0) {
		perror("read_syscall_per_cpu(sigdelset)");
		exit(1);
	}

	if (pthread_sigmask(SIG_SETMASK, &mask_set, NULL) != 0) {
		perror("read_syscall_per_cpu(pthread_sigmask)");
		exit(1);
	}

	output_to_bin_file(tparam->nr_entry, tparam->limit, 
			   tparam->input_fd, tparam->output_fd);
}

static void output_to_bin_file(int entry, int limit,
			       int read_handle, int write_handle)
{
	ssize_t request_size;
	buffer_t *buf;
		
	/* format write file */
	format_bin_file(read_handle, write_handle);
	
	buf = (buffer_t *)malloc(sizeof(buffer_t) * entry);
	if (buf == NULL) {
		perror("output_to_bin_file(malloc)");
		exit(1);
	}

	alarm(limit);

	/* read and then write */
	request_size = entry * sizeof(buffer_t);
	while (1) {
		int read_size, write_size;
		if ((read_size = read(read_handle, buf, request_size)) < 0) {
			perror("output_to_bin_file(read)");
			exit(1);
		}
		if ((write_size = write(write_handle, buf, read_size)) < 0) {
			perror("output_to_bin_file(write)");
			exit(1);
		}
		
		if (read_size != write_size) {
			fprintf(stderr, "Error write failed\n");
			exit(1);
		}

		if (read_size < request_size) break;
	}
	
	/* free user space buffer */
	free(buf);
}

static void format_bin_file(int read_handle, int write_handle)
{
	bin_header_t bin_head;
	
	/* init binary file header */
	bin_head.id = BIN_HEAD_ID;
	bin_head.nr_syscalls = kstrax_get_nr_syscalls();
	strncpy(bin_head.version, KSTRAX_VERSION, 10);
	if (ioctl(read_handle, KSTRAX_IOC_READ_BASETIME, 
		  &bin_head.buf_basetime) < 0) {
		perror("format_bin_file(ioctl:read basetime)");
		exit(1);
	}

	if (write(write_handle, &bin_head, sizeof(bin_header_t)) 
	    != sizeof(bin_header_t)) {
		perror("format_bin_file(write)");
		exit(1);
	}
}

/*--------------------------------------------------------------------------------
 *  print system call information
 *  "-c", "-r", "-t" option
 */
static int count_bin_file(const char *);
static void check_bin_header(per_cpu_t *);
static void marge_and_print(per_cpu_t *, int, FILE *);

void kstrax_print(const char *input_f, const char *output_f)
{
	int i;
	int nr_file;

	per_cpu_t *kst_data;
	FILE *w_fp;
	char filename[1024];

	nr_file = count_bin_file(input_f);

	kst_data = (per_cpu_t *)malloc(sizeof(per_cpu_t) * nr_file);
	if (kst_data == NULL) {
		perror("kstrax_print(malloc)");
		exit(1);
	}

	for (i = 0; i < nr_file; i++) {
		sprintf(filename, "%s_%02d", input_f, i);
		kst_data[i].fd = kstrax_file_open(filename, READ_FILE);
		
		check_bin_header(&kst_data[i]);
	}
	w_fp = fopen(output_f, "w");
	if (w_fp == NULL) {
		perror("kstrax_print(fopen)");
		exit(1);
	}
	
	marge_and_print(kst_data, nr_file, w_fp);

	fclose(w_fp);
	for (i = 0; i < nr_file; i++) {
		close(kst_data[i].fd);
	}
	free(kst_data);
}

static int count_bin_file(const char *input_f)
{
	char filename[1024];
	int nr_file = 0;
		
	while (1) {
		int success = 0;
		sprintf(filename, "%s_%02d", input_f, nr_file);
		success = open(filename, O_RDONLY);
		if (success < 0) {
			break;
		}
		close(success);
		nr_file++;
	}
	if (nr_file == 0) {
		perror("count_bin_file");
		exit(1);
	}
	return nr_file;
}

static void check_bin_header(per_cpu_t *kst_data)
{
	bin_header_t bin_head;

	/* check binary file header */
	if (read(kst_data->fd, &bin_head, sizeof(bin_header_t)) 
	    != sizeof(bin_header_t)) {
		perror("check_bin_header(read)");
		exit(1);
	}
	if (bin_head.id != BIN_HEAD_ID) {
		fprintf(stderr, "error:invalid header\n");
		exit(1);
	}
	kstrax_set_nr_syscalls(bin_head.nr_syscalls);

	if (strcmp(bin_head.version, KSTRAX_VERSION)) {
		fprintf(stderr, "Warning:Another version binary file\n");
	}
	kst_data->basetime = __get_basetime(&bin_head.buf_basetime);
}

/*
 *  print main routine
 */
static void read_entry(per_cpu_t *);
static int earliest_entry(const per_cpu_t *, int);
static unsigned long count_lost_syscall(unsigned long, unsigned long);
static void pre_print_normal(print_data_t *);
static void pre_print_raw(print_data_t *);
static void pre_print_stats(print_data_t *);
static void main_print_normal(print_data_t *);
static void main_print_raw(print_data_t *);
static void main_print_stats(print_data_t *);
static void post_print_normal(print_data_t *);
static void post_print_raw(print_data_t *);
static void post_print_stats(print_data_t *);
static void lost_syscall_normal(print_data_t *, unsigned long);
static void lost_syscall_raw(print_data_t *, unsigned long);
static void lost_syscall_stats(print_data_t *, unsigned long);

static void marge_and_print(per_cpu_t *kst_data, int nr_file, FILE *w_fp)
{
	int i;
	int min = 0;
	int option = 0;
	int nr_round = 0;
	int nr_syscalls = kstrax_get_nr_syscalls();
	unsigned long first_serial, last_serial = 0;
	syscall_count_t total_syscall = 0,  total_nr_lost = 0;
	print_data_t p_data;
	void (*pre_print[])(print_data_t *) = {pre_print_normal, 
					       pre_print_raw, 
					       pre_print_stats};
	void (*main_print[])(print_data_t *) = {main_print_normal,
						main_print_raw,
						main_print_stats};
	void (*post_print[])(print_data_t *) = {post_print_normal,
						post_print_raw,
						post_print_stats};
	void (*lost_syscall[])(print_data_t *, unsigned long) = {
		lost_syscall_normal,
		lost_syscall_raw,
		lost_syscall_stats};

	/* alloc counter */
	kstrax_sum_proc_time = 
		(proc_time_t *)malloc(nr_syscalls * sizeof(proc_time_t));
	kstrax_sum_count =
		(syscall_count_t *)malloc(nr_syscalls * sizeof(syscall_count_t));
	for (i = 0; i < nr_syscalls; i++) {
		kstrax_sum_proc_time[i] = 0;
		kstrax_sum_count[i] = 0;
	}

	if (kstrax_rflag == 1) {
		option = 1;
	} else if (kstrax_cflag == 1) {
		option = 2;
	}

	p_data.w_fp = w_fp;
	(*pre_print[option])(&p_data);
	for (i = 0; i < nr_file; i++) {
		read_entry(&kst_data[i]);
		__tsc_to_gtd(&kst_data[i]);
	}

	/* record first serial */
	min = earliest_entry(kst_data, nr_file);	
	if (min == KS_FINISHED)
		goto out;
	first_serial = kst_data[min].buf.serial;
	last_serial = first_serial - 1;

	/* main routine of print data */
	while (1) {
		unsigned long nr_lost = 0;

		min = earliest_entry(kst_data, nr_file);
		if (min == KS_FINISHED)
			break;

		if ((nr_lost = 
		     count_lost_syscall(last_serial, kst_data[min].buf.serial)) 
		    != 0) {
			total_nr_lost += (syscall_count_t)nr_lost;
			(*lost_syscall[option])(&p_data, nr_lost);
		}
		
		last_serial = kst_data[min].buf.serial;
		p_data.buf = &kst_data[min].buf;
		(*main_print[option])(&p_data);
		
		read_entry(&kst_data[min]);

		/* print filename */
		if (kst_data[min].buf.pid == KS_FILENAME) {
			if (kstrax_cflag == 0 && kstrax_gflag == 0)
				fprintf(p_data.w_fp, "file descriptor %d --> ", 
					kst_data[min].buf.sys_call_number);
			while (kst_data[min].buf.pid == KS_FILENAME) {
				if (kstrax_cflag == 0 && kstrax_gflag == 0)
					fprintf(p_data.w_fp, "%s", 
						(char *)&kst_data[min].buf.time);
				read_entry(&kst_data[min]);
			}
			if (kstrax_cflag == 0)
				fprintf(p_data.w_fp, "\n");
		}

		__tsc_to_gtd(&kst_data[min]);

		if (last_serial == ULONG_MAX)
			nr_round++;
	}

	(*post_print[option])(&p_data);

	if (nr_round) {
		total_syscall = (syscall_count_t)((nr_round - 1) * ULONG_MAX + 
					    (ULONG_MAX - first_serial) + last_serial);
	} else {
		total_syscall = (syscall_count_t)(last_serial - first_serial + 1);
	}

	if (kstrax_gflag == 0) {
		__print_syscall_count(p_data.w_fp, total_syscall, total_nr_lost);
	} else {
		if (kstrax_cflag == 0) {
			__print_syscall_count_graph(p_data.w_fp);
		}
	}

 out:
	free(kstrax_sum_proc_time);
	free(kstrax_sum_count);
}

static void read_entry(per_cpu_t *kst_data)
{
	int retval;

	retval = read(kst_data->fd, &kst_data->buf, sizeof(buffer_t));
	if (retval < 0) {
		perror("read_entry(read)");
		exit(1);
	} else if (retval == 0) {
		kst_data->buf.pid = KS_FILE_END;
	} 
}

static int earliest_entry(const per_cpu_t *kst_data, int nr_file)
{
	int i;
	int min = KS_FINISHED;

	for (i = 0; i < nr_file; i++) {
		if (kst_data[i].buf.pid != KS_FILE_END) {
			min = i;
			break;
		}
	}		
	/* no data for print */
	if (min == KS_FINISHED) {
		goto out;
	}
	
	for (i = min+1; i < nr_file; i++) {
		if (kst_data[i].buf.pid == KS_FILE_END) continue;

		if (kst_data[min].buf.serial > kst_data[i].buf.serial)
			min = i;
#if 0 /* thinking..... */
		if (kst_data[min].buf.time > kst_data[i].buf.time) {
			min = i;
		} else if (kst_data[min].buf.time == kst_data[i].buf.time) {
			if (kst_data[min].buf.utime > kst_data[i].buf.utime) {
				min = i;
			} else if (kst_data[min].buf.utime == kst_data[i].buf.utime) {
				if (kst_data[min].buf.serial > kst_data[i].buf.serial)
					min = i;
			}
		}
#endif 
	}
 out:
	return min;
}

static unsigned long int count_lost_syscall(unsigned long last, unsigned long now)
{
	unsigned long int nr_lost = 0;

	if (now < last) {
		nr_lost = (ULONG_MAX - last + 1) + now;
	} else {
		nr_lost = now - last - 1;
	}
	return nr_lost;
}

/*---------------------------------------------------------------------------------
 *  pre processing for each option
 */
static kst_list_entry_t **alloc_hash_table(int);
static statistics_t *alloc_statistics_array(int);

/* "-t" */
static void pre_print_normal(print_data_t *p_data)
{
	int nr_syscalls = kstrax_get_nr_syscalls();

	p_data->free_list = NULL;
	p_data->hash_table = alloc_hash_table(nr_syscalls);
	p_data->stats_array = NULL;

	if (kstrax_gflag == 0) {
		fprintf(p_data->w_fp, 
			"------------------------------------------------"
			"----------------------------------\n");
		fprintf(p_data->w_fp, 
			" pid     start            end             system_call"
			" \"name,argument,return_value\"\n");
		fprintf(p_data->w_fp, 
			"------------------------------------------------"
			"----------------------------------\n");
	} else  {
		int i;
		fprintf(p_data->w_fp, "name ");
		for (i = 0; i < nr_syscalls; i++) {
			fprintf(p_data->w_fp, "%s ", syscall_name[i].name);
		}
		fprintf(p_data->w_fp, "\n");
	}
	
}

/* "-r" */
static void pre_print_raw(print_data_t *p_data)
{
	p_data->free_list = NULL;
	p_data->hash_table = NULL;
	p_data->stats_array = NULL;

	fprintf(p_data->w_fp, 
		"------------------------------------------------"
		"----------------------------------------------\n");
	fprintf(p_data->w_fp, 
		"        serial  cpu   pid      time          system_call"
		" \"number,name,argument/return_value\"\n");
	fprintf(p_data->w_fp, 
		"------------------------------------------------"
		"----------------------------------------------\n");	
}
/* "-c" */
static void pre_print_stats(print_data_t *p_data)
{
	int nr_stats_entry = __get_nr_syscalls();
	int nr_syscalls = kstrax_get_nr_syscalls();

	p_data->free_list = NULL;
	p_data->hash_table = alloc_hash_table(nr_syscalls);
	p_data->stats_array = alloc_statistics_array(nr_stats_entry);

	if (kstrax_gflag == 0) {
		fprintf(p_data->w_fp, 
			"------------------------------------------------"
			"------------------------------------------------"
			"-------------------------------\n");
		fprintf(p_data->w_fp, 
			" ID  NAME                        count        average"
			"            max            min          total   error"
			"  no call   no return\n");
		fprintf(p_data->w_fp, 
			"------------------------------------------------"
			"------------------------------------------------"
			"-------------------------------\n");
	}
}

static kst_list_entry_t **alloc_hash_table(int nr_entry)
{
	int i;
	kst_list_entry_t **hash_table;

	hash_table = (kst_list_entry_t **)malloc(
		sizeof(kst_list_entry_t *) * nr_entry);
	if (hash_table == NULL) {
		perror("alloc_hash_table(malloc)");
		exit(1);
	}
	for (i = 0; i < nr_entry; i++) {
		hash_table[i] = NULL;
	}
	return hash_table;
}

static statistics_t *alloc_statistics_array(int nr_entry)
{
	int i;
	statistics_t *stats;

	stats = (statistics_t *)malloc(sizeof(statistics_t) * nr_entry);
	if (stats == NULL) {
		perror("alloc_statistics_array(malloc)");
		exit(1);
	}
	for (i = 0; i < nr_entry; i++) {
		stats[i].count    = 0;
		stats[i].nr_error = 0;
		stats[i].nr_no_entry  = 0;
		stats[i].nr_no_return = 0;
		stats[i].max_time     = 0;
		stats[i].min_time     = PROC_TIME_MAX;
		stats[i].total_time   = 0;
	}
	return stats;
}

/*--------------------------------------------------------------------------------
 *  main processing for each option
 */
static void print_call_entry(const buffer_t *, FILE *);
static void print_return_entry(const buffer_t *, FILE *);
static void print_pair_entry(const buffer_t *, const buffer_t *, FILE *);
static void print_call_entry_raw(FILE *, buffer_t *);
static void print_return_entry_raw(FILE *, buffer_t *);
static kst_list_entry_t *create_hash_entry(const buffer_t *, kst_list_entry_t **);
static void add_pair_entry(const buffer_t *, const buffer_t *, statistics_t *);
static void add_no_pair_entry(const buffer_t *, statistics_t *);
static void list_free(kst_list_entry_t **);
static void list_add(kst_list_entry_t *, kst_list_entry_t **);
static kst_list_entry_t *list_del(kst_list_entry_t **);

/* "-t" */
static void main_print_normal(print_data_t *p_data)
{
	int hash;
	int nr_syscalls = kstrax_get_nr_syscalls();
	buffer_t *buffer;
	kst_list_entry_t *copy, *del;
	kst_list_entry_t **flist_head, **hash_table;
	
	buffer = p_data->buf;
	flist_head = &p_data->free_list;
	hash_table = p_data->hash_table;

	if (buffer->sys_call_number > 0) { /* call entry */
		__IS_NO_PAIR {
			/* these have no return entry */
			if (kstrax_gflag == 0)
				print_call_entry(buffer, p_data->w_fp);
			return;
		}	
		copy = create_hash_entry(buffer, flist_head);
		hash = copy->buf.sys_call_number % nr_syscalls;
		list_add(copy, &hash_table[hash]);
	} else { /* return entry */
		kst_list_entry_t **now;

		hash = (-buffer->sys_call_number) % nr_syscalls;
		if (__print_sigreturn_normal(hash, buffer, p_data) == PRINT_END)
			return;

		now = &hash_table[hash];			
		while (*now) {
			if ((*now)->buf.pid == buffer->pid) {
				if (kstrax_gflag == 0)
					print_pair_entry(&(*now)->buf, buffer, 
							 p_data->w_fp);
				else
					__print_pair_entry_graph(&(*now)->buf,
								 buffer,
								 p_data->w_fp);
				del = list_del(now);
				list_add(del, flist_head);
				return;
			} else {
				now = &(*now)->next;
			}
		}
		if (*now == NULL) {
			if (buffer->sys_call_number == 0) {
				copy = create_hash_entry(buffer, flist_head);
				list_add(copy, &hash_table[hash]);
			} else {
				/* there is not call entry but only return entry */
				if (kstrax_gflag == 0)
					print_return_entry(buffer, p_data->w_fp);
			}
		}
	}
}

/* "-r" */
static void main_print_raw(print_data_t *p_data)
{
	int nr_syscalls = kstrax_get_nr_syscalls();
	buffer_t *buffer;
	
	buffer = p_data->buf;
	
	if(nr_syscalls < abs(buffer->sys_call_number)) {
		fprintf(p_data->w_fp, "Unknown syscall number : %d\n", 
			buffer->sys_call_number);
		return;
	}

	if (buffer->sys_call_number == 0) {
		if (buffer->arg_1 == MAGIC_SYSCALL_ZERO &&
		    buffer->arg_3 == MAGIC_SYSCALL_ZERO &&
		    buffer->arg_4 == MAGIC_SYSCALL_ZERO &&
		    buffer->arg_5 == MAGIC_SYSCALL_ZERO &&
		    buffer->arg_6 == MAGIC_SYSCALL_ZERO) {
			print_return_entry_raw(p_data->w_fp, buffer);
		} else {
			print_call_entry_raw(p_data->w_fp, buffer);
		}
	}
	else if (buffer->sys_call_number < 0) {
		print_return_entry_raw(p_data->w_fp, buffer);
	} else {
		print_call_entry_raw(p_data->w_fp, buffer);
	}
}

static void main_print_stats(print_data_t *p_data)
{
	int hash;
	int nr_syscalls = kstrax_get_nr_syscalls();
	buffer_t *buffer;
	kst_list_entry_t *copy, *del;
	kst_list_entry_t **flist_head, **hash_table;
	statistics_t *stats;

	buffer = p_data->buf;
	flist_head = &p_data->free_list;
	hash_table = p_data->hash_table;
	stats = p_data->stats_array;

	if (buffer->sys_call_number > 0) { /* call entry */
		__IS_NO_PAIR {
			/* these have no return entry */
			add_no_pair_entry(buffer, stats);
			return;
		}	
		copy = create_hash_entry(buffer, flist_head);
		hash = copy->buf.sys_call_number % nr_syscalls;
		list_add(copy, &hash_table[hash]);
	} else { /* return entry */
		kst_list_entry_t **now;

		hash = (-buffer->sys_call_number) % nr_syscalls;
		if (__print_sigreturn_stats(hash, buffer, p_data) == PRINT_END)
			return;

		now = &hash_table[hash];
		while (*now) {
			if ((*now)->buf.pid == buffer->pid) {
				add_pair_entry(&(*now)->buf, buffer, stats);
				del = list_del(now);
				list_add(del, flist_head);
				return;
			} else {
				now = &(*now)->next;
			}
		}
		if (*now == NULL) {
			if (buffer->sys_call_number == 0) {
				copy = create_hash_entry(buffer, flist_head);
				list_add(copy, &hash_table[hash]);
			} else {
				/* there is not call entry but only return entry */
				add_no_pair_entry(buffer, stats);
			}
		}
	}
}

static void print_call_entry(const buffer_t *buf, FILE *w_fp)
{
	int nr_syscalls = kstrax_get_nr_syscalls();
	long call_utime;
	struct tm call_time;
	
	if (nr_syscalls < abs(buf->sys_call_number)) {
		fprintf(w_fp, "Unknown syscall number : %d\n", 
			buf->sys_call_number);
		return;
	}

	call_time = __set_time(buf);
	call_utime = __SET_UTIME(buf);

	fprintf(w_fp, "%5hd  %02d:%02d:%02d.%06ld  --:--:--.------   ",
		buf->pid,
		call_time.tm_hour,
		call_time.tm_min,
		call_time.tm_sec,
		call_utime);
	__print_syscall_name(w_fp, buf);
	__print_syscall_arg(w_fp, buf);
	fprintf(w_fp, " = ??\n");
}

static void print_return_entry(const buffer_t *buf, FILE *w_fp)
{
	int nr_syscalls = kstrax_get_nr_syscalls();
	long ret_utime;
	struct tm ret_time;

	if (nr_syscalls < abs(buf->sys_call_number)) {
		fprintf(w_fp, "Unknown syscall number : %d\n", -buf->sys_call_number);
		return;
	}

	ret_time = __set_time(buf);
	ret_utime = __SET_UTIME(buf);

	fprintf(w_fp, "%5hd  --:--:--.------  %02d:%02d:%02d.%06ld   ",
	       buf->pid,
	       ret_time.tm_hour,
	       ret_time.tm_min,
	       ret_time.tm_sec,
	       ret_utime);
	__print_syscall_name(w_fp, buf);
	__print_syscall_arg(w_fp, buf);
	if (-ERR_LIMIT <= buf->return_value && buf->return_value < 0) {
		fprintf(w_fp, " = -1 (%s)\n", strerror(abs(buf->return_value)));
	} else {
		fprintf(w_fp, " = 0x%lx\n", buf->return_value);
	}
}

static void print_pair_entry(const buffer_t *call, const buffer_t *ret, FILE *w_fp)
{
	int nr_syscalls = kstrax_get_nr_syscalls();
	long call_utime, ret_utime;
	struct tm call_time, ret_time;

	if (nr_syscalls < abs(call->sys_call_number)) {
		fprintf(w_fp, "Unknown syscall number : %d\n", call->sys_call_number);
		return;
	}

	call_time = __set_time(call);
	ret_time  = __set_time(ret);
	call_utime = __SET_UTIME(call);
	ret_utime  = __SET_UTIME(ret);

	fprintf(w_fp, "%5hd  %02d:%02d:%02d.%06ld  %02d:%02d:%02d.%06ld   ",
	       call->pid,
	       call_time.tm_hour,
	       call_time.tm_min,
	       call_time.tm_sec,
	       call_utime,
	       ret_time.tm_hour,
	       ret_time.tm_min,
	       ret_time.tm_sec,
	       ret_utime);
	__print_syscall_name(w_fp, call);
	__print_syscall_arg(w_fp, call);
	if (-ERR_LIMIT <= ret->return_value && ret->return_value < 0) {
		fprintf(w_fp, " = -1 (%s)\n", strerror(abs(ret->return_value)));
	} else {
		fprintf(w_fp, " = 0x%lx\n", ret->return_value);
	}
}

static void print_call_entry_raw(FILE *fp, buffer_t *buf)
{
	long exe_utime;
	struct tm exe_time;

	exe_time = __set_time(buf);
	exe_utime = __SET_UTIME(buf);

	fprintf(fp, "RAW   %8ld %3ld  %5hd  "
		"%02d:%02d:%02d.%06ld  %4hd  ",
		buf->serial,
		buf->cpu,
		buf->pid,
		exe_time.tm_hour,
		exe_time.tm_min,
		exe_time.tm_sec,
		exe_utime,
		buf->sys_call_number);
	
	__print_syscall_name(fp, buf);
	
	__print_syscall_arg(fp, buf);
	fprintf(fp, " = ??\n");
}

static void print_return_entry_raw(FILE *fp, buffer_t *buf)
{
	long exe_utime;
	struct tm exe_time;

	exe_time = __set_time(buf);
	exe_utime = __SET_UTIME(buf);
	
	/* return entry */
	fprintf(fp, "RAW   %8ld %3ld  %5hd  "
		"%02d:%02d:%02d.%06ld  %4hd  <",
		buf->serial,
		buf->cpu,
		buf->pid,
		exe_time.tm_hour,
		exe_time.tm_min,
		exe_time.tm_sec,
		exe_utime,
		-buf->sys_call_number);
	
	__print_syscall_name(fp, buf);
	
	__print_syscall_arg(fp, buf);
	if (-ERR_LIMIT <= buf->return_value && buf->return_value < 0) {
		fprintf(fp, " = -1 (errno %d)>\n", 
			abs(buf->return_value));
	} else {
		fprintf(fp, " = 0x%lx>\n", 
			buf->return_value);
	}
}

/*---------------------------------------------------------------------------------
 *  post processing for each option
 */

static void flush_hash_table(print_data_t *);
static void stats_flush_hash_table(print_data_t *);
static void stats_print(print_data_t *);
static void stats_print_graph(print_data_t *);

/* "-t" */
static void post_print_normal(print_data_t *p_data)
{
	flush_hash_table(p_data);
	free(p_data->hash_table);
	list_free(&p_data->free_list);
}

/* "-r" */
static void post_print_raw(print_data_t *p_data)
{
	/* dummy function */
}

/* "-c" */
static void post_print_stats(print_data_t *p_data)
{
	stats_flush_hash_table(p_data);
	if (kstrax_gflag == 1) {
		stats_print_graph(p_data);
	} else {
		stats_print(p_data);
	}

	free(p_data->stats_array);
	list_free(&p_data->free_list);
	free(p_data->hash_table);
}

static void flush_hash_table(print_data_t *p_data)
{
	int i;
	int nr_syscalls = kstrax_get_nr_syscalls();

	while (1) {
		int count = 0;
		kst_list_entry_t **tmp, **min = NULL;

		for (i = 0; i < nr_syscalls; i++) {
			tmp = &p_data->hash_table[i];
			if (*tmp == NULL) {
				count++;
				continue;
			}

			while ((*tmp)->next != NULL) {
				tmp = &(*tmp)->next;
			}

			if (min == NULL || 
			    (*min)->buf.serial > (*tmp)->buf.serial) {
				min = tmp;
			} 
		}
		
		/* hash_table is empty */
		if (count == nr_syscalls) break;

		if (kstrax_gflag == 0)
			print_call_entry(&(*min)->buf, p_data->w_fp);
		list_add(*min, &p_data->free_list);
		*min = NULL;
	}
}

static void stats_flush_hash_table(print_data_t *p_data)
{
	int index = 0;
	int nr_syscalls = kstrax_get_nr_syscalls();
	kst_list_entry_t **tmp;
	kst_list_entry_t *del;

	while (index < nr_syscalls) {
		tmp = &p_data->hash_table[index];
		if (*tmp == NULL) {
			index++;
			continue;
		}
		add_no_pair_entry(&(*tmp)->buf, p_data->stats_array);
		del = list_del(tmp);		
		list_add(del, &p_data->free_list);
	}
}

static void stats_print(print_data_t *p_data)
{
	int i, nr_stats;
	statistics_t *stats = p_data->stats_array;
	unsigned long nr_pair;
	
	nr_stats = __get_nr_syscalls();

	for (i = 0; i < nr_stats; i++) {
		if (stats[i].count == 0)
			continue;

		__print_syscall_name_stats(p_data->w_fp, i);
		fprintf(p_data->w_fp, "%7lu  ", stats[i].count);

		nr_pair = stats[i].count - stats[i].nr_no_entry -
			stats[i].nr_no_return;

		if (nr_pair == 0) {
			int j;
			for (j = 0; j < 4; j++) 
				fprintf(p_data->w_fp, "------.------  ");
		} else {
			__print_proc_time(p_data->w_fp, 
					  (stats[i].total_time / nr_pair));
			__print_proc_time(p_data->w_fp, stats[i].max_time);
			__print_proc_time(p_data->w_fp, stats[i].min_time);
			__print_proc_time(p_data->w_fp, stats[i].total_time);
		}	
		fprintf(p_data->w_fp, "%6lu  %6lu  %6lu\n",
			stats[i].nr_error,
			stats[i].nr_no_entry,
			stats[i].nr_no_return);
	}
}

static void stats_print_graph(print_data_t *p_data)
{
	int i, nr_stats;
	statistics_t *stats = p_data->stats_array;
	unsigned long nr_pair;

	nr_stats = __get_nr_syscalls();
	
	for (i = 0; i < nr_stats; i++) {
		__print_syscall_name_stats_graph(p_data->w_fp, i);
		fprintf(p_data->w_fp, "%lu ", stats[i].count);

		nr_pair = stats[i].count - stats[i].nr_no_entry -
			stats[i].nr_no_return;
		
		if (nr_pair == 0) {
			int j;
			for (j = 0; j < 4; j++) 
				fprintf(p_data->w_fp, "0.000000 ");
		} else {
			__print_proc_time_graph(p_data->w_fp,
						(stats[i].total_time / nr_pair));
			__print_proc_time_graph(p_data->w_fp, stats[i].max_time);
			__print_proc_time_graph(p_data->w_fp, stats[i].min_time);
			__print_proc_time_graph(p_data->w_fp, stats[i].total_time);
		}	
		fprintf(p_data->w_fp, "%lu %lu %lu\n",
			stats[i].nr_error,
			stats[i].nr_no_entry,
			stats[i].nr_no_return);
	}
}

/*-------------------------------------------------------------------------------
 *
 */
static kst_list_entry_t *create_hash_entry(const buffer_t *buf, 
					   kst_list_entry_t **flist_head)
{
	kst_list_entry_t *ret;
	
	/* create cache */
	if (*flist_head == NULL) {
		int i;
		for (i = 0; i < 50; i++) { /* create 50 entry for one time */
			kst_list_entry_t *tmp;
			tmp = (kst_list_entry_t *)malloc(sizeof(kst_list_entry_t));
			if (tmp == NULL) {
				perror("create_hash_entry(malloc)");
				exit(1);
			}
			if (i == 0) {
				tmp->next = NULL;
			} else {
				tmp->next = *flist_head;
			}
			*flist_head = tmp;
		}
	}

	ret = *flist_head;
	*flist_head = ret->next;
	ret->buf = *buf;
	ret->next = NULL;
	return ret;
}

static void add_pair_entry(const buffer_t *call, const buffer_t *ret, 
			   statistics_t *stats)
{
	int number;
	proc_time_t proc_time;

	number = __get_stats_array_index(call);
	if (number < 0)
		return;

	proc_time = __calc_processing_time(call, ret);

	stats[number].count++;
	stats[number].total_time += proc_time;
	if (stats[number].max_time < proc_time) 
		stats[number].max_time = proc_time;
	if (stats[number].min_time > proc_time)
		stats[number].min_time = proc_time;
	if (-ERR_LIMIT <= ret->return_value && ret->return_value < 0)
		stats[number].nr_error++;	
}

static void add_no_pair_entry(const buffer_t *buf, statistics_t *stats)
{
	int number;
	number = __get_stats_array_index(buf);

	if (number > 0) {
		stats[number].nr_no_return++;
	} else {
		stats[-number].nr_no_entry++;
	}
	stats[abs(number)].count++;
}

static void list_free(kst_list_entry_t **list)
{
	while (*list != NULL) {
		kst_list_entry_t *del;
		del = list_del(list);
		free(del);
	}
}

static void list_add(kst_list_entry_t *new, kst_list_entry_t **head)
{
	new->next = *head;
	*head = new;
}

static kst_list_entry_t *list_del(kst_list_entry_t **now)
{
	kst_list_entry_t *ret;
	ret = *now;
	*now = (*now)->next;
	return ret;
}


/*----------------------------------------------------------------------------------
 *  lost syscall processing
 */
static void lost_syscall_normal(print_data_t *p_data, unsigned long nr_lost)
{
	flush_hash_table(p_data);
	fprintf(p_data->w_fp, 
		"\n------- lost %lu syscalls ------\n\n", (nr_lost+1) / 2);
}

static void lost_syscall_raw(print_data_t *p_data, unsigned long nr_lost)
{
	fprintf(p_data->w_fp, 
		"\n------- lost %lu entries ------\n\n", nr_lost);
}

static void lost_syscall_stats(print_data_t *p_data, unsigned long nr_lost)
{
	stats_flush_hash_table(p_data);
}








