/*
 * This program is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any later
 * version.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program. If not, see <http://www.gnu.org/licenses/>.
 */
package handlers.actionhandlers;

import com.l2jserver.Config;
import com.l2jserver.gameserver.datatables.ItemTable;
import com.l2jserver.gameserver.handler.IActionHandler;
import com.l2jserver.gameserver.model.L2DropCategory;
import com.l2jserver.gameserver.model.L2DropData;
import com.l2jserver.gameserver.model.L2Object;
import com.l2jserver.gameserver.model.L2Object.InstanceType;
import com.l2jserver.gameserver.model.actor.L2Npc;
import com.l2jserver.gameserver.model.actor.instance.L2MerchantInstance;
import com.l2jserver.gameserver.model.actor.instance.L2PcInstance;
import com.l2jserver.gameserver.network.serverpackets.MyTargetSelected;
import com.l2jserver.gameserver.network.serverpackets.NpcHtmlMessage;
import com.l2jserver.gameserver.network.serverpackets.StatusUpdate;
import com.l2jserver.gameserver.skills.Stats;
import com.l2jserver.gameserver.templates.item.L2Item;
import com.l2jserver.util.StringUtil;

public class L2NpcActionShift implements IActionHandler
{
	/**
	 * Manage and Display the GM console to modify the L2NpcInstance (GM only).<BR><BR>
	 * 
	 * <B><U> Actions (If the L2PcInstance is a GM only)</U> :</B><BR><BR>
	 * <li>Set the L2NpcInstance as target of the L2PcInstance player (if necessary)</li>
	 * <li>Send a Server->Client packet MyTargetSelected to the L2PcInstance player (display the select window)</li>
	 * <li>If L2NpcInstance is autoAttackable, send a Server->Client packet StatusUpdate to the L2PcInstance in order to update L2NpcInstance HP bar </li>
	 * <li>Send a Server->Client NpcHtmlMessage() containing the GM console about this L2NpcInstance </li><BR><BR>
	 * 
	 * <FONT COLOR=#FF0000><B> <U>Caution</U> : Each group of Server->Client packet must be terminated by a ActionFailed packet in order to avoid
	 * that client wait an other packet</B></FONT><BR><BR>
	 * 
	 * <B><U> Example of use </U> :</B><BR><BR>
	 * <li> Client packet : Action</li><BR><BR>
	 */
	public boolean action(L2PcInstance activeChar, L2Object target_, boolean interact)
	{
		L2Npc target = (L2Npc)target_;

		// Check if the L2PcInstance is a GM
		if (activeChar.getAccessLevel().isGm())
		{
			// Set the target of the L2PcInstance activeChar
			activeChar.setTarget(target);

			// Send a Server->Client packet MyTargetSelected to the L2PcInstance activeChar
			// The activeChar.getLevel() - getLevel() permit to display the correct color in the select window
			MyTargetSelected my = new MyTargetSelected(target.getObjectId(), activeChar.getLevel() - target.getLevel());
			activeChar.sendPacket(my);

			// Check if the activeChar is attackable (without a forced attack)
			if (target.isAutoAttackable(activeChar))
			{
				// Send a Server->Client packet StatusUpdate of the L2NpcInstance to the L2PcInstance to update its HP bar
				StatusUpdate su = new StatusUpdate(target);
				su.addAttribute(StatusUpdate.CUR_HP, (int)target.getCurrentHp());
				su.addAttribute(StatusUpdate.MAX_HP, target.getMaxHp());
				activeChar.sendPacket(su);
			}

			NpcHtmlMessage html = new NpcHtmlMessage(0);
			html.setFile(activeChar.getHtmlPrefix(), "data/html/admin/npcinfo.htm");
			
			html.replace("%objid%", target.getObjectId());
			html.replace("%class%", target.getClass().getSimpleName());
			html.replace("%id%",    target.getTemplate().npcId);
			html.replace("%lvl%",   target.getTemplate().level);
			html.replace("%name%",  target.getTemplate().name);
			html.replace("%tmplid%",target.getTemplate().npcId);
			html.replace("%aggro%", target.getAggroRange());
			html.replace("%hp%",    (int)target.getCurrentHp());
			html.replace("%hpmax%", target.getMaxHp());
			html.replace("%mp%",    (int)target.getCurrentMp());
			html.replace("%mpmax%", target.getMaxMp());
			
			html.replace("%patk%", target.getPAtk(null));
			html.replace("%matk%", target.getMAtk(null, null));
			html.replace("%pdef%", target.getPDef(null));
			html.replace("%mdef%", target.getMDef(null, null));
			html.replace("%accu%", target.getAccuracy());
			html.replace("%evas%", target.getEvasionRate(null));
			html.replace("%crit%", target.getCriticalHit(null, null));
			html.replace("%rspd%", target.getRunSpeed());
			html.replace("%aspd%", target.getPAtkSpd());
			html.replace("%cspd%", target.getMAtkSpd());
			html.replace("%str%",  target.getSTR());
			html.replace("%dex%",  target.getDEX());
			html.replace("%con%",  target.getCON());
			html.replace("%int%",  target.getINT());
			html.replace("%wit%",  target.getWIT());
			html.replace("%men%",  target.getMEN());
			html.replace("%loc%",  target.getX()+" "+target.getY()+" "+target.getZ());
			
			if (target.getSpawn() != null)
			{
				html.replace("%spawn%", target.getSpawn().getLocx()+" "+target.getSpawn().getLocy()+" "+target.getSpawn().getLocz());
				html.replace("%loc2d%", (int)Math.sqrt(target.getPlanDistanceSq(target.getSpawn().getLocx(), target.getSpawn().getLocy())));
				html.replace("%loc3d%", (int)Math.sqrt(target.getDistanceSq(target.getSpawn().getLocx(), target.getSpawn().getLocy(), target.getSpawn().getLocz())));
				html.replace("%resp%",  target.getSpawn().getRespawnDelay() / 1000);
			}
			else
			{
				html.replace("%spawn%", "<font color=FF0000>null</font>");
				html.replace("%loc2d%", "<font color=FF0000>--</font>");
				html.replace("%loc3d%", "<font color=FF0000>--</font>");
				html.replace("%resp%",  "<font color=FF0000>--</font>");
			}
			
			if (target.hasAI())
			{
				html.replace("%ai%",   "<tr><td><table width=270 border=0><tr><td width=100><font color=FFAA00>AI</font></td><td align=right width=170>"+target.getAI().getClass().getSimpleName()+"</td></tr></table></td></tr>");
				html.replace("%aint%", "<tr><td><table width=270 border=0 bgcolor=131210><tr><td width=100><font color=FFAA00>Intention:</font></td><td align=right width=170>"+target.getAI().getIntention().name()+"</td></tr></table></td></tr>");
			}
			else
			{
				html.replace("%ai%",   "");
				html.replace("%aint%", "");
			}
			
			if (target instanceof L2MerchantInstance)
			{
				html.replace("%butt%","<button value=\"Shop\" action=\"bypass -h admin_showShop "+target.getTemplate().npcId+"\" width=60 height=21 back=\"L2UI_ct1.button_df\" fore=\"L2UI_ct1.button_df\">");
			}
			else
			{
				html.replace("%butt%","");
			}
			
			activeChar.sendPacket(html);			
		}
		else if (Config.ALT_GAME_VIEWNPC)
		{
			// Set the target of the L2PcInstance activeChar
			activeChar.setTarget(target);

			// Send a Server->Client packet MyTargetSelected to the L2PcInstance activeChar
			// The activeChar.getLevel() - getLevel() permit to display the correct color in the select window
			MyTargetSelected my = new MyTargetSelected(target.getObjectId(), activeChar.getLevel() - target.getLevel());
			activeChar.sendPacket(my);

			// Check if the activeChar is attackable (without a forced attack)
			if (target.isAutoAttackable(activeChar))
			{
				// Send a Server->Client packet StatusUpdate of the L2NpcInstance to the L2PcInstance to update its HP bar
				StatusUpdate su = new StatusUpdate(target);
				su.addAttribute(StatusUpdate.CUR_HP, (int) target.getCurrentHp());
				su.addAttribute(StatusUpdate.MAX_HP, target.getMaxHp());
				activeChar.sendPacket(su);
			}

			NpcHtmlMessage html = new NpcHtmlMessage(0);
			final StringBuilder html1 = StringUtil.startAppend(
					1000,
					"<html><body>" +
					"<br><center><font color=\"LEVEL\">[Combat Stats]</font></center>" +
					"<table border=0 width=\"100%\">" +
					"<tr><td>Max.HP</td><td>",
					String.valueOf((int) (target.getMaxHp() / target.getStat().calcStat(Stats.MAX_HP, 1, target, null))),
					"*",
					String.valueOf((int) target.getStat().calcStat(Stats.MAX_HP, 1, target, null)),
					"</td><td>Max.MP</td><td>",
					String.valueOf(target.getMaxMp()),
					"</td></tr>" +
					"<tr><td>P.Atk.</td><td>",
					String.valueOf(target.getPAtk(null)),
					"</td><td>M.Atk.</td><td>",
					String.valueOf(target.getMAtk(null, null)),
					"</td></tr>" +
					"<tr><td>P.Def.</td><td>",
					String.valueOf(target.getPDef(null)),
					"</td><td>M.Def.</td><td>",
					String.valueOf(target.getMDef(null, null)),
					"</td></tr>" +
					"<tr><td>Accuracy</td><td>",
					String.valueOf(target.getAccuracy()),
					"</td><td>Evasion</td><td>",
					String.valueOf(target.getEvasionRate(null)),
					"</td></tr>" +
					"<tr><td>Critical</td><td>",
					String.valueOf(target.getCriticalHit(null, null)),
					"</td><td>Speed</td><td>",
					String.valueOf(target.getRunSpeed()),
					"</td></tr>" +
					"<tr><td>Atk.Speed</td><td>",
					String.valueOf(target.getPAtkSpd()),
					"</td><td>Cast.Speed</td><td>",
					String.valueOf(target.getMAtkSpd()),
					"</td></tr>" +
					"<tr><td>Race</td><td>",
					target.getTemplate().getRace().toString(),
					"</td><td></td><td></td></tr>" +
					"</table>" +
					"<br><center><font color=\"LEVEL\">[Basic Stats]</font></center>" +
					"<table border=0 width=\"100%\">" +
					"<tr><td>STR</td><td>",
					String.valueOf(target.getSTR()),
					"</td><td>DEX</td><td>",
					String.valueOf(target.getDEX()),
					"</td><td>CON</td><td>",
					String.valueOf(target.getCON()),
					"</td></tr>" +
					"<tr><td>INT</td><td>",
					String.valueOf(target.getINT()),
					"</td><td>WIT</td><td>",
					String.valueOf(target.getWIT()),
					"</td><td>MEN</td><td>",
					String.valueOf(target.getMEN()),
					"</td></tr>" +
					"</table>"
					);

			if (target.getTemplate().getDropData() != null)
			{
				StringUtil.append(html1,
						"<br><center><font color=\"LEVEL\">[Drop Info]</font></center>" +
						"<br>Rates legend: <font color=\"ff0000\">50%+</font> <font color=\"00ff00\">30%+</font> <font color=\"0000ff\">less than 30%</font>" +
						"<table border=0 width=\"100%\">"
						);
				for (L2DropCategory cat : target.getTemplate().getDropData())
				{
					for (L2DropData drop : cat.getAllDrops())
					{
						final L2Item item = ItemTable.getInstance().getTemplate(drop.getItemId());
						if (item == null)
							continue;

						final String color;

						if (drop.getChance() >= 500000)
							color = "ff0000";
						else if (drop.getChance() >= 300000)
							color = "00ff00";
						else
							color = "0000ff";

						StringUtil.append(html1,
								"<tr><td><font color=\"",
								color,
								"\">",
								item.getName(),
								"</font></td><td>",
								(drop.isQuestDrop() ? "Quest" : (cat.isSweep() ? "Sweep" : "Drop")),
								"</td></tr>"
								);
					}
				}
				html1.append("</table>");
			}
			html1.append("</body></html>");

			html.setHtml(html1.toString());
			activeChar.sendPacket(html);
		}
		return true;
	}

	public InstanceType getInstanceType()
	{
		return InstanceType.L2Npc;
	}
}
