// This file is part of Notepad++ project
// Copyright (C)2003 Don HO <don.h@free.fr>
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// Note that the GPL places important restrictions on "derived works", yet
// it does not provide a detailed definition of that term.  To avoid      
// misunderstandings, we consider an application to constitute a          
// "derivative work" for the purpose of this license if it does any of the
// following:                                                             
// 1. Integrates source code from Notepad++.
// 2. Integrates/includes/aggregates Notepad++ into a proprietary executable
//    installer, such as those produced by InstallShield.
// 3. Links to a library or executes a program that does any of the above.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.


#include "ansiCharPanel.h"
#include "ScintillaEditView.h"

#ifdef MOD_CHAR_PANEL_ENCODING

#define IDM_CHARPANEL_ENCODE WM_APP
#define IDM_CHARPANEL_ENCODE_LIMIT IDM_CHARPANEL_ENCODE + _countof(ENCODINGS) - 1

#endif

#ifdef MOD_CHAR_PANEL_ENCODING
#else
void AnsiCharPanel::switchEncoding()
{
	int codepage = (*_ppEditView)->getCurrentBuffer()->getEncoding();
	_listView.resetValues(codepage);
}
#endif

#ifdef MOD_CHAR_PANEL_ENCODING
void AnsiCharPanel::trackPopupMenu()
{
	if (_hPopupMenu == nullptr) {
		std::wstring ini(NppParameters::getInstance()->getNppPath());
		PathAppend(ini, L"customCharPanelMenu");
		FILE *fp = _wfopen(ini.c_str(), L"rt");
		if (fp != NULL) {
			_hPopupMenu = CreatePopupMenu();
			nEncoding = 0;
			bool checked = false;
			const auto acp = _listView.getCodepage();	// == ::GetACP()
			WcharMbcsConvertor *wmc = WcharMbcsConvertor::getInstance();
			char buf[80];
			while (fgets(buf, sizeof(buf), fp) != NULL) {
				if (buf[0] == '\xEF' && buf[1] == '\xBB' && buf[2] == '\xBF') strcpy(buf, buf + 3);	// BOM
				char *n;
				if ((n = strchr(buf, L'\r')) != nullptr) *n = '\0';
				if ((n = strchr(buf, L'\n')) != nullptr) *n = '\0';

				char *m;
				if ('0' <= buf[0] && buf[0] <= '9' && (m = strchr(buf, L'\t')) != nullptr) {
					int encoding = atoi(buf); if (encoding == 0) continue;
					char *menu = m + 1;
					if (*menu  == '\0') continue;

					ENCODINGS[nEncoding] = encoding;
					::InsertMenu(_hPopupMenu, nEncoding, MF_BYPOSITION, IDM_CHARPANEL_ENCODE + nEncoding, wmc->char2wchar(menu, 65001));
					if (!checked && encoding == acp) {
						::CheckMenuItem(_hPopupMenu, nEncoding, MF_BYPOSITION | MF_CHECKED);
						checked = true;
					}
					++nEncoding;
					if (nEncoding >= _countof(ENCODINGS) - 1) break;
				}
			}
			fclose(fp);
			if (!checked) {
				ENCODINGS[nEncoding] = acp;
				::InsertMenu(_hPopupMenu, nEncoding, MF_BYPOSITION | MF_CHECKED, IDM_CHARPANEL_ENCODE + nEncoding, L"Default");
				++nEncoding;
			}
		}
	}
	if (_hPopupMenu != nullptr) {
		POINT p;
		::GetCursorPos(&p);
		::TrackPopupMenu(_hPopupMenu, 0, p.x, p.y, 0, _hSelf, NULL);
	}
}
#endif

INT_PTR CALLBACK AnsiCharPanel::run_dlgProc(UINT message, WPARAM wParam, LPARAM lParam)
{
    switch (message)
    {
        case WM_INITDIALOG :
        {
			_listView.init(_hInst, _hSelf);
#ifdef MOD_CHAR_PANEL_ENCODING
			_listView.setValues(::GetACP());
#else
			int codepage = (*_ppEditView)->getCurrentBuffer()->getEncoding();
			_listView.setValues(codepage==-1?0:codepage);
#endif
			_listView.display();

            return TRUE;
        }
#ifdef MOD_CHAR_PANEL_ENCODING
		case WM_DESTROY:
			if (_hPopupMenu) DestroyMenu(_hPopupMenu);
			break;
#endif

#ifdef MOD_CHAR_PANEL_ENCODING
		case WM_COMMAND:
		{
			int i = LOWORD(wParam) - IDM_CHARPANEL_ENCODE;
			if (0 <= i && i < nEncoding) {
				int codepage = ENCODINGS[i];
				if (codepage != _listView.getCodepage()) {
					_listView.changeCodepage(codepage);
					for (int x = 0; x < nEncoding; ++x)
						::CheckMenuItem(_hPopupMenu, x, x == i ? MF_BYPOSITION | MF_CHECKED : MF_BYPOSITION | MF_UNCHECKED);
				}
				return TRUE;
			}
			break;
		}
#endif
		case WM_NOTIFY:
		{
			switch (((LPNMHDR)lParam)->code)
			{
#ifdef MOD_CHAR_PANEL_ENCODING
				case NM_RCLICK:
					trackPopupMenu();
					return TRUE;
#endif
				case NM_DBLCLK:
				{
					LPNMITEMACTIVATE lpnmitem = (LPNMITEMACTIVATE) lParam;
					int i = lpnmitem->iItem;

					if (i == -1)
						return TRUE;

					insertChar((unsigned char)i);
					return TRUE;
				}

				case LVN_KEYDOWN:
				{
					switch (((LPNMLVKEYDOWN)lParam)->wVKey)
					{
#ifdef MOD_CHAR_PANEL_ENCODING
						case VK_APPS:
							trackPopupMenu();
							return TRUE;
#endif
						case VK_RETURN:
						{
							int i = ListView_GetSelectionMark(_listView.getHSelf());

							if (i == -1)
								return TRUE;

							insertChar((unsigned char)i);
							return TRUE;
						}
						default:
							break;
					}
				}
				break;

				default:
					break;
			}
		}
		return TRUE;

        case WM_SIZE:
        {
            int width = LOWORD(lParam);
            int height = HIWORD(lParam);
			::MoveWindow(_listView.getHSelf(), 0, 0, width, height, TRUE);
            break;
        }

        default :
            return DockingDlgInterface::run_dlgProc(message, wParam, lParam);
    }
	return DockingDlgInterface::run_dlgProc(message, wParam, lParam);
}

void AnsiCharPanel::insertChar(unsigned char char2insert) const
{
    const char charStr[] = { (char)char2insert, '\0' };
    WCHAR wCharStr[10];
    char multiByteStr[10];
#ifdef FIX_CHAR_PANEL_CODEPAGE
	if (MultiByteToWideChar(_listView.getCodepage(), MB_ERR_INVALID_CHARS, charStr, -1, wCharStr, _countof(wCharStr)) > 0)
		WideCharToMultiByte(65001/*CP_UTF8*/, 0, wCharStr, -1, multiByteStr, _countof(multiByteStr), NULL, NULL);
	else
		{ multiByteStr[0] = char2insert; multiByteStr[1] = '\0'; }
#else
	int codepage = (*_ppEditView)->getCurrentBuffer()->getEncoding();
	if (codepage == -1)
	{
		bool isUnicode = ((*_ppEditView)->execute(SCI_GETCODEPAGE) == SC_CP_UTF8);
		if (isUnicode)
		{
			MultiByteToWideChar(0, 0, charStr, -1, wCharStr, sizeof(wCharStr));
			WideCharToMultiByte(CP_UTF8, 0, wCharStr, -1, multiByteStr, sizeof(multiByteStr), NULL, NULL);
		}
		else // ANSI
		{
			multiByteStr[0] = charStr[0];
			multiByteStr[1] = charStr[1];
		}
	}
	else
	{
		MultiByteToWideChar(codepage, 0, charStr, -1, wCharStr, sizeof(wCharStr));
		WideCharToMultiByte(CP_UTF8, 0, wCharStr, -1, multiByteStr, sizeof(multiByteStr), NULL, NULL);
	}
#endif
	(*_ppEditView)->execute(SCI_REPLACESEL, 0, (LPARAM)"");
	int len = (char2insert < 128)?1:strlen(multiByteStr);
    (*_ppEditView)->execute(SCI_ADDTEXT, len, (LPARAM)multiByteStr);
	(*_ppEditView)->getFocus();
}