﻿/*
This file is:
The zlib/libpng License 
Copyright (c) 2012 freesftys<freesftys@gmail.com>

This software is provided 'as-is', without any express or implied warranty. 
In no event will the authors be held liable for any damages arising from 
the use of this software.

Permission is granted to anyone to use this software for any purpose, including 
commercial applications, and to alter it and redistribute it freely, subject to 
the following restrictions:

    1. The origin of this software must not be misrepresented; you must not 
	   claim that you wrote the original software. If you use this software in 
	   a product, an acknowledgment in the product documentation would be 
	   appreciated but is not required.
	
    2. Altered source versions must be plainly marked as such, and must not be 
       misrepresented as being the original software.
	
    3. This notice may not be removed or altered from any source distribution.

原文
http://opensource.org/licenses/Zlib

日本語訳
http://sourceforge.jp/projects/opensource/wiki/licenses%2Fzlib_libpng_license
*/

#include <windows.h>
#include <windowsx.h>	// +[JOJO]
#include <commctrl.h>
#include <stdlib.h>

#include "NppHorizontalRuler.h"
#include "HorizontalRuler.h"
#include "menuCmdID.h"

HorizontalRuler::HorizontalRuler()
{
}

HorizontalRuler::~HorizontalRuler()
{
}

void HorizontalRuler::Init(HWND npp, HWND scintilla, HWND tab)
{
	this->nppHwnd = npp;
	this->scintillaHwnd = scintilla;
	this->tabHwnd = tab;
	this->_pScintillaFunc = (SCINTILLA_FUNC)::SendMessage(this->scintillaHwnd, SCI_GETDIRECTFUNCTION, 0, 0);
	this->_pScintillaPtr = (SCINTILLA_PTR)::SendMessage(this->scintillaHwnd, SCI_GETDIRECTPOINTER, 0, 0);
}

bool HorizontalRuler::IsInit()
{
	return this->tabHwnd != NULL && Ruler::visible;
}

void HorizontalRuler::GetInitPos()
{
	if (!Ruler::visible) return;

	const int cxedge = GetSystemMetrics(SM_CXEDGE);
	const int cyedge = GetSystemMetrics(SM_CYEDGE);

	RECT rc; GetClientRect(this->tabHwnd, &rc);
	TabCtrl_AdjustRect(this->tabHwnd, FALSE, &rc);
	if (GetWindowLongPtr(this->tabHwnd, GWL_STYLE) & TCS_VERTICAL) {
		rc.top  -= cyedge;
		rc.left -= cxedge;
	} else {
		RECT scClientRc;
		GetClientRect(this->scintillaHwnd, &scClientRc);
	 //	rc.top  ... nothing todo
		rc.left = scClientRc.left;
	}
	rc.right += cxedge;
	rc.bottom = rc.top + Ruler::nDrawHeight;

	this->nDrawStartX = rc.left;
	this->nDrawEndX   = rc.right;
		SCROLLBARINFO scrollInfo = { sizeof(scrollInfo), };
		GetScrollBarInfo(this->scintillaHwnd, OBJID_VSCROLL, &scrollInfo);
		if (scrollInfo.rgstate[0] != STATE_SYSTEM_INVISIBLE)
			this->nDrawEndX -= scrollInfo.rcScrollBar.right - scrollInfo.rcScrollBar.left;
	if (GetWindowLongPtr(this->scintillaHwnd, GWL_EXSTYLE) & WS_EX_CLIENTEDGE) {	// NPPM_SETEDITORBORDEREDGE
		this->nDrawStartX += cxedge;
		this->nDrawEndX   -= cxedge;
	}

	this->rulerRect = rc;
}

bool HorizontalRuler::isCursorMoved()
{
	if (this->IsInit())
		return this->invalid
		    || this->xOffset != execute(SCI_GETXOFFSET)
		    || this->caretX != execute(SCI_MAINCARETX)
		    || this->spaceWidth != execute(SCI_GETSPACEWIDTH)
		    || this->textStart != execute(SCI_GETTEXTSTARTX);
	else
		return false;
}

void HorizontalRuler::GetRuleArea()
{
	this->xOffset = (int) execute(SCI_GETXOFFSET);
	this->caretX = (int) execute(SCI_MAINCARETX);
	this->spaceWidth = (int) execute(SCI_GETSPACEWIDTH);
	this->textStart = (int) execute(SCI_GETTEXTSTARTX);
 //	this->nRightMargin = (int) execute(SCI_GETMARGINRIGHT);
}

void HorizontalRuler::PaintRuler()
{
	if (!Ruler::visible) return;

	// 描画範囲
	const int nRulerStartX = this->nDrawStartX + this->textStart;
	const int nRulerEndX   = this->nDrawEndX   - this->nRightMargin;

	// 横スクロールの計算
	const int nStartCol  = this->xOffset / this->spaceWidth;
	const int nScrollMod = this->xOffset % this->spaceWidth;
	const int nMinCol = nScrollMod > 0 ? 1 : 0;
	const int nMaxCol = 1 + (nRulerEndX - nRulerStartX) / this->spaceWidth;

	// キャレットの桁位置
	const int nCaret = this->caretX / this->spaceWidth;

	// 描画スタート
	const HDC hDC = GetWindowDC(this->tabHwnd);	
	const HFONT oldFont = (HFONT)SelectObject(hDC, Ruler::hFont);

	// 背景塗りつぶし
	RECT rc = this->rulerRect;
	HBRUSH hBrush = Ruler::BGColor ? CreateSolidBrush(Ruler::BGColor) : CreateSolidBrush(GetSysColor(COLOR_BTNFACE));
	FillRect(hDC, &rc, hBrush);
	DeleteObject((HGDIOBJ)hBrush); hBrush = nullptr;

	// 目盛り・文字の描画
	rc = this->rulerRect;
	rc.right = nRulerEndX;
	TEXTMETRIC tm;
	GetTextMetrics(hDC, &tm);
	const int textTop = rc.top + 1 - tm.tmInternalLeading;
	for (int i = nMinCol; i < nMaxCol; ++i) {
		const int colum = nStartCol + i;
		const int x = nRulerStartX - nScrollMod + (i * this->spaceWidth);
		int h;
		switch (colum % 10) {
		case 0:
			{
				WCHAR sColumNumber[10];
				int nLength = swprintf_s(sColumNumber, _countof(sColumNumber), L"%d", colum);
				SetBkMode(hDC, TRANSPARENT);
				ExtTextOut(hDC, x + 1, textTop, ETO_CLIPPED, &rc, sColumNumber, nLength, NULL);
			}
			h = Ruler::nDrawHeight;
			break;
		case 1:
		case 2:
			h = 1;
			break;
		case 5:
			h = Ruler::nDrawHeight * 2 / 4;
			break;
		default:
			h = Ruler::nDrawHeight * 1 / 4;
			break;
		}
		if (colum == nCaret) {
			PatBlt(hDC, x, rc.top, this->spaceWidth + 1, 1 + Ruler::nDrawHeight, DSTINVERT);
		}
		else {
			MoveToEx(hDC, x, rc.bottom - h, NULL);
			LineTo(hDC, x, 1 + rc.bottom);
		}
	}

	SelectObject(hDC, oldFont);
	ReleaseDC(this->tabHwnd, hDC);
}

bool HorizontalRuler::HitDrawArea(int /*x*/, int y)
{
	this->rulerDesctopRect = this->rulerRect;
	MapWindowPoints(this->tabHwnd, HWND_DESKTOP, (LPPOINT)&this->rulerDesctopRect, 2);
	return y >= this->rulerDesctopRect.top && y <  this->rulerDesctopRect.bottom;
}

void HorizontalRuler::EdgeLine(int x, int /*y*/)
{
	const int safetyWidth = this->spaceWidth;

	if (x < (this->rulerDesctopRect.left + this->textStart) + safetyWidth) {
		// Left Margin Area
	}
	else if (x < (this->rulerDesctopRect.right - this->nRightMargin) - safetyWidth) {
		// Ruler Area
		const int px = x - (this->rulerDesctopRect.left + this->textStart);
		const int nSetEdgeLine = (this->xOffset + px + this->spaceWidth / 2) / this->spaceWidth;

		const int em           = (int) execute(SCI_GETEDGEMODE);
		const int nEdgeMode = em & ~EDGE_WRAP, nEdgeWrap = em & EDGE_WRAP;
		const int nNowEdgeLine = (int) execute(SCI_GETEDGECOLUMN);

		if (nEdgeMode == EDGE_NONE) {	// show
			execute(SCI_SETEDGECOLUMN, nSetEdgeLine);
			execute(SCI_SETEDGEMODE, nEdgeWrap | EDGE_LINE);
		}
		else if (nSetEdgeLine == nNowEdgeLine) {	// hide
			execute(SCI_SETEDGEMODE, nEdgeWrap | EDGE_NONE);
		}
		else {	// move
			execute(SCI_SETEDGECOLUMN, nSetEdgeLine);
		}
	}
	else
	{
		// Right Margin Area
	}
}

void HorizontalRuler::Invalidate() { this->invalid = true; }

void HorizontalRuler::Redraw()
{
	if (this->invalid) { this->invalid = false; this->GetInitPos(); }
	this->GetRuleArea();
	this->PaintRuler();
}

void HorizontalRuler::RedrawIfCursorMoved()
{
	if (this->isCursorMoved())
		this->Redraw();
}

LRESULT HorizontalRuler::TabWndProc(WNDPROC lpPrevWndFunc, HWND hwnd, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
	switch (uMsg)
	{
	case WM_PAINT:
		if (this->isCursorMoved())
			this->Redraw();
		else
			this->PaintRuler();
		break;
	case WM_NCLBUTTONDBLCLK:
		if (this->IsInit())
			this->EdgeLine(GET_X_LPARAM(lParam), GET_Y_LPARAM(lParam));
		break;
	case WM_NCHITTEST:
		if (this->IsInit()) {
			if (this->HitDrawArea(GET_X_LPARAM(lParam), GET_Y_LPARAM(lParam)))
				return HTBORDER;
		}
		break;
	case WM_ERASEBKGND:	/* 初回、フォントを変更したとき、とか */
		this->Invalidate();
		break;
	//ドッキングボックスの枠の移動時の再描画で有効なメッセージがなくしょうがないので
	//下記の GETCURSEL GETIMAGELIST GETITEMWの内一つを利用するとりあえず GETIMAGELIST
	case TCM_GETIMAGELIST:	/* 左右2画面モード、とか */
		this->RedrawIfCursorMoved();
		break;
	}
	return CallWindowProc(lpPrevWndFunc, hwnd, uMsg, wParam, lParam);
}
