// This file is part of Notepad++ project
// Copyright (C)2003 Don HO <don.h@free.fr>
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// Note that the GPL places important restrictions on "derived works", yet
// it does not provide a detailed definition of that term.  To avoid      
// misunderstandings, we consider an application to constitute a          
// "derivative work" for the purpose of this license if it does any of the
// following:                                                             
// 1. Integrates source code from Notepad++.
// 2. Integrates/includes/aggregates Notepad++ into a proprietary executable
//    installer, such as those produced by InstallShield.
// 3. Links to a library or executes a program that does any of the above.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.


#include "ansiCharPanel.h"
#include "ScintillaEditView.h"
#include "localization.h"

#ifdef MOD_CHAR_PANEL_ENCODING

#define IDM_CHARPANEL_ENCODE WM_APP
#define IDM_CHARPANEL_ENCODE_LIMIT IDM_CHARPANEL_ENCODE + _countof(ENCODINGS) - 1

#endif

#ifdef MOD_CHAR_PANEL_ENCODING
#else
void AnsiCharPanel::switchEncoding()
{
	int codepage = (*_ppEditView)->getCurrentBuffer()->getEncoding();
	_listView.resetValues(codepage);
}
#endif

#ifdef MOD_CHAR_PANEL_ENCODING
void AnsiCharPanel::trackPopupMenu()
{
	if (_hPopupMenu == nullptr) {
		std::wstring ini(NppParameters::getInstance()->getNppPath());
		PathAppend(ini, L"customCharPanelMenu");
		FILE *fp = _wfopen(ini.c_str(), L"r,ccs=UTF-8");
		if (fp != NULL) {
			_hPopupMenu = CreatePopupMenu();
			nEncoding = 0;
			bool checked = false;
			const auto acp = _listView.getCodepage();	// == ::GetACP()
			WCHAR *buf, _buf[80];
			while ((buf = fgetws(_buf, _countof(_buf), fp)) != NULL) {
				if (buf[0] == 0xFEFF) ++buf;	// BOM
				WCHAR *n;
				if ((n = wcschr(buf, L'\r')) != nullptr) *n = L'\0';
				if ((n = wcschr(buf, L'\n')) != nullptr) *n = L'\0';

				WCHAR *m;
				if (iswdigit(buf[0]) && (m = wcschr(buf, L'\t')) != nullptr) {
					int encoding = _wtoi(buf); if (encoding == 0) continue;
					WCHAR *menu = m + 1;
					if (*menu  == L'\0') continue;

					ENCODINGS[nEncoding] = encoding;
					::InsertMenu(_hPopupMenu, nEncoding, MF_BYPOSITION, IDM_CHARPANEL_ENCODE + nEncoding, menu);
					if (!checked && encoding == acp) {
						::CheckMenuItem(_hPopupMenu, nEncoding, MF_BYPOSITION | MF_CHECKED);
						checked = true;
					}
					++nEncoding;
					if (nEncoding >= _countof(ENCODINGS) - 1) break;
				}
			}
			fclose(fp);
			if (!checked) {
				ENCODINGS[nEncoding] = acp;
				::InsertMenu(_hPopupMenu, nEncoding, MF_BYPOSITION | MF_CHECKED, IDM_CHARPANEL_ENCODE + nEncoding, L"Default");
				++nEncoding;
			}
		}
	}
	if (_hPopupMenu != nullptr) {
		POINT p;
		::GetCursorPos(&p);
		::TrackPopupMenu(_hPopupMenu, 0, p.x, p.y, 0, _hSelf, NULL);
	}
}
#endif

INT_PTR CALLBACK AnsiCharPanel::run_dlgProc(UINT message, WPARAM wParam, LPARAM lParam)
{
    switch (message)
    {
        case WM_INITDIALOG :
        {
			NppParameters *nppParam = NppParameters::getInstance();
			NativeLangSpeaker *pNativeSpeaker = nppParam->getNativeLangSpeaker();
			generic_string valStr = pNativeSpeaker->getAttrNameStr(TEXT("Value"), "AsciiInsertion", "ColumnVal");
			generic_string octStr = pNativeSpeaker->getAttrNameStr(TEXT("Oct"), "AsciiInsertion", "ColumnOct");	//+[JOJO]
			generic_string hexStr = pNativeSpeaker->getAttrNameStr(TEXT("Hex"), "AsciiInsertion", "ColumnHex");
			generic_string charStr = pNativeSpeaker->getAttrNameStr(TEXT("Character"), "AsciiInsertion", "ColumnChar");

			_listView.addColumn(columnInfo(valStr, nppParam->_dpiManager.scaleX(38)));
			_listView.addColumn(columnInfo(octStr, nppParam->_dpiManager.scaleX(34)));	//+[JOJO]
			_listView.addColumn(columnInfo(hexStr, nppParam->_dpiManager.scaleX(38)));
			_listView.addColumn(columnInfo(charStr, nppParam->_dpiManager.scaleX(48)));
			_listView.addColumn(columnInfo(L"", nppParam->_dpiManager.scaleX(52)));	//+[JOJO]

			_listView.init(_hInst, _hSelf);
#ifdef MOD_CHAR_PANEL_ENCODING
			_listView.setValues(::GetACP());
#else
			int codepage = (*_ppEditView)->getCurrentBuffer()->getEncoding();
			_listView.setValues(codepage==-1?0:codepage);
#endif
			_listView.display();

            return TRUE;
        }
#ifdef MOD_CHAR_PANEL_ENCODING
		case WM_DESTROY:
			if (_hPopupMenu) DestroyMenu(_hPopupMenu);
			break;
#endif

#ifdef MOD_CHAR_PANEL_ENCODING
		case WM_COMMAND:
		{
			int i = LOWORD(wParam) - IDM_CHARPANEL_ENCODE;
			if (0 <= i && i < nEncoding) {
				int codepage = ENCODINGS[i];
				if (codepage != _listView.getCodepage()) {
					_listView.changeCodepage(codepage);
					for (int x = 0; x < nEncoding; ++x)
						::CheckMenuItem(_hPopupMenu, x, x == i ? MF_BYPOSITION | MF_CHECKED : MF_BYPOSITION | MF_UNCHECKED);
				}
				return TRUE;
			}
			break;
		}
#endif
		case WM_NOTIFY:
		{
			switch (((LPNMHDR)lParam)->code)
			{
#ifdef MOD_CHAR_PANEL_ENCODING
				case NM_RCLICK:
					trackPopupMenu();
					return TRUE;
#endif
				case NM_DBLCLK:
				{
					LPNMITEMACTIVATE lpnmitem = (LPNMITEMACTIVATE) lParam;
					int i = lpnmitem->iItem;

					if (i == -1)
						return TRUE;

					insertChar(static_cast<unsigned char>(i));
					return TRUE;
				}

				case LVN_KEYDOWN:
				{
					switch (((LPNMLVKEYDOWN)lParam)->wVKey)
					{
#ifdef MOD_CHAR_PANEL_ENCODING
						case VK_APPS:
							trackPopupMenu();
							return TRUE;
#endif
						case VK_RETURN:
						{
							int i = _listView.getSelectedIndex();

							if (i == -1)
								return TRUE;

							insertChar(static_cast<unsigned char>(i));
							return TRUE;
						}
						default:
							break;
					}
				}
				break;

				default:
					break;
			}
		}
		return TRUE;

        case WM_SIZE:
        {
            int width = LOWORD(lParam);
            int height = HIWORD(lParam);
			::MoveWindow(_listView.getHSelf(), 0, 0, width, height, TRUE);
            break;
        }

        default :
            return DockingDlgInterface::run_dlgProc(message, wParam, lParam);
    }
	return DockingDlgInterface::run_dlgProc(message, wParam, lParam);
}

void AnsiCharPanel::insertChar(unsigned char char2insert) const
{
#ifdef FIX_CHAR_PANEL_CODEPAGE
	WCHAR strW[2];
	char strA[4/*UTF8MaxBytes*/];
	size_t lenA;
	const int lenW = MultiByteToWideChar(_listView.getCodepage(), MB_ERR_INVALID_CHARS, (char*)&char2insert, 1, strW, _countof(strW));
	if (lenW > 0)
		lenA = WideCharToMultiByte(65001/*CP_UTF8*/, 0, strW, lenW, strA, _countof(strA), NULL, NULL);
	else
		{ lenA = 1; strA[0] = (char)char2insert; }
	(*_ppEditView)->execute(SCI_REPLACESEL, 0, (LPARAM)"");
	(*_ppEditView)->execute(SCI_ADDTEXT, lenA, (LPARAM)strA);
#else
    char charStr[2];
    charStr[0] = char2insert;
    charStr[1] = '\0';
    wchar_t wCharStr[10];
    char multiByteStr[10];
	int codepage = (*_ppEditView)->getCurrentBuffer()->getEncoding();
	if (codepage == -1)
	{
		bool isUnicode = ((*_ppEditView)->execute(SCI_GETCODEPAGE) == SC_CP_UTF8);
		if (isUnicode)
		{
			MultiByteToWideChar(0, 0, charStr, -1, wCharStr, sizeof(wCharStr));
			WideCharToMultiByte(CP_UTF8, 0, wCharStr, -1, multiByteStr, sizeof(multiByteStr), NULL, NULL);
		}
		else // ANSI
		{
			multiByteStr[0] = charStr[0];
			multiByteStr[1] = charStr[1];
		}
	}
	else
	{
		MultiByteToWideChar(codepage, 0, charStr, -1, wCharStr, sizeof(wCharStr));
		WideCharToMultiByte(CP_UTF8, 0, wCharStr, -1, multiByteStr, sizeof(multiByteStr), NULL, NULL);
	}
	(*_ppEditView)->execute(SCI_REPLACESEL, 0, reinterpret_cast<LPARAM>(""));
	size_t len = (char2insert < 128) ? 1 : strlen(multiByteStr);
	(*_ppEditView)->execute(SCI_ADDTEXT, len, reinterpret_cast<LPARAM>(multiByteStr));
#endif
	(*_ppEditView)->getFocus();
}