/****************************************************************************/
/* The developnent of this program is partly supported by IPA.              */
/* (Infomation-Technology Promotion Agency, Japan).                         */
/****************************************************************************/

/****************************************************************************/
/*  cloop_optimizer.c                                                       */
/*  Copyright : Copyright (C) 2006 ALPHA SYSTEMS INC.                       */
/*  Authors   : TAN Hideyuki (tanh@alpha.co.jp)                             */
/*              ALPHA SYSTEMS INC. knoppix team (knoppix@alpha.co.jp)       */
/*                                                                          */
/*  This is free software; you can redistribute it and/or modify            */
/*  it under the terms of the GNU General Public License as published by    */
/*  the Free Software Foundation; either version 2 of the License, or       */
/*  (at your option) any later version.                                     */
/*                                                                          */
/*  This software is distributed in the hope that it will be useful,        */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of          */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           */
/*  GNU General Public License for more details.                            */
/*                                                                          */
/*  You should have received a copy of the GNU General Public License       */
/*  along with this software; if not, write to the Free Software            */
/*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,  */
/*  USA.                                                                    */
/****************************************************************************/

#include <stdio.h>
#include <stdlib.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <netinet/in.h>
#include <string.h>
#include <ctype.h>

#include "compressed_loop.h"
#include "cloop_optimizer.h"
#include "cloopprofiler_msg.h"

static int debug = 0;
float opt_progress;
int do_optimize;
int verbose;
char cloop_opt_msg[CLOOP_OPT_MSG_SIZE/2];

struct optimizer {
  int in_fd;
  int out_fd;
  char *cloop_outfile;

  struct cloop_head header;
  unsigned int total_offsets;
  loff_t *offsets;

  unsigned int *boot_blk_list;
  unsigned int boot_blk_count;

  unsigned int *appli_blk_list;
  unsigned int appli_blk_count;

  unsigned int *opt_blk_list;

  void (*progress)(void *);
  void *progress_args;
};

int cloop_opt_check_infile(const char *infile, struct optimizer *opt) {

  opt->in_fd = open(infile, O_RDONLY);
  if (opt->in_fd < 0) {
    memcpy(&cloop_opt_msg, infile, strlen(infile));
    return CLOOP_OPT_FOPENERR;
  }

  if (read(opt->in_fd, &opt->header, sizeof(struct cloop_head))
      != sizeof(struct cloop_head)) {
    memcpy(&cloop_opt_msg, infile, strlen(infile));
    return CLOOP_OPT_FREADERR;
  }

  if (opt->header.preamble[0x0B] != 'V'
      || opt->header.preamble[0x0C] != '2'
      || strstr(opt->header.preamble, "OPT")) {
    memcpy(&cloop_opt_msg, infile, strlen(infile));
    return CLOOP_OPT_CLOOP_VERERR;
  }

  opt->total_offsets = ntohl(opt->header.num_blocks) +1;
  opt->offsets = malloc(sizeof(loff_t) * opt->total_offsets);
  if (! opt->offsets) {
    return CLOOP_OPT_MEMERR;
  }

  if (read(opt->in_fd, opt->offsets, (sizeof(loff_t) * opt->total_offsets))
      != sizeof(loff_t) * opt->total_offsets) {
    memcpy(&cloop_opt_msg, infile, strlen(infile));
    return CLOOP_OPT_FREADERR;
  }

  return CLOOP_OPT_OK;
}

void cloop_opt_clear(struct optimizer *opt) {

  if (opt->in_fd > 0) close(opt->in_fd);
  if (opt->out_fd > 0) close(opt->out_fd);
  if (opt->offsets) free(opt->offsets);
  if (opt->boot_blk_list) free(opt->boot_blk_list);
  if (opt->appli_blk_list) free(opt->appli_blk_list);
  if (opt->opt_blk_list) free(opt->opt_blk_list);
}

int cloop_opt_read_listfile(const char *listfile, struct optimizer *opt,
			    unsigned int **blk_list, unsigned int *read_count) {

  FILE *fp;
  char buf[1024];
  int len = 0;
  unsigned int i;

  unsigned int max_blknum = ntohl(opt->header.num_blocks);

  fp = fopen(listfile, "r");
  if (! fp) {
    memcpy(&cloop_opt_msg, listfile, strlen(listfile));
    return CLOOP_OPT_FOPENERR;
  }

  *blk_list = malloc(sizeof(unsigned int) * max_blknum);
  if (! *blk_list) {
    return CLOOP_OPT_MEMERR;
  }

  *read_count = 0;
  while(fgets(buf, 512, fp) != 0) {
    if (*read_count > max_blknum)
      break;

    len = strlen(buf);
    buf[len -1] = '\0';

    for (i = 0; i < len; i++)
      if (! (isdigit(buf[i]) || buf[i] == '.' || buf[i] == ' '))
	break;

    if (i == len -1) { // ok line.

      char *work = strtok(buf, " ");
      if (! work) break;
      int array = 0;
      int no = atoi(work);
      if (no > max_blknum) break; // onakaippai.

      int blk_num = 0;      
      for (array++; ; array++) {
	work = strtok(NULL, " ");
	if (! work) break;
	if (array == 1) {
	  blk_num = atoi(work);
	  break;
	}
      }

      if (blk_num > max_blknum) continue; // oioi.

      for(i = 0; i < *read_count; i++)
	if ((*blk_list)[i] == blk_num) break;
      if (i != *read_count) continue; // found.

      (*blk_list)[*read_count] = blk_num;
      (*read_count)++;
    }
  }

  fclose(fp);

  return CLOOP_OPT_OK;
}

int cloop_opt_mklist(struct optimizer *opt) {

  unsigned int i, j, idx;

  opt->opt_blk_list = malloc(sizeof(unsigned int) * opt->total_offsets);
  if (! opt->opt_blk_list) {
    return CLOOP_OPT_MEMERR;
  }

  // padding other blocks
  idx = 0;
  for (i = 0; i < opt->total_offsets -1; i++) {

    // check boot block list
    for (j = 0; j < opt->boot_blk_count; j++)
      if (opt->boot_blk_list[j] == i) break;
    if (j != opt->boot_blk_count) continue;


    // check application block list
    if (opt->appli_blk_list) {
      for (j = 0; j < opt->appli_blk_count; j++)
	if (opt->appli_blk_list[j] == i) break;
      if (j != opt->appli_blk_count) continue;
    }

    // padding
    opt->opt_blk_list[idx] = i;
    idx++;
  }
  
  // append boot blocks
  for (i = 0; i < opt->boot_blk_count; i++) {
    opt->opt_blk_list[opt->total_offsets - 1 - opt->boot_blk_count + i] = opt->boot_blk_list[i];
  }

  // append application blocks
  if (opt->appli_blk_list) {

    for (i = 0; i < opt->appli_blk_count; i++) {

      // check boot block list
      for (j = 0; j < opt->boot_blk_count; j++)
	if (opt->boot_blk_list[j] == opt->appli_blk_list[i]) break;
      if (j != opt->boot_blk_count) continue;

      // append
      opt->opt_blk_list[idx] = opt->appli_blk_list[i];
      idx++;
    }
  }

  if (debug) { // dump
    for (i = 0; i < 10; i++)
      fprintf(stderr, "boot_blk_list[%05u] : %05u\n", i, opt->boot_blk_list[i]);
    for (i = opt->boot_blk_count - 10; i < opt->boot_blk_count; i++)
      fprintf(stderr, "boot_blk_list[%05u] : %05u\n", i, opt->boot_blk_list[i]);
    if (opt->appli_blk_list)
      for (i = 0; i < 10; i++)
	fprintf(stderr, "appli_blk_list[%05u] : %05u\n", i, opt->appli_blk_list[i]);
    if (opt->appli_blk_list)
      for (i = opt->appli_blk_count - 10; i < opt->appli_blk_count; i++)
	fprintf(stderr, "appli_blk_list[%05u] : %05u\n", i, opt->appli_blk_list[i]);
    for (i = 0; i < opt->total_offsets -1; i++)
      fprintf(stderr, "opt_blk_list[%05u] : %05u\n", i, opt->opt_blk_list[i]);
  }

  free(opt->boot_blk_list);
  opt->boot_blk_list = NULL;

  if (opt->appli_blk_list) {
    free(opt->appli_blk_list);
    opt->appli_blk_list = NULL;
  }

  return CLOOP_OPT_OK;
}

int cloop_opt_write(struct optimizer *opt) {

  unsigned int i;
  struct cloop_head opt_header;

  struct block_info *opt_blk_info_tbl;
  unsigned long long offset_tmp;

  unsigned long opt_blk_info_sum;
  unsigned long write_sum;

  // header
  memset(opt_header.preamble, 0, sizeof(opt_header.preamble));
  memcpy(opt_header.preamble, CLOOP_OPT_PREAMBLE, sizeof(CLOOP_OPT_PREAMBLE));

  opt_header.block_size = opt->header.block_size;
  opt_header.num_blocks = opt->header.num_blocks;

  if (verbose) fprintf(stderr, "header write.\n");
  if (write(opt->out_fd, &opt_header, sizeof(opt_header)) < 0) {
    memcpy(&cloop_opt_msg, opt->cloop_outfile, strlen(opt->cloop_outfile));
    return CLOOP_OPT_FWRITEERR;
  }

  // block info table
  opt_blk_info_tbl = malloc(sizeof(struct block_info) * opt->total_offsets);
  if (! opt_blk_info_tbl) {
    return CLOOP_OPT_MEMERR;
  }

  // calc offset
  for (i = 0; i < opt->total_offsets -1; i++) {
    opt_blk_info_tbl[i].optidx = htonl(opt->opt_blk_list[i]);
    opt_blk_info_tbl[i].size
      = htonl(be64_to_cpu(opt->offsets[i+1]) - be64_to_cpu(opt->offsets[i]));
  }

  offset_tmp = sizeof(opt_header) + sizeof(struct block_info) * opt->total_offsets;
  if (verbose) fprintf(stderr, "start_offset = %Lu\n", offset_tmp);

  opt_blk_info_sum = 0;
  for(i = 0; i < opt->total_offsets -1; i++) {
    opt_blk_info_tbl[opt->opt_blk_list[i]].offset = cpu_to_be64(offset_tmp);
    offset_tmp += be64_to_cpu(opt->offsets[opt->opt_blk_list[i]+1]) - be64_to_cpu(opt->offsets[opt->opt_blk_list[i]]);
    opt_blk_info_sum += ntohl(opt_blk_info_tbl[opt->opt_blk_list[i]].size);
  }
  if (verbose) fprintf(stderr, "opt_blk_info_sum = %lu, (%luKB)\n",  opt_blk_info_sum, opt_blk_info_sum/1024);

  // block_info table write
  if (verbose) fprintf(stderr, "block info table write.\n");
  if (write(opt->out_fd, opt_blk_info_tbl, sizeof(struct block_info) * opt->total_offsets) < 0) {
    memcpy(&cloop_opt_msg, opt->cloop_outfile, strlen(opt->cloop_outfile));
    return CLOOP_OPT_FWRITEERR;
  }

  // block data write
  if (verbose) fprintf(stderr, "block data write.\n");
  write_sum = 0;
  for(i = 0; i < opt->total_offsets -1; i++) {
    void *blk_data = NULL;
    int ret;
    unsigned long blk_len;
    unsigned int index;

    blk_len = ntohl(opt_blk_info_tbl[opt->opt_blk_list[i]].size);
    blk_data = malloc(blk_len);
    if (! blk_data) {
      return CLOOP_OPT_MEMERR;
    }
    memset(blk_data, 0, blk_len);

    ret = lseek(opt->in_fd, be64_to_cpu(opt->offsets[opt->opt_blk_list[i]]), SEEK_SET);
    if (ret < 0) {
      memcpy(&cloop_opt_msg, opt->cloop_outfile, strlen(opt->cloop_outfile));
      return CLOOP_OPT_FSEEKERR;
    }

    ret = read(opt->in_fd, blk_data, blk_len);
    if (ret != blk_len) {
      memcpy(&cloop_opt_msg, opt->cloop_outfile, strlen(opt->cloop_outfile));
      return CLOOP_OPT_FREADERR;
    }
    
    ret = write(opt->out_fd, blk_data, blk_len);
    if (ret != blk_len) {
      memcpy(&cloop_opt_msg, opt->cloop_outfile, strlen(opt->cloop_outfile));
      return CLOOP_OPT_FWRITEERR;
    }

    free(blk_data);

    index = ntohl(opt_blk_info_tbl[i].optidx);

    opt_progress = i / (float)(opt->total_offsets -1);

    if (opt->progress)
      opt->progress(opt->progress_args);

    if (verbose)
      fprintf(stderr,
	      "%05d/%05d: %2.03f%%, optidxno=%05d, optidxsize=%u, write=%d\n",
	      i +1, opt->total_offsets -1, opt_progress * 100, index,
	      ntohl(opt_blk_info_tbl[index].size), ret);
    write_sum += ret;
  }
  if (verbose) fprintf(stderr, "write sum = %lu, (%luKB)\n", write_sum, write_sum/1024);

  return CLOOP_OPT_OK;
}
      
int cloop_optimizer(const char *cloop_infile, const char* cloop_outfile,
		    const char *boot_list_file, const char *appli_list_file,
		    void (*progress)(void *), void *progress_args) {

  struct optimizer opt;
  int ret = 0;
  memset(&opt, 0, sizeof(struct optimizer));
  memset(&cloop_opt_msg, 0, sizeof(cloop_opt_msg));

  opt.cloop_outfile = (char *)cloop_outfile;

  opt.progress = progress;
  opt.progress_args = progress_args;

  ret = cloop_opt_check_infile(cloop_infile, &opt);
  if (ret != CLOOP_OPT_OK) {
    cloop_opt_clear(&opt);
    return ret;
  }

  if (verbose)
    fprintf(stderr, "block_size = %u\nnum_blocks = %u\n",
	    ntohl(opt.header.block_size), ntohl(opt.header.num_blocks));

  if (! do_optimize) {
    cloop_opt_clear(&opt);
    return CLOOP_OPT_ABORT;
  }

  ret = cloop_opt_read_listfile(boot_list_file, &opt, &opt.boot_blk_list, &opt.boot_blk_count);
  if (ret != CLOOP_OPT_OK) {
    cloop_opt_clear(&opt);
    return ret;
  }

  if (! do_optimize) {
    cloop_opt_clear(&opt);
    return CLOOP_OPT_ABORT;
  }

  if (appli_list_file) {
    ret = cloop_opt_read_listfile(appli_list_file, &opt, &opt.appli_blk_list, &opt.appli_blk_count);
    if (ret != CLOOP_OPT_OK) {
      cloop_opt_clear(&opt);
      return ret;
    }
  }

  if (! do_optimize) {
    cloop_opt_clear(&opt);
    return CLOOP_OPT_ABORT;
  }

  // output
  if (cloop_outfile) {
    opt.out_fd = open(cloop_outfile, O_CREAT|O_WRONLY);
    if (opt.out_fd < 0) {
      memcpy(&cloop_opt_msg, cloop_outfile, strlen(cloop_outfile));
      cloop_opt_clear(&opt);
      return CLOOP_OPT_FOPENERR;
    }
  } else {
    opt.out_fd = STDOUT_FILENO;
  }

  // Go!
  ret = cloop_opt_mklist(&opt);

  if (! do_optimize) {
    cloop_opt_clear(&opt);
    return CLOOP_OPT_ABORT;
  }
  
  ret = cloop_opt_write(&opt);

  cloop_opt_clear(&opt);

  return ret;
}

void cloop_opt_putmsg(char *dst, char *src) {
  if (cloop_opt_msg != NULL && strlen(cloop_opt_msg))
    sprintf(dst, "%s : %s\n", cloop_opt_msg, src);
  else
    sprintf(dst, "%s\n", src);
}

char* get_optimizer_message(int msgid) {
  char *msg = NULL;

  msg = malloc(CLOOP_OPT_MSG_SIZE);
  if (! msg) {
    fprintf(stderr, "get_optimizer_message malloc error.");
    abort();
  }
  memset(msg, '\0', CLOOP_OPT_MSG_SIZE);

  switch(msgid) {
  case CLOOP_OPT_OK:
    sprintf(msg, "%s", CLOOP_OPT_MSG_OK);
    break;
  case CLOOP_OPT_FOPENERR:
    cloop_opt_putmsg(msg, ERR_FOPEN);
    break;
  case CLOOP_OPT_FWRITEERR:
    cloop_opt_putmsg(msg, ERR_WRITE);
    break;
  case CLOOP_OPT_FREADERR:
    cloop_opt_putmsg(msg, ERR_READ);
    break;
  case CLOOP_OPT_FSEEKERR:
    cloop_opt_putmsg(msg, ERR_SEEK);
    break;
  case CLOOP_OPT_MEMERR:
    cloop_opt_putmsg(msg, ERR_MEMORY);
    break;
  case CLOOP_OPT_CLOOP_VERERR:
    cloop_opt_putmsg(msg, ERR_VERSION);
    break;
  case CLOOP_OPT_ABORT:
    cloop_opt_putmsg(msg, CLOOP_OPT_MSG_ABORT);
    break;
  default:
    break;
  }

  return msg;
}
