require 'log'

class WikiController < ApplicationController
  before_filter :set_charset
  caches_action :view
  
  def set_charset
    @headers["Content-Type"] = "text/html; charset=UTF-8" 
    #@headers["Content-Type"] = "application/xhtml+xml; charset=UTF-8" 
  end
  
  def index
    @tiddlers = Tiddler.find(:all, :order => "length(title) desc, title")
    @color_tiddlers = { "none" => Tiddler.find(:all, :conditions => "color is null", :order => 'title') }
    Tiddler::COLORS.each do |c|
      @color_tiddlers[c] = Tiddler.find(:all, :conditions => "color = '#{c}'", :order => 'title')
    end
        
    if @params['title']
      @tiddler = Tiddler.find(:first, :conditions => "title = '#{unescape_slash(@params['title'])}'")
    else
      @tiddler = Tiddler.find(:first, :conditions => "title = 'StartHere'")
    end
    unless @tiddler
      @tiddler = Tiddler.new
      if @params['title']
        @tiddler.title = unescape_slash(@params['title'])
      else
        @tiddler.title = "StartHere"
      end
      @tiddler.content = nil
      @tiddler.autolink = 1
    end
    @tiddler.render!(WikiLinkManager.new, @tiddlers)
    
    @menu = Tiddler.find(:first, :conditions => "title = 'MainMenu'")
    unless @menu
      @menu = Tiddler.new
      @menu.title = "MainMenu"
      @menu.content = nil
      @menu.autolink = 1
    end
    @menu.render!(MenuLinkManager.new, @tiddlers)
    
    # expire the cache of the tiddler that contains dynamic content such as Howm schedule
    Tiddler.find(:all, :conditions => "expire_cache = 1").each do |t|
      expire_action :action => "view", :title => t.escaped_title
    end
  end
  
  def delete
    tiddlers = Tiddler.find(:all, :order => "length(title) desc, title")
    @tiddler = Tiddler.find(:first, :conditions => "title = '#{unescape_slash(@params['title'])}'")
    if @tiddler
      LesserWiki::Log.delete(@tiddler)
      delete_and_expire(@tiddler, tiddlers)
    end
  end
  
  def new
    @tiddler = Tiddler.find(:first, :conditions => "title = '#{unescape_slash(@params['title'])}'")
    if @tiddler
      redirect_to :action => 'view', :title => @params['title']
    end
    @tiddler = Tiddler.new
    @tiddler.title = unescape_slash(@params['title'])
    @tiddler.content = ""
    @tiddler.autolink = 1
    @tiddler.render!(WikiLinkManager.new)
  end
  
  def view
    @tiddlers = Tiddler.find(:all, :order => "length(title) desc, title")
    @tiddler = Tiddler.find(:first, :conditions => "title = '#{unescape_slash(@params['title'])}'")
    unless @tiddler
      redirect_to :action => 'view_nonexist', :title => @params['title']
    end
    @tiddler.render!(WikiLinkManager.new)
    @use_effect = true
  end
  
  def view_nonexist
    @tiddler = Tiddler.new
    @tiddler.title = unescape_slash(@params['title'])
    @tiddler.content = nil
    @tiddler.autolink = 1
    @tiddler.render!(WikiLinkManager.new)
    @use_effect = true
  end
  
  def save
    @message = nil
    autolink = @params['autolink'].to_i
    color = Tiddler::COLORS.include?(@params['color']) ? @params['color'] : nil
    tiddlers = Tiddler.find(:all, :order => "length(title) desc, title")
    
    if @params['title'] == @params['last_title']
      @tiddler = Tiddler.find(:first, :conditions => "title = '#{unescape_slash(@params['title'])}'")
      if @tiddler
        if @tiddler.content_digest == @params['last_content_digest']
          if !@tiddler.same_content(@params['content']) or
             @tiddler.autolink != autolink or
             @tiddler.color != color then
            save_and_expire(@tiddler, unescape_slash(@params['title']), @params['content'],
                            @params['aliases'], autolink, color, tiddlers)
            LesserWiki::Log.edit(@tiddler)
          end
        else
          @message = "Conflict detected."
          @tiddler.last_content = @tiddler.content
          @tiddler.content = @params['content']
          @tiddler.autolink = autolink
          render_action :view
        end
      else  # @tiddler == nil
        @tiddler = Tiddler.new
        save_and_expire(@tiddler, unescape_slash(@params['title']), @params['content'],
                        @params['aliases'], autolink, color, tiddlers)
        LesserWiki::Log.create(@tiddler)
      end
    else  # @params['title'] != @params['last_title']
      @tiddler = Tiddler.find(:first, :conditions => "title = '#{unescape_slash(@params['title'])}'")
      last_tiddler = Tiddler.find(:first, :conditions => "title = '#{unescape_slash(@params['last_title'])}'")
      if @tiddler
        @message = "#{@tiddler.title} is already exist."
        @tiddler.title = @params['last_title']
        @tiddler.last_content = @tiddler.content
        @tiddler.content = @params['content']
        @tiddler.autolink = autolink
        render_action :view
      else
        if last_tiddler and last_tiddler.content_digest != @params['last_content_digest']
          @message = "Conflict detected."
          @tiddler = last_tiddler
          @tiddler.last_content = @tiddler.content
          @tiddler.content = @params['content']
          @tiddler.autolink = autolink
          render_action :view
        else
          @tiddler = Tiddler.new
          save_and_expire(@tiddler, unescape_slash(@params['title']), @params['content'],
                          @params['aliases'], autolink, color, tiddlers)
          if last_tiddler
            save_and_expire(last_tiddler, @params['last_title'],
                            "#{last_tiddler.title} &rarr; #{@tiddler.title}",
                            @params['aliases'], last_tiddler.autolink,
                            last_tiddler.color, tiddlers)
          end
          LesserWiki::Log.rename(@tiddler)
        end
      end
    end
    
    # expire_cache flag is set by HowmChunk::Schedule or HowmChunk::Todo
    last_expire_cache = @tiddler.expire_cache
    @tiddler.expire_cache = false
    @tiddler.render!(WikiLinkManager.new)
    if last_expire_cache != @tiddler.expire_cache
      @tiddler.update_attribute("expire_cache", @tiddler.expire_cache)
    end
    
    reminder_updated = false
    unless @tiddler.rendered_content.reminders
      @tiddler.reminders.each {|r| r.destroy }
    else
      @tiddler.reminders.each do |r|
        if @tiddler.rendered_content.reminders.select {|e| e.equals(r) }.empty?
          r.destroy
          reminder_updated = true
        end
      end
    end
    if @tiddler.rendered_content.reminders
      @tiddler.rendered_content.reminders.each do |r|
        if @tiddler.reminders.select {|e| e.equals(r) }.empty?
          r.save
          reminder_updated = true
        end
      end
    end
    
    if reminder_updated
      # expire the cache of the tiddler that contains dynamic content such as Howm schedule
      Tiddler.find(:all, :conditions => "expire_cache = 1").each do |t|
        expire_action :action => "view", :title => t.escaped_title
      end
    end
    
    @tiddler.saved = true;
  end
  
  def search
    link_manager = WikiLinkManager.new
    tiddlers_for_render = Tiddler.find(:all, :order => "length(title) desc, title")
    
    search = @params['search'].gsub(/%/, '\%').gsub(/_/, '\_')
    @tiddlers = Tiddler.find(:all, :conditions => "title like '%#{search}%' or content like '%#{search}%'",
                             :order => "length(title) desc, title")
    @tiddlers.each do |t|
      t.render!(link_manager, tiddlers_for_render)
    end
  end
  
  private
  
  def save_and_expire(tiddler, title, content, alias_str, autolink, color, tiddlers)
    expire_caches(tiddler, tiddlers, content)
    
    Tiddler.transaction do
      tiddler.title = title
      tiddler.content = content
      tiddler.autolink = autolink
      tiddler.color = color
      tiddler.save
      
      # split the alias names (this is ugly code...)
      alias_str = alias_str.gsub(/^\s*(.*)\s*$/, '\\1').gsub(/\s+/, " ")
      alias_str.gsub!(/\[\[\s*(.*)\s*\]\]/) {|s| $1.sub(/ $/, "").gsub(/ /, "\t") }
      aliases = alias_str.split(/ /)
      aliases.each {|a| a.gsub!(/\t/, " ") }.uniq!
      
      # delete old aliases
      tiddler.aliases.select {|a| !aliases.include?(a.name) }.each do |a|
        delete_alias(tiddler, a)
      end
      
      # create new aliases
      aliases.each do |a|
        l = Alias.find(:first, :conditions => "tiddler_id = #{tiddler.id} and name = '#{a}'")
        create_alias(tiddler, a) unless l
      end
    end
  end
  
  def create_alias(tiddler, name)
    new_alias = Alias.new
    new_alias.tiddler = tiddler
    new_alias.name = name
    tiddler.aliases << new_alias
  end
  
  def delete_alias(tiddler, alias_)
    tiddler.aliases.delete(alias_)
    alias_.destroy
  end
  
  def delete_and_expire(tiddler, tiddlers)
    expire_caches(tiddler, tiddlers)
    tiddler.destroy
  end
  
  def expire_caches(tiddler, tiddlers, new_content = nil)
    # expire the cache of the tiddler that *might* contain the title or alias of this tiddler
    tiddlers.select {|t|
      /#{Regexp.escape(t.title)}/ =~ tiddler.content or
      (new_content and /#{Regexp.escape(t.title)}/ =~ new_content) or
      /#{Regexp.escape(tiddler.title)}/ =~ t.content or
      t.aliases.select {|a| /#{Regexp.escape(a.name)}/ =~ tiddler.content or
                            (new_content and /#{Regexp.escape(a.name)}/ =~ new_content) }.size > 0
    }.each do |t|
      expire_action :action => "view", :title => t.escaped_title
    end
    
    # expire the cache of this tiddler
    expire_action :action => "view", :title => tiddler.escaped_title
    
    # expire the cache of RSS
    expire_page :controller => "rss", :action => "index"
  end
  
end
