require 'digest/md5'
require 'wiki_content'
require 'link_manager'
require 'my_doc_diff'

class Tiddler < ActiveRecord::Base
  attr_writer :link_manager
  attr_accessor :last_title, :last_content, :saved
  attr_reader :rendered_content
  has_many :aliases, :dependent => true
  has_many :reminders, :class_name => "LesserWiki::Reminder", :dependent => true
  has_many :logs, :class_name => "LesserWiki::Log", :order => "created_at desc"
  
  COLORS = %w{red orange yellow green blue purple gray}.freeze
  
  def escaped_title
    escape_slash(self.title)
  end
  
  def render!(link_manager = nil, tiddlers = nil)
    # Give priority to the longest title name. 
    tiddlers = Tiddler.find(:all, :order => "length(title) desc, title") unless tiddlers
    
    if link_manager
      @link_manager = link_manager
    else
      @link_manager = LinkManager.new
    end
    
    @rendered_content = WikiContent.new(self, tiddlers, { :mode => :show }).render!
  end
  
  def make_link(*args, &block)
    args[2][:tiddler] = self
    if @link_manager
      @link_manager.make_link(*args, &block)
    else
      LinkManager.new.make_link(*args, &block)
    end
  end
  
  def make_alias_link(*args, &block)
    args[2][:tiddler] = self
    if @link_manager
      @link_manager.make_link(*args, &block)
    else
      LinkManager.new.make_link(*args, &block)
    end
  end
  
  def make_reminder_link(*args, &block)
    args[2][:tiddler] = self
    unless @reminder_link_manager
      @reminder_link_manager = ReminderLinkManager.new
    end
    @reminder_link_manager.make_link(*args, &block)
  end
  
  def same_content(content)
    Digest::MD5.hexdigest("#{self.content}") == Digest::MD5.hexdigest("#{content}")
  end

  def content_digest
    Digest::MD5.hexdigest("#{self.content}")
  end

  def last_content_digest
    if @last_content
      Digest::MD5.hexdigest("#{self.last_content}")
    else
      nil
    end
  end
  
  def diff
    if @last_content
      Tiddler.diff_between_2contents("#{self.last_content}", "#{self.content}")
    else
      nil
    end
  end
  
  def self.diff_between_2contents(old_content, new_content, resolution = 'line', digest = false)
    docdiff = MyDocDiff.new
    
    old_doc = Document.new("#{old_content}", 'UTF-8', 'LF')
    new_doc = Document.new("#{new_content}", 'UTF-8', 'LF')
    
    docdiff_config = {
      :resolution    => resolution,
      :encoding      => "auto",
      :eol           => "auto",
      :format        => "user",
      :cache         => true,
      :digest        => digest,
      :verbose       => false,
      
      #:tag_common_start  => '<span class="docdiff_common">',
      #:tag_common_end    => '</span>',
      :tag_del_start     => '<span class="docdiff_del"><del>',
      :tag_del_end       => '</del></span>',
      :tag_add_start     => '<span class="docdiff_add"><ins>',
      :tag_add_end       => '</ins></span>',
      :tag_change_before_start => '<span class="docdiff_before_change"><del>',
      :tag_change_before_end => '</del></span>',
      :tag_change_after_start => '<span class="docdiff_after_change"><ins>',
      :tag_change_after_end => '</ins></span>',
      :tag_position_start => '<!-- ',
      :tag_position_end => ' -->'
    }
    
    docdiff.config.update(docdiff_config)
    docdiff.run(old_doc, new_doc,
    docdiff.config[:resolution],
    docdiff.config[:format],
    docdiff.config[:digest])    
  end
  
  private
  
  def escape_slash(str)
    if str
      str.gsub(/\//, "  ")
    else
      nil
    end
  end
end
