require 'yaml'
require 'thread'

module I18n
=begin rdoc
  General purpose class for retrieving messages to support i18n.
  Designed similar to i18n support available in java web frameworks.
  
  The locale specific files are identified using a simple naming convention.
  The language and country code are appended to the end of the filename before
  the extension. You can make up your own codes but it is better to use standard
  codes. Valid ISO language codes can be found here:
  http://www.ics.uci.edu/pub/ietf/http/related/iso639.txt .
  Also you can find a list of ISO country codes here:
  http://www.chemie.fu-berlin.de/diverse/doc/ISO_3166.html.
  
  Example filenames: (Note that all files have extention 'translation')
    messages.translation (The default. It has no languagecode-countrycode)
    messages_en-us.translation
    messages_fr-fr.translation
    
  The translation files are simple yaml files with entries of the following format:
    key: value
    range_error: Value not within range {0} to {1}
  Applications are coded to the 'key'. So when translating, translate <b>only</b> the
  'value'. 
  

  ==== General Ruby on Rails setup for i18n:
  - All files in the application including the translation files should be saved
    using UTF-8 encoding. If your editor does not support this get an editor which does.
  - Let ruby know that it needs to handle unicode.
    Add the following to the environment.rb file (or your custom environment file)
      $KCODE = 'u'
      require 'jcode'
  - Make sure the appropriate headers are sent back by the application to the 
    browser. Add following lines to the application.rb:
      before_filter :set_content_type
      def set_content_type
        @response.headers["Content-Type"] = "text/html; charset=UTF-8"
      end
      
  ==== Module specific setup:
  - Copy the message_resources.rb file in the distribution to your rails applications
    'lib' directory.
    
  - Set the basename for the translation files. Add this to the environment.rb file
    (or your custom environment file)
      require 'message_resources'
      I18n::MessageResources.basename="#{RAILS_ROOT}/config/i18n/messages"
 
  - Create the family of translation files in RAILS_ROOT/config/i18n directory. For example
    if you have to support english and french create the following files. Note that
    the file extention is 'translation'. The file without the language and country code
    is the default. If a locale is not found the value is picked up from the default file
      messages.translation
      messages_fr-fr.translation
      
  ==== Usage:
  - Code examples:
     I18n::MessageResources.instance.message(key)
    returns the value corresponding to key from messages.translation  (the default file)
      I18n::MessageResources.instance.message(key, "fr-fr")
    returns the value corresponding to key from messages_fr-fr.translation. The idea
    is to match the locale string ('fr-fr' in this case) to the languagecode-countrycode file name.
    
    Assumming entry in messages.translation file of:
      range_error: Value not within range {0} to {1}

      I18n::MessageResources.instance.message("range_error", nil, ["15", "20"])
    returns 'Value not within range 15 to 20'. The {0} and {1} .. are replaced by 
    values in the array argument. Supports any number of replaceable parameters.
    
    To pickup the locale per request in the application_helper.rb file add the following:
      def m(key, args=nil)
        I18n::MessageResources.instance.message(key, request.env['HTTP_ACCEPT_LANGUAGE'],args)
      end
      
    In the case where the users locale is stored in the session for example in 
    session[:locale] (probably set when the user logged in):
      def m(key, args=nil)
        I18n::MessageResources.instance.message(key, session[:locale], args)
      end
      
  - Rails .rhtml code (assuming the method 'm' as above is available in application_helper.rb)
      <%= m 'user_id' %> 
      <%= m ('range_error', ['15', '20']) %>
      
  - Others
    - If the 'return_key_when_not_found' argument is true (the default) and a message 
      is not found for the key, the key itself will be returned, otherwise, nil is returned

  ==== Notes:
  - I18n::MessageResources.instance returns a singleton object.
  - Make sure that the YAML syntax is valid. Otherwise you will get parse errors. 
  - The translations are stored in in-process memory. This means that each fastcgi 
    process will have their own copy of these translations.
  - All translation files are loaded during initialization. This occurs when the
    instance() method is invoked the first time in your application.
  - Thread safe just in case someone uses this module in a multi threaded environment
    
  Author: Antony Joseph
=end
  
  class MessageResources
    private_class_method :new
    @@message_resources = nil
    @@basename = nil
    @@bundles = {}
    @@mutex = Mutex.new
    
    # basename for the translation files. Should be set before invoking any methods.
    def self.basename= (basename)
      if (basename == nil || basename.empty?)
        raise ("basename cannot be empty")
      end
      # for some reason windows was giving problems. replace the back slashes if any
      @@basename = basename.gsub("\\", "/")
    end
    
    #returns a singleton
    def self.instance
      unless @@message_resources
        # Just in case some one uses this in a multi threaded environment
        @@mutex.synchronize do
          @@message_resources = new
          init_bundles
        end
      end 
      @@message_resources
    end
    
    def message(key, locale=nil, args=nil, return_key_when_not_found=true)
      bundle =  @@bundles[locale] || @@bundles["default"]
      if return_key_when_not_found
        msg = bundle[key] || key
      else
        msg = bundle[key]
      end
      if (args && msg)
        args.each_index{|i| r=Regexp.new("\\{#{i}\\}"); msg=msg.gsub(r,args[i].to_s) }
      end
      msg
    end
    
    private 
    # initialize the bundle with locale as key and  the properties hash as its value
    def self.init_bundles
      raise "basename attribute not set." unless @@basename
      if (Dir["#{@@basename}.translation"].length == 0 &&
        Dir["#{@@basename}_*.translation"].length == 0)
        raise "No files found for matching basename #{@@basename} pattern " 
      end
      
      # get the default translation file if any
      files = Dir["#{@@basename}.translation"]
      if files.length == 1
        @@bundles["default"] = load_translation(files[0])
      end
      
      # get files with "_languagecode-countrycode". 
      files = Dir["#{@@basename}_*.translation"]
      for file in files
        # if there is no default translation file set the first one found as the 
        # default translation file.
        if @@bundles.length == 0
          @@bundles["default"] = load_translation(file)
        else
          regexp = Regexp.new("#{@@basename}_(.*)\.translation")
          file =~ regexp
          # the $1 will be equal to something like "en-us"
          @@bundles["#$1"] = load_translation(file)
        end
      end
    end
    
    def self.load_translation(file_name)
      File.open(file_name) { |f| YAML::load(f) }
    end
  end
end