require 'log'
require 'link_manager'

class WikiController < ApplicationController
  before_filter :set_charset
  caches_action :view
  layout "wiki", :except => [ :view, :view_nonexist, :view_tag, :search ]
    
  def set_charset
    @headers["Content-Type"] = "text/html; charset=UTF-8"
    #@headers["Content-Type"] = "application/xhtml+xml; charset=UTF-8"
  end
  
  def index
    @message = nil
    @tag = nil
    @search = nil
    @result = nil
    @tiddlers = Tiddler.find(:all, :order => "length(title) desc, title")
    @color_tiddlers = { "none" => Tiddler.find(:all, :conditions => "color is null",
                                               :order => 'title') }
    Tiddler::COLORS.each do |c|
      @color_tiddlers[c] = Tiddler.find(:all, :conditions => ["color = ?", c],
                                        :order => 'title')
    end

    if @params['search']
      @search = @params['search']
      tiddlers1 = Tiddler.find(:all, :conditions => "title like '%#{@search}%'",
                               :order => "title")
      tiddlers2 = (Tiddler.find(:all, :conditions => "content like '%#{@search}%'",
                                :order => "title"))
      @result = tiddlers1 | tiddlers2
      if @result.empty?
        @message = i18n_message("flash_search_nonexist", [ @search ])
      end
    elsif @params['name']
      @tag = Tag.find(:first, :conditions => ["name = ?", @params['name']])
      unless @tag
        @tag = Tag.new
        @tag.id = 0
        @tag.name = @params['name']
        @message = i18n_message("flash_tag_nonexist", [@params['name']])
      end
    else
      if @params['title']
        @tiddler = Tiddler.find(:first, :conditions => "title = '#{unescape_slash(@params['title'])}'")
      else
        @tiddler = Tiddler.find(:first, :conditions => ["title = ?", "StartHere"])
      end
      unless @tiddler
        @tiddler = Tiddler.new
        if @params['title']
          @tiddler.title = unescape_slash(@params['title'])
        else
          @tiddler.title = "StartHere"
        end
        @tiddler.content = nil
        @tiddler.autolink = 1
      end
      @tiddler.render!(WikiLinkManager.instance, @tiddlers)
    end
    
    @menu = Tiddler.find(:first, :conditions => ["title = ?", "MainMenu"])
    unless @menu
      @menu = Tiddler.new
      @menu.title = "MainMenu"
      @menu.content = nil
      @menu.autolink = 1
    end
    @menu.render!(MenuLinkManager.instance, @tiddlers)
    
    cloud = TagCloud.new
    Tag.find(:all, :order => 'name').each do |t|
      cloud.add(t, t.tiddlers.size)
    end
    @tags = cloud.calculate
    
    # expire the cache of the tiddler that contains dynamic content such as Howm schedule
    Tiddler.find(:all, :conditions => "expire_cache = 1").each do |t|
      expire_action :action => "view", :title => t.escaped_title
    end
  end
  
  def delete
    tiddlers = Tiddler.find(:all, :order => "length(title) desc, title")
    @tiddler = Tiddler.find(:first, :conditions => "title = '#{unescape_slash(@params['title'])}'")
    if @tiddler
      LesserWiki::Log.delete(@tiddler)
      delete_and_expire(@tiddler, tiddlers)
    end
  end
  
  def new
    @tiddler = Tiddler.find(:first, :conditions => "title = '#{unescape_slash(@params['title'])}'")
    if @tiddler
      redirect_to :action => 'view', :title => @params['title']
    else
      @tiddler = Tiddler.new
      @tiddler.title = unescape_slash(@params['title'])
      @tiddler.content = ""
      @tiddler.autolink = 1
      @tiddler.render!(WikiLinkManager.instance)
    end
  end
  
  def view
    @tiddler = Tiddler.find(:first, :conditions => "title = '#{unescape_slash(@params['title'])}'")
    unless @tiddler
      redirect_to :action => "view_nonexist", :title => @params['title']
    else
      @tiddler.render!(WikiLinkManager.instance)
      @use_effect = true
    end
  end
  
  def view_nonexist
    @tiddler = Tiddler.new
    @tiddler.title = unescape_slash(@params['title'])
    @tiddler.content = nil
    @tiddler.autolink = 1
  end
  
  def save
    autolink = @params['autolink'].to_i
    color = Tiddler::COLORS.include?(@params['color']) ? @params['color'] : nil
    aliases = split_strings(@params['aliases'])
    tags = split_strings(@params['tags'])
    @message = nil
    
    tiddlers = Tiddler.find(:all, :order => "length(title) desc, title")
    
    if @params['title'] == @params['last_title']
      @tiddler = Tiddler.find(:first, :conditions => "title = '#{unescape_slash(@params['title'])}'")
      if @tiddler
        if @tiddler.content_digest == @params['last_content_digest']
          if !@tiddler.same_content(@params['content']) or
             @tiddler.autolink != autolink or
             @tiddler.color != color or
             @tiddler.aliases.map {|a| a.name } != aliases or
             @tiddler.tags.map {|t| t.name } != tags then
            save_and_expire(@tiddler, unescape_slash(@params['title']), @params['content'],
                            aliases, tags, autolink, color, tiddlers)
            LesserWiki::Log.edit(@tiddler)
          end
        else
          @message = i18n_message("flash_conflict")
          @tiddler.last_content = @tiddler.content
          @tiddler.content = @params['content']
          @tiddler.autolink = autolink
          @tiddler.color = color
          render_action :view
        end
      else  # @tiddler == nil
        @tiddler = Tiddler.new
        @tiddler.title = ""
        @tiddler.content = ""
        save_and_expire(@tiddler, unescape_slash(@params['title']), @params['content'],
                        aliases, tags, autolink, color, tiddlers)
        LesserWiki::Log.create(@tiddler)
      end
    else  # @params['title'] != @params['last_title']
      @tiddler = Tiddler.find(:first, :conditions => "title = '#{unescape_slash(@params['title'])}'")
      last_tiddler = Tiddler.find(:first, :conditions => "title = '#{unescape_slash(@params['last_title'])}'")
      if @tiddler
        @message = i18n_message("flash_exist", [@tiddler.title])
        #@tiddler.title = @params['last_title']
        @tiddler.last_content = @tiddler.content
        @tiddler.content = @params['content']
        @tiddler.autolink = autolink
        @tiddler.color = color
        render_action :view
      else
        if last_tiddler and last_tiddler.content_digest != @params['last_content_digest']
          @message = i18n_message("flash_conflict")
          @tiddler = last_tiddler
          @tiddler.last_content = @tiddler.content
          @tiddler.content = @params['content']
          @tiddler.autolink = autolink
          render_action :view
        else
          @tiddler = Tiddler.new
          @tiddler.title = ""
          @tiddler.content = ""
          save_and_expire(@tiddler, unescape_slash(@params['title']), @params['content'],
                          aliases, tags, autolink, color, tiddlers)
          if last_tiddler
            save_and_expire(last_tiddler, @params['last_title'],
                            "#{last_tiddler.title} &rarr; #{@tiddler.title}",
                            [], tags, last_tiddler.autolink,
                            last_tiddler.color, tiddlers)
          end
          LesserWiki::Log.rename(@tiddler)
        end
      end
    end
    
    # expire_cache flag is set by HowmChunk::Schedule or HowmChunk::Todo
    last_expire_cache = @tiddler.expire_cache
    @tiddler.expire_cache = false
    @tiddler.render!(WikiLinkManager.instance)
    if last_expire_cache != @tiddler.expire_cache
      @tiddler.update_attribute("expire_cache", @tiddler.expire_cache)
    end
    
    reminder_updated = false
    unless @tiddler.rendered_content.reminders
      @tiddler.reminders.each {|r| r.destroy }
    else
      @tiddler.reminders.each do |r|
        if @tiddler.rendered_content.reminders.select {|e| e.equals(r) }.empty?
          r.destroy
          reminder_updated = true
        end
      end
    end
    if @tiddler.rendered_content.reminders
      @tiddler.rendered_content.reminders.each do |r|
        if @tiddler.reminders.select {|e| e.equals(r) }.empty?
          r.save
          reminder_updated = true
        end
      end
    end
    
    if reminder_updated
      # expire the cache of the tiddler that contains dynamic content such as Howm schedule
      Tiddler.find(:all, :conditions => "expire_cache = 1").each do |t|
        expire_action :action => "view", :title => t.escaped_title
      end
    end
  end
  
  def search
    @message = nil
    @search = @params['search']
    tiddlers1 = Tiddler.find(:all, :conditions => "title like '%#{@search}%'",
                             :order => "title")
    tiddlers2 = (Tiddler.find(:all, :conditions => "content like '%#{@search}%'",
                              :order => "title"))
    @tiddlers = tiddlers1 | tiddlers2
    if @tiddlers.empty?
      @message = i18n_message("flash_search_nonexist", [ @search ])
    end
  end
  
  def view_tag
    @message = nil
    @tag = Tag.find(:first, :conditions => [ "name = ?", @params['name'] ])
    unless @tag
      @tag = Tag.new
      @tag.id = 0
      @tag.name = @params['name']
      @message = i18n_message("flash_tag_nonexist", [ @params['name'] ])
    end
    @use_effect = true
  end
end
