require 'date'
require 'chunks/chunk'
require 'tiddler'
require 'reminder'

module HowmChunk
  include Chunk
  
  class Reminder < Chunk::Abstract
    REMINDER_PATTERN = /^\[(\d\d\d\d-\d\d-\d\d)\]([\-\+~\!@\.])(\d+)?\s+(.*)$/
    
    def self.pattern
      REMINDER_PATTERN
    end
    
    def initialize(match_data, content)
      super    
      reminder = LesserWiki::Reminder.new
      reminder.date = Date.parse(match_data[1])
      reminder.priority_type = match_data[2]
      reminder.priority_number = (/\A\d+\Z/ =~ match_data[3]) ? match_data[3].to_i : nil
      reminder.content = match_data[4]
      reminder.tiddler = @content.tiddler
      @content.add_reminder(reminder)
      @unmask_text = %{<div class="howm#{reminder.type_name}"><span class="howmDate">#{reminder.date}</span> <span class="howm#{reminder.type_name}Mark">#{reminder.priority_type}#{reminder.priority_number}</span> #{reminder.content}</div>}
    end
  end
  
  class Summary < Chunk::Abstract
    SCHEDULE_PATTERN = /^\{\{(schedule|todo)\}\}$/
    
    def self.pattern
      SCHEDULE_PATTERN
    end
    
    def initialize(match_data, content)
      super
      @unmask_text = ""
      today = Date.today
      
      conditions = nil
      if /\Aschedule\Z/ =~ match_data[1]
        conditions = "date >= '#{today - 10}' and date <= '#{today + 3}' and ("
        LesserWiki::Reminder::SCHEDULE_TYPES.each_with_index do |t, i|
          conditions << " or " if i > 0
          conditions << "priority_type = '#{t}'"
        end
        conditions << ")"
        reminders = LesserWiki::Reminder.find(:all, :conditions => conditions, :order => "date, priority_type")
      elsif /\Atodo\Z/ =~ match_data[1]
        conditions = ""
        LesserWiki::Reminder::TODO_TYPES.each_with_index do |t, i|
          conditions << " or " if i > 0
          conditions << "priority_type = '#{t}'"
        end
        reminders = LesserWiki::Reminder.find(:all, :conditions => conditions).sort_by {|r| -r.priority }[0..49]
      end
      
      reminders.each do |r|
        next if r.priority < r.priority_bottom
        date_class = "howmDate"
        if r.date === today
          date_class = "howmDateToday"
        elsif r.date === today + 1
          date_class = "howmDateTomorrow"
        end
        link = @content.reminder_link(r.tiddler.title, ">")
        @unmask_text << %{<div class="howm#{r.type_name}">#{link} <span class="#{date_class}">#{r.date}</span> <span class="howm#{r.type_name}Mark">#{r.priority_type}#{r.priority_number}</span> #{r.content}</div><br />}
      end
      @content.tiddler.expire_cache = true
    end
  end
end