require 'wiki_words'
require 'chunks/chunk'
require 'chunks/wiki'
require 'cgi'
require 'tiddler'

# Contains all the methods for finding and replacing wiki related links.
module WikiChunk
  include Chunk

  # A wiki reference is the top-level class for anything that refers to
  # another wiki tiddler.
  class WikiReference < Chunk::Abstract

    # Name of the referenced tiddler
    attr_reader :tiddler_name
    
    # the referenced tiddler
    def reftiddler
      @content.web.tiddlers[@tiddler_name]
    end
  
  end

  # A wiki link is the top-level class for links that refers to
  # another wiki tiddler.
  class WikiLink < WikiReference
 
    attr_reader :link_text, :link_type

    def initialize(match_data, content)
      super(match_data, content)
      @link_type = :show
    end

    def self.apply_to(content, tiddlers)
      content.gsub!( self.pattern ) do |matched_text|
        chunk = self.new($~, content)
        if chunk.textile_url?
          # do not substitute
          matched_text
        else
          content.add_chunk(chunk)
          chunk.mask
        end
      end
    end

    # the referenced tiddler
    def reftiddler
      @content.web.tiddlers[@tiddler_name]
    end

    def textile_url?
      not @textile_link_suffix.nil?
    end

  end

  # This chunk matches a WikiWord. WikiWords can be escaped
  # by prepending a '\'. When this is the case, the +escaped_text+
  # method will return the WikiWord instead of the usual +nil+.
  # The +tiddler_name+ method returns the matched WikiWord.
  class Word < WikiLink

    attr_reader :escaped_text
    
    unless defined? WIKI_WORD
      WIKI_WORD = Regexp.new('(":)?(\\\\)?(' + WikiWords::WIKI_WORD_PATTERN + ')\b', 0, "utf-8")
    end

    def self.pattern
      WIKI_WORD
    end

    def initialize(match_data, content)
      super
      @textile_link_suffix, @escape, @tiddler_name = match_data[1..3]
      if @escape 
        @unmask_mode = :escape
        @escaped_text = @tiddler_name
      else
        @escaped_text = nil
      end
      @link_text = WikiWords.separate(@tiddler_name)
      @unmask_text = (@escaped_text || @content.tiddler_link(@tiddler_name, @link_text))
    end

  end

  # This chunk handles [[bracketted wiki words]] and 
  # [[AliasedWords|aliased wiki words]]. The first part of an
  # aliased wiki word must be a WikiWord. If the WikiWord
  # is aliased, the +link_text+ field will contain the
  # alias, otherwise +link_text+ will contain the entire
  # contents within the double brackets.
  #
  # NOTE: This chunk must be tested before WikiWord since
  #       a WikiWords can be a substring of a WikiLink. 
  class Link < WikiLink
    
    unless defined? WIKI_LINK
      WIKI_LINK = /(":)?\[\[([^\]]+)\]\]/
      LINK_TYPE_SEPARATION = Regexp.new('^(.+):((file)|(pic))$', 0, 'utf-8')
      ALIAS_SEPARATION = Regexp.new('^(.+)\|(.+)$', 0, 'utf-8')
    end    
        
    def self.pattern() WIKI_LINK end

    def initialize(match_data, content)
      super
      @textile_link_suffix, @tiddler_name = match_data[1..2]
      @link_text = @tiddler_name
      separate_link_type
      separate_alias
      @unmask_text = @content.tiddler_link(@tiddler_name, @link_text)
    end

    private

    # if link wihin the brackets has a form of [[filename:file]] or [[filename:pic]], 
    # this means a link to a picture or a file
    def separate_link_type
      link_type_match = LINK_TYPE_SEPARATION.match(@tiddler_name)
      if link_type_match
        @link_text = @tiddler_name = link_type_match[1]
        @link_type = link_type_match[2..3].compact[0].to_sym
      end
    end

    # link text may be different from tiddler name. this will look like [[actual tiddler|link text]]
    def separate_alias
      alias_match = ALIAS_SEPARATION.match(@tiddler_name)
      if alias_match
        @tiddler_name, @link_text = alias_match[1..2]
      end
      # note that [[filename|link text:file]] is also supported
    end
  end
  
  class AutoLink < WikiLink
    class LinkTarget
      attr_reader :name, :tiddler, :priority

      def initialize(name, tiddler, priority)
        @name = name
        @tiddler = tiddler
        @priority = priority
      end
      
      def <=>(other)
        if (len = other.name.length <=> @name.length) == 0
          if (cmp = other.name <=> @name) == 0
            other.priority <=> @priority
          else
            cmp
          end
        else
          len
        end
      end
    end
    
    def self.apply_to(content, tiddlers)
      targets = []
      tiddlers.select {|t| t.autolink == 1 }.each do |t|
        targets << LinkTarget.new(t.title, t, 1)
      end
      Alias.find(:all, :order => "length(name) desc, name").each do |a|
        targets << LinkTarget.new(a.name, a.tiddler, 0)
      end
      targets.sort.each do |t|
        content.gsub!( /(#{Regexp.escape(t.name)})(?!chunk)/ ) do |match|
          new_chunk = self.new($~, content, t)
          content.add_chunk(new_chunk)
          new_chunk.mask
        end
      end
    end
    
    def initialize(match_data, content, target)
      super(match_data, content)
      @tiddler_name = target.tiddler.title
      @link_text = match_data[0]
      @unmask_text = @content.tiddler_link(@tiddler_name, @link_text)
    end
  end
  
  class AliasLink < WikiLink
    def self.apply_to(content, tiddlers)
      tiddlers.each do |t|
        t.aliases.each do |a|
          content.gsub!( /(#{Regexp.escape(a.name)})(?!chunk)/ ) do |match|
            new_chunk = self.new($~, content, a)
            content.add_chunk(new_chunk)
            new_chunk.mask
          end
        end
      end
    end

    def initialize(match_data, content, alias_)
      super(match_data, content)
      @tiddler_name = alias_.tiddler.title
      @link_text = match_data[0]
      @unmask_text = @content.tiddler_link(@tiddler_name, @link_text)
    end
  end
end
