require_dependency 'tag'
require_dependency 'search_result'
require_dependency 'wiki_renderer'

class WikiController < ApplicationController
  model :tiddler
  layout "wiki", :only => [ "index", "tag", "search" ]
  
  before_filter :verify_settings
  before_filter :verify_permission,
                :except => [ "index", "view", "view_tag", "view_search",
                             "login", "logout", "permission_denied",
                             "main_menu", "sidebar" ]
  
  PER_PAGE = 30
  
  def index
    @permission = check_permission
    
    if @permission
      if params[:title]
        @title = params[:title]
        unless read_fragment(:action => "index", :title => params[:title])
          @start_here = Tiddler.find_by_title(params[:title]) || Tiddler.new_not_found
          # XXX aliases also have to be searched
        end
      else
        @title = Tiddler.start_here_title
        unless read_fragment(:action => "index", :title => Tiddler.start_here_title)
          @start_here = Tiddler.start_here
        end
        
        # create MainMenu if is is not exist
        Tiddler.main_menu
      end
    else
      unless read_fragment(:action => "index", :title => Tiddler.not_found_title)
        @start_here = Tiddler.new_login_required
      end
    end
  end
  
  def tag
    @permission = check_permission
    
    if @permission
      @tiddlers = Tiddler.find(:all)
      if params[:name]
        @tag = Tag.find_by_name(params[:name])
        unless @tag
          redirect_to :action => "index", :title => Tiddler.not_found_title
          return
        end
      else
        redirect_to :action => "index"
        return
      end
    else
      redirect_to :action => "index"
      return
    end
    
    render :action => "index"
  end
  
  def search
    @permission = check_permission
    
    if @permission
      @tiddlers = Tiddler.find(:all)
      if params[:search]
        tiddlers = Tiddler.find(:all,
                                :conditions => [ "title like ?", "%#{params[:search]}%" ],
                                :order => "title") |
                   Tiddler.find(:all,
                                :conditions => [ "content like ?", "%#{params[:search]}%" ],
                                :order => "title")
        @search = SearchResult.new(params[:id], params[:search], tiddlers)
        unless @search
          redirect_to :action => "index", :title => Tiddler.not_found_title
          return
        end
      else
        redirect_to :action => "index"
        return
      end
    else
      redirect_to :action => "index"
      return
    end
    
    render :action => "index"
  end
  
  def view
    if @permission = check_permission
      @tiddler = Tiddler.find(params[:id]) rescue nil
      
      if @tiddler
        render :update do |page|
          page.replace_html @tiddler.element_id, :partial => "tiddler_body"
        end
      else
        @tiddler = Tiddler.new_not_found
        @tiddler.id = params[:id]
        
        render :update do |page|
          page.replace Tiddler.element_prefix + params[:id], :partial => "tiddler"
        end
      end
    else  # !check_permission
      @tiddler = Tiddler.new_login_required
      @tiddler.id = params[:id]
      
      render :update do |page|
        page.replace Tiddler.element_prefix + params[:id], :partial => "tiddler"
      end
    end
  end
  
  def view_tag
    if @permission = check_permission
      @tag = Tag.find(params[:id]) rescue nil
      
      if @tag
        render :update do |page|
          page.replace_html @tag.element_id, :partial => "tag_body"
        end
      else
        @tiddler = Tiddler.new_not_found
        @tiddler.id = params[:id]
        
        render :update do |page|
          page.replace Tiddler.element_prefix + params[:id], :partial => "tiddler"
        end
      end
    else  # !check_permission
      @tiddler = Tiddler.new_login_required
      @tiddler.id = params[:id]
      
      render :update do |page|
        page.replace Tiddler.element_prefix + params[:id], :partial => "tiddler"
      end
    end
  end
  
  def view_search
    if @permission = check_permission
      tiddlers = Tiddler.find(:all,
                              :conditions => [ "title like ?", "%#{params[:search]}%" ],
                              :order => "title") |
                 Tiddler.find(:all,
                              :conditions => [ "content like ?", "%#{params[:search]}%" ],
                              :order => "title")
      @search = SearchResult.new(params[:id], params[:search], tiddlers)
      
      render :update do |page|
        page.call "command_panel_handler.searchDone"
        page.replace_html @search.element_id, :partial => "search_body"
        page.visual_effect :safe_highlight, @search.element_id
      end
    else
      # !check_permission
      @tiddler = Tiddler.new_login_required
      @tiddler.id = params[:id]
      
      render :update do |page|
        page.replace Tiddler.element_prefix + params[:id], :partial => "tiddler"
      end
    end      
  end
  
  def update
    return unless @request.post?
    
    @tiddler = nil
    if params[:id]
      # update the existing tiddler
      @tiddler = Tiddler.find(params[:id])
      
      # conflict check
      if params[:digest] != @tiddler.digest
        # expire caches
        expire_caches_of_tiddler(@tiddler)
        
        # store params and redirect to conflict
        flash[:params] = params
        redirect_to :action => "conflict"
        return
      end
      
      @tiddler.attributes = params[:tiddler]
    else
      # create a new tiddler
      @tiddler = Tiddler.new(params[:tiddler])
      expire_caches_of_related_tiddlers(@tiddler)
    end
    
    # expire caches
    expire_fragment(:action => "sidebar")
    expire_caches_of_tiddler(@tiddler)
    
    # save the tiddler with aliases and tags
    save_aliases(@tiddler, params[:aliases].split_by_space_and_brackets)
    save_tags(@tiddler, params[:tags].split_by_space_and_brackets)
    
    @tiddler.written_by = @session[:login] if @session[:login]
    
    @tiddler.save
    
    # if saving tiddler was failed, display messages and return immediately
    unless @tiddler.errors.empty?
      display_message(@tiddler.errors.full_messages)
      return
    end
    
    # store rendered strings of main manu and sidebar to local variables
    # because render_* method can not be called in render method
    main_menu = render_main_menu(@tiddler)
    sidebar = render_sidebar
    
    @tiddler.reload
    
    # tiddler.id == nil => NotFound
    # tiddler.id == 0   => NewTiddler
    # tiddler.id > 0    => normal tiddlers
    
    render :update do |page|
      if params[:id]
        # delete the old observer
        page.call "if ($('#{@tiddler.element_id}')) $('#{@tiddler.element_id}').finalize"
        
        # replace current tiddler's body
        page.replace_html @tiddler.element_id, :partial => "tiddler_body"
        page.visual_effect :safe_highlight, @tiddler.element_id
      else
        # NewTiddler
        tiddler = Tiddler.new
        tiddler.id = 0
        
        # hide NewTiddler
        page.hide tiddler.element_id
        
        # insert the new tiddler
        page.insert_html :after, tiddler.element_id, :partial => "tiddler"
        page.visual_effect :safe_highlight, @tiddler.element_id
      end
      
      if main_menu
        page.replace "main_menu", main_menu
        page.visual_effect :safe_highlight, "main_menu"
      end
      if sidebar
        page.replace "sidebar", sidebar
        page.visual_effect :highlight, "sidebar"
      end
    end
  end
  
  def delete
    return unless @request.post?
    
    @tiddler = Tiddler.find(params[:id]) rescue nil
    if @tiddler    
      if Tiddler.special_titles.include?(@tiddler.title)
        display_message(_("can_not_delete", @tiddler.title))
        return
      end
      
      # expire caches
      expire_fragment(:action => "sidebar")
      expire_caches_of_tiddler(@tiddler)
      save_aliases(@tiddler, [])
      save_tags(@tiddler, [])
            
      @tiddler.tags.destroy_all
      @tiddler.destroy
    end
    
    # store rendered strings of main manu and sidebar to local variables
    # because render_* method can not be called in render method
    main_menu = render_main_menu(@tiddler)
    sidebar = render_sidebar
    
    # reload MainMenu
    render :update do |page|
      if main_menu
        page.replace "main_menu", main_menu
        page.visual_effect :safe_highlight, "main_menu"
      end
      if sidebar
        page.replace "sidebar", sidebar
        page.visual_effect :highlight, "sidebar"
      end
    end
  end
  
  def conflict
    params = flash[:params]
    @tiddler = Tiddler.find(params[:id])
    @conflict = diff_between_contents(@tiddler.content, params[:tiddler][:content])
    
    render :update do |page|
      # delete the old observer
      page.call "if ($('#{@tiddler.element_id}')) $('#{@tiddler.element_id}').finalize"
      
      # replace current tiddler's body
      page.replace_html @tiddler.element_id, :partial => "tiddler_body"
      page.call "#{@tiddler.listener_id}.edit"
      page.visual_effect :safe_highlight, @tiddler.element_id
    end
  end
  
  def timeline_page
    @timeline_pages, @timeline_tiddlers =
      paginate(:tiddler, :per_page => PER_PAGE,
               :order => "updated_at DESC")
    
    render :update do |page|
      page.replace_html "sidebar_timeline", :partial => "sidebar_timeline"
      page.visual_effect :highlight, "sidebar_timeline"
    end
  end
  
  def color_page
    @color_pages, @color_tiddlers =
      paginate(:tiddler, :per_page => PER_PAGE,
               :conditions => [ "color = ?", params[:color] ],
               :order => "title")
    
    render :update do |page|
      page.replace_html "color_contents", :partial => "color_page"
      page.visual_effect :highlight, "color_contents"
    end
  end
  
  def login
    return unless @request.post?
    
    if authenticate?(params[:login][:owner_login], params[:login][:owner_password])
      @session[:login] = params[:login][:owner_login]
      @tiddlers = Tiddler.find(:all)
      @permission = true
      
      # create system tiddlers if they are not exist
      Tiddler.start_here
      Tiddler.main_menu
      
      # store rendered strings of main manu and sidebar to local variables
      # because render_* method can not be called in render method
      main_menu = render_main_menu
      sidebar = render_sidebar
      
      render :update do |page|
        if @@permissions[config_values[:permission].to_i] == :login_required_all
          page.replace "main_menu", main_menu
          page.visual_effect :safe_highlight, "main_menu"
        end
        page.replace "sidebar", sidebar
        page.visual_effect :highlight, "sidebar"
      end
    else
      display_message(_("login_failed"))
    end
  end
  
  def logout
    @session[:login] = nil
    @permission = false
    
    # store rendered strings of main manu and sidebar to local variables
    # because render_* method can not be called in render method
    main_menu = render_main_menu
    sidebar = render_sidebar
        
    render :update do |page|
      if @@permissions[config_values[:permission].to_i] == :login_required_all
        page.replace "main_menu", main_menu
        page.visual_effect :safe_highlight, "main_menu"
        
        page.replace "sidebar", sidebar
        page.visual_effect :highlight, "sidebar"
      else
        page.replace_html "login_panel", :partial => "login_panel"
        page.visual_effect :highlight, "login_panel"
      end
    end
  end
  
  def permission_denied
    display_message(_("login_required"))
  end
  
  # Render MainMenu
  # This action is used as a component.
  def main_menu
    @permission = check_permission
  end
  
  # Render the sidebar.
  # This action is used as a component.
  def sidebar
    if @permission = check_permission
      unless read_fragment(:action => "sidebar")
        @timeline_pages, @timeline_tiddlers =
          paginate(:tiddler, :per_page => PER_PAGE, :order => "updated_at DESC")
        @tags = collect_tags
      end
    end
  end
  
  protected
  
  # If 'title' parameter is specified, check whether MainMenu's content
  # contains the title or not.
  # If the check is true, return rendered string, otherwise return nil.
  #
  # This method calls 'main_menu' action as a component.
  def render_main_menu(tiddler = nil)
    main_menu = Tiddler.main_menu
    
    if !tiddler or
       tiddler.title == main_menu.title or
       /#{Regexp.escape(tiddler.title)}/ =~ main_menu.content
      expire_fragment(:action => "main_menu")
      expire_caches_of_tiddler(main_menu)
      
      render_component_as_string(:action => "main_menu")
    else
      nil
    end
  end
  
  # This method calls 'sidebar' action as a component.
  def render_sidebar
    render_component_as_string(:action => "sidebar")
  end
  
  def collect_tags
    cloud = TagCloud.new
    Tag.find(:all).each do |t|
      cloud.add(t, t.taggings.size)
    end
    cloud.calculate
  end
  
  def save_aliases(tiddler, alias_strs)
    tiddler.aliases.select {|a| !alias_strs.include?(a.name) }.each do |a|
      Tiddler.find(:all, :conditions => [ "content like ?", "%#{a.name}%" ]).each do |t|
        logger.info "expire old alias #{a.name}"
        expire_fragment(:action => "index", :title => t.title)
        expire_fragment(:action => "view", :id => t.id)
      end
      
      tiddler.aliases.delete(a)
      a.destroy
    end
    
    # save new aliases
    alias_strs.select {|a| !Alias.find_by_name(a) }.each do |a|
      next if a.length < 3
      tiddler.aliases << Alias.new(:name => a, :tiddler => tiddler)
      
      Tiddler.find(:all, :conditions => [ "content like ?", "%#{a}%" ]).each do |t|
        logger.info "expire new alias #{a}"
        expire_fragment(:action => "index", :title => t.title)
        expire_fragment(:action => "view", :id => t.id)
      end
    end
  end
  
  def save_tags(tiddler, tag_strs)    
    # expire the cache of old tags
    tiddler.tags.select {|tag| !tag_strs.include?(tag.name) }.each do |tag|
      tag.tagged.each do |t|
        logger.info "expire old tag #{t.title}"
        expire_fragment(:action => "index", :title => t.title)
        expire_fragment(:action => "view", :id => t.id)
      end
    end
    
    tag_strs.select {|tag_str| Tag.find_by_name(tag_str) }.each do |tag_str|
      # expire the cache of new tags
      if tag = Tag.find_by_name(tag_str)
        tag.tagged.each do |t|
          logger.info "expire new tag #{t.title}"
          expire_fragment(:action => "index", :title => t.title)
          expire_fragment(:action => "view", :id => t.id)
        end
      end
    end
    
    # update tags
    tiddler.tag_with(tag_strs.join_with_quotes)
  end
  
  def expire_caches_of_tiddler(tiddler)
    logger.info "expire self #{tiddler.title}"
    expire_fragment(:action => "index", :title => tiddler.title)
    expire_fragment(:action => "view", :id => tiddler.id)
  end
  
  # Expire caches that contains this tiddler's title
  def expire_caches_of_related_tiddlers(tiddler)
    Tiddler.find(:all, :conditions => [ "content like ?", "%#{tiddler.title}%" ]).each do |t|
      logger.info "expire related #{tiddler.title}"
      expire_fragment(:action => "index", :title => t.title)
      expire_fragment(:action => "view", :id => t.id)
    end
  end

  def display_message(args)
    if args.class == Array and args.size > 1
      @message = ""
      @message << "<ul>"
      args.each do |arg|
        @message << "<li>#{arg}</li>"
      end
      @message << "</ul>"
    else
      @message = args.to_s
    end
    render :update do |page|
      page.replace_html "message_area", :partial => "message"
      page["message_area"].show
      page.visual_effect :highlight, "message_area"
      if flash[:element_id]
        page.call "enableForm", flash[:element_id]
      end
    end
  end
  
  def verify_permission
    @permission = false
    
    unless check_permission
      if @action_name == "update"
        flash[:element_id] = params[:id] ? "editor_#{params[:id]}" : "editor_0"
      end
      
      redirect_to :action => "permission_denied"
      return
    end    
    
    @permission = true
  end
  
  def check_permission
    return false unless config_values[:permission]
    
    case @@permissions[config_values[:permission].to_i]
      when :login_required_none
        return true
      when :login_required_edit
        if @action_name == "update" or @action_name == "delete"
          return @session[:login] != nil
        else
          return true
        end
      when :login_required_all
        return @session[:login] != nil
      else
        return false
    end
  end
  
  def verify_settings
    unless config_values.is_ok?
      redirect_to :controller => "admin", :action => "index"
    end
  end
end
