module ActiveRecord
  class Errors
    @@default_error_messages.merge!({
      :not_a_integer => "is not a integer number",
      :not_a_unsigned => "is not a unsigned number",
      :associated => "is a invalid associated object"
    })
    
    def full_messages
      full_messages = []
      @errors.each_key do |attribute|
        @errors[attribute].each do |message|
          full_messages << (attribute == 'base' ? message : attribute.to_s.humanize + " #{message}") unless message.nil?
        end
      end
      full_messages
    end
  end
  
  module Validations
  
    # Allows to validate user input data independently of a model.
    # 
    # Can be used by two ways; Creating a +ActiveRecord::Validations::Validator+ class
    # descendant where we declare sets of validation rules that can be used later.
    #
    #   # Validator
    #   class UserValidator < ActiveRecord::Validations::Validator
    #     validates :card_type, :include => Customer::CARD_TYPES
    #     validates :card_number, :length => 13..16, :number => :unsigned_integer
    #   end
    #
    #   # Controller
    #   class BlogController < ApplicationController
    #     def validate
    #        validator = UserValidator.new(params[:user])
    #        validator.validate(:card_type, :card_number)
    #        puts validator.errors.full_messages unless validator.errors.empty?
    #      end
    #    end
    #
    # Or using directly the +ActiveRecord::Validations::Validator+ class
    #   
    #   # Controller
    #   class BlogController < ApplicationController
    #     def validate
    #        validator = UserValidator.new(params[:user])
    #        validator.validate_with(:card_type, :include => Customer::CARD_TYPES)
    #        validator.validate_with(:card_number, :length => 13..16, :number => :unsigned_integer)
    #        puts validator.errors.full_messages unless validator.errors.empty?
    #      end
    #    end
    #
    # The validation rules in a set are chained, so that the validation in a set stops in the first
    # rule that is not fulfilled. In this case, a new error message is added to validator +Errors+ object.
    # 
    # The validation rules can be:
    # * +strip+ - Leading and trailing whitespace is removed in each value by default. Set to +false+ to disable striping. 
    # * +filled - Requires that the value be filled. Set to +:optional+ to bypass the validation silently if value is empty.
    # * +number+ - Validates as a number (other options :integer|:unsigned|:unsigned_integer)
    # * +length+ - Validates length. (examples: 34|2..12|{:min => 2, :max => 12})
    # * +format+ - Regular expression to match
    # * +include+ - Enumeration of possible rigth values
    # * +exclude+ - Enumeration of wrong values
    # * +confirm+ - true|:pass_confirm
    # * +accept+ - true|'true' 
    # * +unique+ - 'Customer.user'|['Customer.user', 1]
    # * +associated+ - 'Customer.user'
    class Validator
      @@validations = {}
      
      # Defines a set of validation rules that can be used later
      # Works in a +Validator+ descendant, at class declaration scope.
      def Validator.validates(attribute, options, &block)
        @@validations[attribute] = [options, block]
      end
      
      # Defines a block to be called to validate two a more attributes relateds
      # Works in a +Validator+ descendant, at class declaration scope.
      def Validator.validates_related(*attributes, &block)
        @@validations[attributes] = block
      end
      
      attr_reader :errors, :params      
      
      # Creates a new +Validator+ for the +target+
      # +target+ can be a +Hash+ or a +ActiveRecord+
      def initialize(target)
        case target
        when Hash
          @target = target
          @errors = ActiveRecord::Errors.new(nil)
        when ActiveRecord::Base
          @target = target
          @target.errors.clear
          @errors = @target.errors
        else
          raise ArgumentError, 'target must be a Hash or ActiveRecord object'
        end
        @params = {}
      end
      
      # Validates +attributes+ applying the validation set rules previously defined with +Validator.validates+
      # in a +Validator+ descendant class.
      #
      # Examples:
      #
      #   # validates value referenced by the key :email in the +target+ with the validation rules set named +:email+
      #   validator.validate(:email)
      #   # validates several values at time
      #   validator.validate(:email, :url, :nick)
      #   # validates value referenced by the key :customer_email in the +target+ with the validation rules set named +:email+
      #   # and the same with +:url+
      #   validator.validate({:customer_email => :email})
      #   # validates the value 'josh@vectrice.com' with the validation rules set named +:email+
      #   validator.validate('josh@vectrice.com' => :email)
      #   # Combining
      #   validator.validate(:nick, {:customer_email => :email}, :url, :name)
      def validate(*attributes)
        result = true
        attributes.each do |attribute|
          if attribute.kind_of?(Hash)
            attribute.each do |key, value|
              result &&= case key
              when Symbol
                result = false unless perform_validation(key, @target[key], @@validations[value].first, &@@validations[value].last)
              else
                result = false unless perform_validation(value, key, @@validations[value].first, &@@validations[value].last)
              end
            end
          else
            result = false unless perform_validation(attribute, @target[attribute], @@validations[attribute].first, &@@validations[attribute].last)
          end
        end
        result
      end
      
      # Validates two o more attributes relateds calling a previosusly defined block with +Validator.validates_related+
      # in a +Validator+ descendant class.
      #
      # Example:
      #   validator.validate_related(:card_numer, :card_type)
      def validate_related(*attributes)
        hash = {}
        attributes.each do |attribute|
          return false if @errors.on(attribute)
          hash[attribute] = @target[attribute]
        end
        @@validations[attributes].call(self, hash)
      end
      
      # Validates +subject+ applying a set of validation rules and a optional block.
      # Example:
      #
      #   # validates value referenced by the key :customer_email in the +target+ with the passed validation rules
      #   validate_with(:customer_email, format => //)
      #   # validates the value 'josh@vectrice.com' as :customer_email with the passed validation rules
      #   validate_with({:customer_email => 'josh@vectrice.com'}, :presence => true, format => //)
      #   # validates two values as two attributes with the passed validation rules
      #   validate_with({:customer_email => 'josh@vectrice.com', :name => 'Josh'}, :presence => true, format => //)
      def validate_with(subject, options, &block)
        result = true
        case subject
        when Symbol
          result = false unless perform_validation(subject, @target[subject], options, &block)
        when Hash
          subject.each do |attribute, value|
            result = false unless perform_validation(attribute, value, options, &block)
          end
        else
          raise ArgumentError
        end
        result
      end
      
      def validate_related_with
      end
      
      private
        def perform_validation(attribute, value, options = {})
          value = value.kind_of?(String) ? value.dup : value.to_s
          value = value.strip unless options[:strip] == false
          @params[attribute] = value
          if options[:filled] == :optional && value.empty?
            return true
          elsif value.empty?
            @errors.add(attribute, ActiveRecord::Errors.default_error_messages[:blank])
          elsif [:integer, :unsigned_integer].include?(options[:number]) && /^-?\d+$/ !~ value
            @errors.add(attribute, ActiveRecord::Errors.default_error_messages[:not_a_integer])
          elsif [:unsigned, :unsigned_integer].include?(options[:number]) && /^\d+(?:\.\d+)?$/ !~ value
            @errors.add(attribute, ActiveRecord::Errors.default_error_messages[:not_a_unsigned])
          elsif options[:number] && /^-?\d+(?:\.\d+)?$/ !~ value
            @errors.add(attribute, ActiveRecord::Errors.default_error_messages[:not_a_number])
          elsif options[:length].kind_of?(Fixnum) && value.length != options[:length]
            @errors.add(attribute, ActiveRecord::Errors.default_error_messages[:wrong_length] % options[:length])
          elsif (options[:length].kind_of?(Range) && value.length < options[:length].begin) || (options[:length].kind_of?(Hash) && options[:length][:min] && value.length < options[:length][:min])
            @errors.add(attribute, ActiveRecord::Errors.default_error_messages[:too_short] % (options[:length].kind_of?(Range) ? options[:length].begin : options[:length][:min]))
          elsif (options[:length].kind_of?(Range) && value.length > options[:length].end) || (options[:length].kind_of?(Hash) && options[:length][:max] && value.length > options[:length][:max])
            @errors.add(attribute, ActiveRecord::Errors.default_error_messages[:too_long] % (options[:length].kind_of?(Range) ? options[:length].end : options[:length][:max]))
          elsif options[:format] && options[:format] !~ value
            @errors.add(attribute, ActiveRecord::Errors.default_error_messages[:invalid])
          elsif options[:include] && !options[:include].collect { |v| v.to_s}.include?(value)
            @errors.add(attribute, ActiveRecord::Errors.default_error_messages[:inclusion])
          elsif options[:exclude] && options[:exclude].collect { |v| v.to_s}.include?(value)
            @errors.add(attribute, ActiveRecord::Errors.default_error_messages[:exclusion])
          elsif options[:confirm] && value != @target[options[:confirm] == true ? (attribute.kind_of?(Symbol) ? "#{attribute.to_s}_confirmation".to_sym : "#{attribute.to_s}_confirmation") : options[:confirm]]
            @errors.add(attribute, ActiveRecord::Errors.default_error_messages[:confirmation])
          elsif options[:accept] && value != (options[:accept] == true ? '1' : options[:accept])
            @errors.add(attribute, ActiveRecord::Errors.default_error_messages[:accepted])
          elsif options[:unique] && associated_exists?(value, options[:unique])
            @errors.add(attribute, ActiveRecord::Errors.default_error_messages[:taken])
          elsif options[:associated] && !associated_exists?(value, options[:associated])
            @errors.add(attribute, ActiveRecord::Errors.default_error_messages[:associated])
          else
            result = true
            result &&= yield(self, attribute, value) if block_given?
            return result
          end
          false
        end
        
        def associated_exists?(value, option)
          option, exclude_id = option if option.kind_of?(Array)
          model, attribute = option.to_s.split('.', 2)
          model = self.class.const_get(model)
          if exclude_id || attribute
            conditions = [ActiveRecord::Base.connection.quote_column_name(model.columns_hash[attribute].name) + '=?']
            values = [value]
            if exclude_id
              conditions.unshift(ActiveRecord::Base.connection.quote_column_name(model.columns_hash[model.primary_key].name) + '<>?')
              values.unshift(exclude_id)
            end
            !model.count([conditions.join(' AND '), *values]).zero?
          else
            model.exists?(value)
          end
        end
    end
  end
end