require 'erb'
require_dependency 'wiki_content'
require_dependency 'chunks/engines'

class AbstructRenderer
  include Singleton
  include ERB::Util  # html_escape
  
  def render(tiddler, engine, tiddlers, aliases)
    @tiddler = tiddler
    WikiContent.new(self, tiddlers, aliases,
                    { :mode => :show, :engine => engine }).render!
  end
  
  def tiddler_content
    sanitize(@tiddler.content)
  end
  
  def url_link(url, text, options = {})
    "[url: #{url} text: #{text}]"
  end
  
  def tiddler_link(tiddler, text, options = {})
    tiddler_name = tiddler.is_a?(Tiddler) ? tiddler.title : tiddler.to_s
    "[tiddler: #{tiddler_name} text: #{text}]"
  end
  
  def reminder_link(tiddler, text, options = {})
    "[reminder: #{tiddler.title} text: #{text}]"
  end
  
  protected

  # ported from rails/actionpack/lib/action_view/helpers/text_helper.rb
  # <iframe> was added to VERBOTEN_TAGS by LeserWiki
  VERBOTEN_TAGS = %w(form script iframe) unless defined?(VERBOTEN_TAGS)
  VERBOTEN_ATTRS = /^on/i unless defined?(VERBOTEN_ATTRS)
  def sanitize(html)
    # only do this if absolutely necessary
    if html.index("<")
      tokenizer = HTML::Tokenizer.new(html)
      new_text = ""

      while token = tokenizer.next
        node = HTML::Node.parse(nil, 0, 0, token, false)
        new_text << case node
          when HTML::Tag
            if VERBOTEN_TAGS.include?(node.name)
              node.to_s.gsub(/</, "&lt;")
            else
              if node.closing != :close
                node.attributes.delete_if { |attr,v| attr =~ VERBOTEN_ATTRS }
                if node.attributes["href"] =~ /^javascript:/i
                  node.attributes.delete "href"
                end
              end
              node.to_s
            end
          else
            node.to_s.gsub(/</, "&lt;")
        end
      end

      html = new_text
    end

    html
  end
end

class WikiRenderer < AbstructRenderer
  include Singleton
  
  def url_link(url, text, options = {})
    %[<a href="#{url}" class="external_link" target="_blank"
         title="#{_("external_link", "#{url}")}">#{html_escape(text)}</a>
    ].gsub(/\n\s*/, ' ').strip
  end
  
  def tiddler_link(tiddler, text, options = {})
    tiddler, tiddler_name = parse_tiddler(tiddler)
    if tiddler      
      %[<a href="javascript:;" class="tiddly_link_existing color_#{tiddler.color_name}"
           title="#{tiddler.adjusted_updated_at}"
           onclick="openTiddler('#{html_escape(build_json_data(@tiddler, tiddler))}')">#{html_escape(text)}</a>
      ].gsub(/\n\s*/, ' ').strip
    else
      %[<a href="javascript:;" class="tiddly_link_non_existing"
           title="#{_("non_existing", text)}"
           onclick="nonexistingTiddler('#{html_escape(text)}')">#{html_escape(text)}</a>
      ].gsub(/\n\s*/, ' ').strip
    end
  end
  
  #def reminder_link(tiddler_name, text, options = {})
  #  ""
  #end
  
  protected
  
  def parse_tiddler(arg)
    tiddler = nil
    tiddler_name = nil
    
    if arg.is_a?(Tiddler)
      tiddler = arg
      tiddler_name = tiddler.title
    else
      tiddler_name = arg
      tiddler = Tiddler.find_by_title(tiddler_name)
    end
    
    [ tiddler, tiddler_name ]
  end
  
  def build_json_data(current_tiddler, new_tiddler)
    json_data = {
      :current_tiddler => current_tiddler ? current_tiddler.element_id : "tiddler_0",
      :new_tiddler => new_tiddler.element_id,
      :title => new_tiddler.title,
      :color => new_tiddler.color_name,
      :url => "/wiki/view/#{new_tiddler.id}"
    }.to_json
  end
end 

class MenuRenderer < WikiRenderer
  include Singleton
  
  def tiddler_link(tiddler, text, options = {})
    super(tiddler, text, options) + "<br />"
  end
  
  def build_json_data(current_tiddler, new_tiddler)
    super(nil, new_tiddler)
  end
end

class RSSRenderer < WikiRenderer
  include Singleton
  
  def url_link(url, text, options = {})
    %[<a href="#{url}">#{html_escape(text)}</a>]
  end
  
  def tiddler_link(tiddler, text, options = {})
    tiddler, tiddler_name = parse_tiddler(tiddler)
    if tiddler      
      %[<a href="/tiddler/#{html_escape(tiddler_name)}">#{html_escape(text)}</a>]
    else
      html_escape(text)
    end
  end
end
