/**
 * module [log4js]
 *
 * <quick start>
 * 1.include js.
 *   <script src='/js/log4.js' type='text/javascript'></script>
 * 2.configure
 *   LogConfig.add({
 *	     appender : Appender.consoleFloat,
 *	     category : 'test',
 *	     level    : LOG_LEVEL.INFO
 *       });
 * 3.create logger
 *   var logger = LoggerFactory.create("test.category");
 * 4.print
 *   logger.error("error message");
 *   //...methods fatal,error,warn,info,debug
 */
 
/** ログレベル定数 */
var LOG_LEVEL = {
	FATAL   : 1,
	ERROR   : 2,
	WARN    : 3,
	INFO    : 4,
	DEBUG   : 5
};

/** ログ設定コレクションオブジェクト */
var LogConfig = {
	size : function() {
		return this.list.length;
	},
	list  : new Array(),
	get : function(index) {
		return this.list[index];
	},
	add : function(config) {
		if (!config.appender) {
			config.appender = {
					obj      : Appender.consoleFloat,
					holdSize : 99,
					order    : 'desc'};
		}
		this.list[this.size()] = config;
	}
};

/** ロガーファクトリ */
var LoggerFactory = {
	create: function(category) {
		return new Logger(category);
	}
};

/** ロガークラス */
function Logger(category) {
	this.category = category;
};
Logger.prototype = {
	fatal  : function(msg) {this.log(LOG_LEVEL.FATAL, msg);},
	error  : function(msg) {this.log(LOG_LEVEL.ERROR, msg);},
	warn   : function(msg) {this.log(LOG_LEVEL.WARN , msg);},
	info   : function(msg) {this.log(LOG_LEVEL.INFO , msg);},
	debug  : function(msg) {this.log(LOG_LEVEL.DEBUG, msg);},
	log    : function(level, msg) {
		for (var i = 0; i < LogConfig.size(); i++) {
			var config = LogConfig.get(i);

			if (config.category) {
				var chk = this.category.substring(0, config.category.length);
				if (chk != config.category) continue;				
//				if (this.category.indexOf(config.category) != 0) continue;
			}
			if (config.level < level) continue;
			if (!config.appender) continue;
			config.appender.obj.write(config.appender, level, this.now(), msg);
		}
	},
	now : function() {
		var d = new Date();
		return	this.format(d.getHours(), "00") + ":" +
				this.format(d.getMinutes(), "00") + ":" +
				this.format(d.getSeconds(), "00");
	},
	format : function(val, formatString) {
		var tmp = formatString + val;
		return tmp.substring(tmp.length - formatString.length);
	}
};

var Layout = {
	dump : function(obj) {
		if (obj) {
			var msg = "";
			for (var o in obj) {
				msg += o + "={" + obj[o] + "} ";
			}
			return msg;
		} else {
			return 'undefined';
		}
	}
};



/** 同一ドキュメント内にDIVで表示するコンソールアペンダ */
var InsideConsoleAppender = {
	tableBody  : null,
	doc        : null,
	createTableBody : function(doc) {
		var win = doc.createElement('DIV');
		win.setAttribute('id', 'log4jsDialog');

		var header = doc.createElement('DIV');
		header.setAttribute('id', 'log4jsDialog_header');
		var c1 = doc.createElement('CENTER');
		var title = doc.createTextNode("log4js console");
		c1.appendChild(title);
		header.appendChild(c1);
		
		var body = doc.createElement('DIV');
		body.setAttribute('id', 'log4jsDialog_body');
		var table = doc.createElement('TABLE');
		var tb = doc.createElement('TBODY');
		table.appendChild(tb);	
		body.appendChild(table);

		var resizer = doc.createElement('DIV');
		resizer.setAttribute('id', 'log4jsDialog_rezier');
		
		win.appendChild(header);
		win.appendChild(body);
		win.appendChild(resizer);

		Element.setStyle(win,		this.style.win);
		Element.setStyle(header,	this.style.header);
		Element.setStyle(body,		this.style.body);
		Element.setStyle(resizer,	this.style.resizer);

		Element.setStyle(header,	this.style.font);
		Element.setStyle(body,		this.style.font);
		Element.setStyle(resizer,	this.style.font);

		win.style['width']			= this.style.windowWidth + "px";
		win.style['height']			= this.style.windowHeight + "px";
		header.style['height']		= this.style.headerHeight + "px";
		body.style['height']		= this.style.windowHeight - this.style.headerHeight - this.style.footerHeight + "px";
		resizer.style['height']		= this.style.footerHeight + "px";

		var b = doc.getElementsByTagName('BODY')[0];
		if (b) {
			b.appendChild(win);
			this.setEventHandler();
		}
		return tb;
	},
	write : function(config, level, ts, msg) {
		if (!this.tableBody) {
			this.tableBody = this.createTableBody(document);
		}

		if (typeof(msg) == "object") {
			var s = "";
			for (property in msg) {
				if (typeof(msg[property]) != "function") {
					s += property + "[" + msg[property] + "] ";
				}
			}
			msg = s;
		}
		
		var tr = document.createElement('TR');
		var h1 = document.createElement('TD');
		h1.setAttribute("wrap", "false");
		var t1 = document.createTextNode("[" + level + "]");
		h1.appendChild(t1);
		tr.appendChild(h1);
		var h2 = document.createElement('TD');
		h2.setAttribute("wrap", "false");
		var t2 = document.createTextNode(ts);
		h2.appendChild(t2);
		tr.appendChild(h2);
		var h3 = document.createElement('TD');
		h3.setAttribute("wrap", "false");
		var t3 = document.createTextNode(msg);
		h3.appendChild(t3);
		tr.appendChild(h3);

		Element.setStyle(h1, this.style.font);		
		Element.setStyle(h2, this.style.font);
		Element.setStyle(h3, this.style.font);

		var target = this.tableBody;
		var cut = (config.holdSize <= target.childNodes.length);
		if (config.order.toLowerCase() == "desc") {
			target.insertBefore(tr, target.firstChild);
			if (cut) {
				target.removeChild(target.lastChild);
			}
		} else {
			target.appendChild(tr);
			if (cut) {
				target.removeChild(target.firstChild);
			}
		}
	},
	setEventHandler : function() {
		Event.observe(document,        'mouseup',   this.mouseup, true);
		Event.observe(document,        'mousemove', this.mousemove, true);
		Event.observe('log4jsDialog_header', 'mousedown', this.dialogTitleClicked, true);
		Event.observe('log4jsDialog_rezier', 'mousedown', this.dialogResizeClicked, true);
	},
	sx : 0,
	sy : 0,
	wx : 0,
	wy : 0,
	dragging  : false,
	resizeing : false,
	activeWin : null,
	dialogTitleClicked : function(e) {
		var ap = InsideConsoleAppender;
		ap.activeWin	= $('log4jsDialog');
	
		ap.sx = Event.pointerX(e);
		ap.sy = Event.pointerY(e);
		ap.wx = Element.getStyle(ap.activeWin, 'left');
		ap.wy = Element.getStyle(ap.activeWin, 'top');
		ap.wx = new Number(ap.wx.substring(0, ap.wx.length - 2));
		ap.wy = new Number(ap.wy.substring(0, ap.wy.length - 2));
		
		ap.dragging = true;
		Element.setStyle(ap.activeWin, {borderColor: 'red'});
	},
	dialogResizeClicked : function(e) {
		var ap = InsideConsoleAppender;
		ap.activeWin	= $('log4jsDialog');
	
		ap.sx = Event.pointerX(e);
		ap.sy = Event.pointerY(e);
		ap.wx = Element.getStyle(ap.activeWin, 'left');
		ap.wy = Element.getStyle(ap.activeWin, 'top');
		ap.wx = new Number(ap.wx.substring(0, ap.wx.length - 2));
		ap.wy = new Number(ap.wy.substring(0, ap.wy.length - 2));
			
		ap.resizing = true;
		Element.setStyle(ap.activeWin, {borderColor: 'red'});
	},
	mousemove : function(e) {
		var ap = InsideConsoleAppender;
	
		if (ap.dragging) {
			var x = Event.pointerX(e);
			var y = Event.pointerY(e);
			Event.stop(e);//ffoxにてドラック中にテキストが選択されてしまうのを抑止する。
			var mx = (x - ap.sx + ap.wx) + "px";
			var my = (y - ap.sy + ap.wy) + "px";
			Element.setStyle(ap.activeWin, {top: my,left: mx});

		} else if (ap.resizing) {
			var x = Event.pointerX(e);
			var y = Event.pointerY(e);
			Event.stop(e);//ffoxにてドラック中にテキストが選択されてしまうのを抑止する。
			var w = (x - ap.wx) + "px";
			var h = (y - ap.wy) + "px";
			var h2 = (y - ap.wy - ap.style.headerHeight - ap.style.footerHeight) + "px";
			Element.setStyle(ap.activeWin, {width: w,height: h});
			Element.setStyle("log4jsDialog_body", {height: h2});
		}
	},
	mouseup : function(e) {
		var ap = InsideConsoleAppender;

		ap.dragging	= false;
		ap.resizing	= false;
		if (ap.activeWin) {
			Element.setStyle(ap.activeWin, {borderColor: '#666666'});
		}
		ap.activeWin	= null;
	},
	closeDialog : function() {
		Element.hide('log4jsDialog');
	},
	style : {
		windowWidth		: 220,
		windowHeight	: 300,
		headerHeight	: 16,
		footerHeight	: 16,
		win    : {
			position		: "absolute",
			top				: "0px",
			left			: "0px",
			borderStyle		: "solid",
			borderWidth		: "1px",
			borderColor		: "#666666"
		},
		header : {
			color			: "#eeeeee",
			backgroundColor	: "#666666",
			width			: "100%",
			cursor			: "move"
		},
		body   : {
			color			: "#333333",
			backgroundColor	: "#dddddd",
			width			: "100%",
			overflow		: "scroll",
			filter          : 'alpha(opacity=75)',
			opacity         : '0.75'	
		},
		resizer : {
			color			: "#333333",
			backgroundColor	: "#aaaaaa",
			width			: "100%",
			cursor			: "nw-resize"
		},
		font    : {
			fontFamily		: "MS UI Gothic",
			fontSize		: "9pt"
		}		
	}
};

/** リモートへログ取得要求を行うアペンダ(phpのみ実装あり) */
var ReomteAppender = {
	cache      : "",
	cacheCount : 0,
	write : function(config, level, ts, msg) {	
		this.cache += ts + "\t" + msg;
		this.cacheCount++;
		var sz = config.cacheSize;
		if (this.cacheCount >= sz) {
			param = "mode=" + level + "&msg=" + encodeURIComponent(msg);
			new Ajax.Request(config.uri, {method: 'get', parameters: param});
			this.cache = "";
			this.cacheCount = 0;
		} else {
			this.cache += "\r\n";
		}
	}
};

/** アペンダのショートカット */
var Appender = {
	console      : InsideConsoleAppender,
	consoleFloat : InsideConsoleAppender,
	remote       : ReomteAppender
};




/*
LogConfig.add(
	{
	    category : null,
	    level    : LOG_LEVEL.DEBUG,
		appender : {
		    obj      : Appender.consoleFloat,  //アペンダオブジェクト.consoleFloat:ウインドウ内のコンソール
		    holdSize : 99,                     //最大表示件数
		    order    : 'desc'                  //方向　asc:新しいログを末尾に追加 | desc:先頭に追加
		}
	}, {
	    category : 'common.inputCheck',
	    level    : LOG_LEVEL.WARN,
		appender : {
		    obj      : Appender.remote,         //アペンダオブジェクト.
		    uri      : '../server/php/log.php', //要求ＵＲＩ
		    cacheSize: 1                        //ログを貯めて要求する。1で毎回要求
		}
	}
);
*/



