/*
  liboftp: this is an FTP library to simplify the work to a Developer
  who want to work with FTP servers (RFC 959).

  Copyright (c) 2009 hirohito higashi. All rights reserved.
  This file is distributed under BSD license.
*/


/***** Feature test switches ************************************************/
/***** System headers *******************************************************/
#include <unistd.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>


/***** Local headers ********************************************************/
#include "liboftp.h"
#include "sub.h"


/***** Constat values *******************************************************/
#define	TRANSFER_SEGMENT_SIZE	4096		/* 一度のrecv()で転送するバイト数 */


/***** Macros ***************************************************************/
/***** Typedefs *************************************************************/
/***** Function prototypes **************************************************/
/***** Local variables ******************************************************/
/***** Global variables *****************************************************/
/***** Signal catching functions ********************************************/
/***** Local functions ******************************************************/

/****************************************************************************/
/*! バッファへリスト取得 作業ルーチン
 *
 *@param	ftp	LIBOFTPへのポインタ。
 *@param	cmd	コマンド (RETR|LIST|NLST)
 *@param	fname	ファイル名または、グロブ。
 *@param	buf	バッファへのポインタ
 *@param	bufsiz	バッファサイズ
 *@retval	int	取得したバイト数　マイナス値ならエラーコード
 *@note
 */
static int ftp_get_buffer_main( LIBOFTP *ftp, const char *cmd, const char *fname, char *buf, int bufsiz )
{
    int data_socket;
    char *p = buf;
    int res;

    /*
     * 受信準備
     */
    if( ftp->flag_passive ) {
	data_socket = ftp_getready_pasv( ftp, cmd, fname );
    } else {
	data_socket = ftp_getready_active( ftp, cmd, fname );
    }
    if( data_socket < 0 ) {
	return data_socket;
    }

    /*
     * タイムアウトが意図通りに働くように、分割してrecvする。
     */
    while( 1 ) {
	int n;
	int len = bufsiz;
	if( len > TRANSFER_SEGMENT_SIZE ) {
	    len = TRANSFER_SEGMENT_SIZE;
	}
	n = recv( data_socket, p, len, 0 );
	DEBUGPRINT1( "RECV: n=%d\n", n );
	if( n < 0 ) {
	    int ret;
	    DEBUGPRINT1( "recv error. %s\n", strerror(errno) );
	    if( errno == EAGAIN ) {
		ret = LIBOFTP_ERROR_TIMEOUT;
	    } else {
		ret = LIBOFTP_ERROR_OS;
		copy_strerror();
	    }
	    close( data_socket );
	    return ret;
	}
	if( n == 0 ) {
	    break;
	}
	p += n;
	bufsiz -= n;
	if( bufsiz <= 0 ) {
	    break;		/* buffer too small */
	}
    }

    if( bufsiz > 0 ) {		/* バッファ不足だったか？ */
	/*
	 * 不足していない
	 */
	close( data_socket );

	/* receive response. */
	res = ftp_receive_response( ftp, ftp->error_message, sizeof(ftp->error_message)-1 );
	if( res != 226 ) {						/* 226: Closing data connection. */
	    DEBUGPRINT1( "got illegal response %d\n", res );
	    return res < 0? res: LIBOFTP_ERROR_PROTOCOL;
	}

	return p - buf;		/* return with transfered bytes */
	
    } else {
	/*
	 * バッファ不足時
	 */
	DEBUGPRINT1( "buffer too small. %s\n", "" );
	strncpy( ftp->error_message, "buffer too small", sizeof( ftp->error_message ) - 1 );

	if( ftp_send_command( ftp, "ABOR\r\n" ) < 0 ) {
	    DEBUGPRINT1( "command sending error. %s\n", "ABOR" );
	    close( data_socket );
	    return LIBOFTP_ERROR_BUFFER;
	}

	res = ftp_receive_response( ftp, 0, 0 );
	if( res == 426 ) {						/* 426: Connection closed; transfer aborted. */
	    res = ftp_receive_response( ftp, 0, 0 );
	}
	close( data_socket );
	if( res != 226 ) {						/* 226: Closing data connection. */
	    DEBUGPRINT1( "got illegal response %d\n", res );
	    return res < 0? res: LIBOFTP_ERROR_PROTOCOL;
	}

	return LIBOFTP_ERROR_BUFFER;
    }
}



/***** Global functions *****************************************************/

/****************************************************************************/
/*! バッファへファイル取得
 *
 *@param	ftp	LIBOFTPへのポインタ。
 *@param	fname	サーバ上のファイル名
 *@param	buf	バッファへのポインタ
 *@param	bufsiz	バッファサイズ
 *@retval	int	取得したバイト数　マイナス値ならエラーコード
 *@note
 */
int ftp_get_buffer( LIBOFTP *ftp, const char *fname, char *buf, int bufsiz )
{
    return ftp_get_buffer_main( ftp, "RETR", fname, buf, bufsiz );
}



/****************************************************************************/
/*! ディレクトリリスト(LIST) 取得
 *
 *@param	ftp	LIBOFTPへのポインタ。
 *@param	fglob	ファイルリストグロブ (ex: *.txt) or NULL
 *@param	buf	バッファへのポインタ
 *@param	bufsiz	バッファサイズ
 *@retval	int	エラーコード
 *@note
 */
int ftp_list( LIBOFTP *ftp, const char *fglob, char *buf, int bufsiz )
{
    int ret;

    ret = ftp_get_buffer_main( ftp, "LIST", fglob, buf, bufsiz );
    if( ret > 0 ) {
	buf[ret] = 0;
	return LIBOFTP_NOERROR;
    }

    return ret;
}



/****************************************************************************/
/*! ディレクトリリスト(NLST) 取得
 *
 *@param	ftp	LIBOFTPへのポインタ。
 *@param	fglob	ファイルリストグロブ (ex: *.txt) or NULL
 *@param	buf	バッファへのポインタ
 *@param	bufsiz	バッファサイズ
 *@retval	int	エラーコード
 *@note
 */
int ftp_nlist( LIBOFTP *ftp, const char *fglob, char *buf, int bufsiz )
{
    int ret;

    ret = ftp_get_buffer_main( ftp, "NLST", fglob, buf, bufsiz );
    if( ret > 0 ) {
	buf[ret] = 0;
	return LIBOFTP_NOERROR;
    }

    return ret;
}
