/*
  liboftp: this is an FTP library to simplify the work to a Developer
  who want to work with FTP servers (RFC 959).

  Copyright (c) 2009 hirohito higashi. All rights reserved.
  This file is distributed under BSD license.
*/


/***** Feature test switches ************************************************/
/***** System headers *******************************************************/
#include <unistd.h>
#include <string.h>
#include <netdb.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <errno.h>


/***** Local headers ********************************************************/
#include "liboftp.h"
#include "sub.h"


/***** Constat values *******************************************************/
/***** Macros ***************************************************************/
/***** Typedefs *************************************************************/
/***** Function prototypes **************************************************/
/***** Local variables ******************************************************/
/***** Global variables *****************************************************/
/***** Signal catching functions ********************************************/
/***** Local functions ******************************************************/
static const char * my_hstrerror( int no )
{
    switch( no ) {
    case HOST_NOT_FOUND:
        return "Unknown host";
	
    case TRY_AGAIN:
        return "Host name lookup failure";

    case NO_RECOVERY:
        return "Unknown server error";

    case NO_DATA:
        return "No address associated with name";

    default:
	return "Unknown error";
    }
}



/***** Global functions *****************************************************/

/****************************************************************************/
/*! ftp構造体初期化
 *
 *@param	ftp	LIBOFTPへのポインタ。
 *@retval	int	エラーコード
 *@note
 * ftp構造体は、呼出側で確保する。
 */
int ftp_initialize( LIBOFTP *ftp )
{
    memset( ftp, 0, sizeof( LIBOFTP ) );
    ftp->timeout_sec = 10;			/* default timeout */

    return LIBOFTP_NOERROR;
}



/****************************************************************************/
/*! ftpサーバへ接続
 *
 *@param	ftp	LIBOFTPへのポインタ。
 *@param	host	ホスト名または、IPアドレス文字列。
 *@param	port	ポート番号。ゼロならデフォルトを使用する。
 *@retval	int	エラーコード
 *@note
 * ftp構造体は、呼出側で確保する。
 */
int ftp_open( LIBOFTP *ftp, const char *host, int port )
{
    struct hostent *p_hostent;
    int res;

    if( port == 0 ) port = 21;

    /*
     * search host entry
     */
    p_hostent = gethostbyname( host );
    if( p_hostent == NULL ) {
	DEBUGPRINT1( "gethostbyname() function failed. %s\n", my_hstrerror(h_errno) );
	strncpy( ftp->error_message, my_hstrerror(h_errno), sizeof(ftp->error_message)-1 );
	return LIBOFTP_ERROR_OS;
    }
    ftp->saddr.sin_family = AF_INET;
    memcpy( &ftp->saddr.sin_addr, p_hostent->h_addr, p_hostent->h_length );
    ftp->saddr.sin_port = htons( port );

    /*
     * make control connection.
     */
    ftp->socket = socket( PF_INET, SOCK_STREAM, IPPROTO_TCP );
    if( ftp->socket < 0 ) {
	DEBUGPRINT1( "can't create socket. %s\n", strerror(errno) );
	copy_strerror();
	ftp->socket = 0;
	return LIBOFTP_ERROR_OS;
    }

    if( connect( ftp->socket, (struct sockaddr *)&(ftp->saddr), sizeof(struct sockaddr) ) < 0 ) {
	DEBUGPRINT1( "can't connect socket. %s\n", strerror(errno) );
	copy_strerror();
	close( ftp->socket );
	ftp->socket = 0;
	return LIBOFTP_ERROR_OS;
    }

    res = ftp_timeout( ftp, ftp->timeout_sec );
    if( res < 0 ) {
	close( ftp->socket );
	ftp->socket = 0;
	return res;
    }

    /*
     * receive banner
     */
    res = ftp_receive_response( ftp, ftp->error_message, sizeof(ftp->error_message) );
    if( res != 220 ) {						/* 220: Service ready for new user. */
	DEBUGPRINT1( "Illegal response. %d\n", res );
	close( ftp->socket );
	ftp->socket = 0;
	return res < 0? res: LIBOFTP_ERROR_PROTOCOL;
    }

    return LIBOFTP_NOERROR;
}
