/*
  liboftp: this is an FTP library to simplify the work to a Developer
  who want to work with FTP servers (RFC 959).

  Copyright (c) 2009 hirohito higashi. All rights reserved.
  This file is distributed under BSD license.
*/


/***** Feature test switches ************************************************/
/***** System headers *******************************************************/
#include <unistd.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/uio.h>
#include <fcntl.h>
#include <string.h>
#include <errno.h>


/***** Local headers ********************************************************/
#include "liboftp.h"
#include "sub.h"


/***** Constat values *******************************************************/
#define	TRANSFER_SEGMENT_SIZE	1024		/* 一度のrecv()で転送するバイト数 */


/***** Macros ***************************************************************/
/***** Typedefs *************************************************************/
/***** Function prototypes **************************************************/
/***** Local variables ******************************************************/
/***** Global variables *****************************************************/
/***** Signal catching functions ********************************************/
/***** Local functions ******************************************************/
/****************************************************************************/
/*! ファイル送信メイン
 *
 *@param	ftp	LIBOFTPへのポインタ。
 *@param	local_fname	ローカルファイル名
 *@param	fname	サーバ上のファイル名
 *@param	cmd	送信するFTPコマンド
 *@retval	int	エラーコード
 *@note
 */
static int ftp_put_file_main( LIBOFTP *ftp, const char *local_fname, const char *fname, const char *cmd )
{
    int data_socket;
    int fd;
    char buf[TRANSFER_SEGMENT_SIZE];
    int res;

    /*
     * ローカルファイルオープン
     */
    fd = open( local_fname, O_RDONLY );
    if( fd < 0 ) {
	DEBUGPRINT1( "local file open error. %s\n", strerror(errno) );
	return LIBOFTP_ERROR_OS;
    }
	
    /*
     * 送信準備
     */
    if( ftp->flag_passive ) {
	data_socket = ftp_getready_pasv( ftp, cmd, fname );
    } else {
	data_socket = ftp_getready_active( ftp, cmd, fname );
    }
    if( data_socket < 0 ) {
	close( fd );
	return data_socket;
    }

    /*
     * タイムアウトが意図通りに働くように、分割してsendする。
     */
    while( 1 ) {
	int n_rd, n_wr;
	n_rd = read( fd, buf, TRANSFER_SEGMENT_SIZE );
	if( n_rd == 0 ) break;

	if( n_rd < 0 ) {
	    if( errno == EINTR ) continue;

	    DEBUGPRINT1( "local file read error. %s\n", strerror(errno) );
	    copy_strerror();
	    close( data_socket );
	    return LIBOFTP_ERROR_OS;
	}

	n_wr = sendn( data_socket, buf, n_rd, 0 );
	DEBUGPRINT1( "SEND: n=%d\n", n_wr );
	if( n_wr < 0 ) {
	    copy_strerror();
	    close( data_socket );
	    return LIBOFTP_ERROR_OS;
	}
    }
    close( fd );
    close( data_socket );

    /*
     * receive response.
     */
    res = ftp_receive_response( ftp, ftp->error_message, sizeof(ftp->error_message)-1 );
    if( res != 226 ) {						/* 226: Closing data connection. */
	DEBUGPRINT1( "got illegal response %d\n", res );
	return res<0? res: LIBOFTP_ERROR_PROTOCOL;
    }

    return LIBOFTP_NOERROR;
}



/***** Global functions *****************************************************/

/****************************************************************************/
/*! ファイル送信
 *
 *@param	ftp	LIBOFTPへのポインタ。
 *@param	local_fname	ローカルファイル名
 *@param	fname	サーバ上のファイル名
 *@retval	int	エラーコード
 *@note
 */
int ftp_put_file( LIBOFTP *ftp, const char *local_fname, const char *fname )
{
    return ftp_put_file_main( ftp, local_fname, fname, "STOR" );
}



/****************************************************************************/
/*! ファイル送信 アペンドモード
 *
 *@param	ftp	LIBOFTPへのポインタ。
 *@param	local_fname	ローカルファイル名
 *@param	fname	サーバ上のファイル名
 *@retval	int	エラーコード
 *@note
 */
int ftp_append_file( LIBOFTP *ftp, const char *local_fname, const char *fname )
{
    return ftp_put_file_main( ftp, local_fname, fname, "APPE" );
}

