// Copyright (C) 2010 Kosei Demura
// This file is part of the irrDrawStuff library.
// Please read copyright notice in irrdrawstuff.h for conditions of distribution and usage.


#include "misc.h"
#include "vision.h"
#include "drawPrimitives.h"

namespace {
	SimLoop *sim = SimLoop::getInstance();
}

namespace {
	bool octree;
	scene::ISceneNode* model;
	core::stringc startUpModelFile;
	core::stringw caption;
}

Misc::Misc(IVideoDriver *driver, IrrlichtDevice *device)
{
	idriver = driver;
	idevice = device;
	octree = false;
	model  = 0;
}

Misc::~Misc()
{
}

void Misc::normalizeVector3(float v[3])
{
    float len = v[0]*v[0] + v[1]*v[1] + v[2]*v[2];
    if (len <= 0.0f)
    {
        v[0] = 1;
        v[1] = 0;
        v[2] = 0;
    }
    else
    {
        len = 1.0f / (float)sqrt(len);
        v[0] *= len;
        v[1] *= len;
        v[2] *= len;
    }
}


matrix4 Misc::setTransform(const float pos[3], const float R[12])
{
  matrix4 m;
  float matrix[16];
  // drawstuff 右手系
  // matrix[0]=R[0];
  // matrix[1]=R[4];
  // matrix[2]=R[8];
  // matrix[3]=0;
  // matrix[4]=R[1];
  // matrix[5]=R[5];
  // matrix[6]=R[9];
  // matrix[7]=0;
  // matrix[8]=R[2];
  // matrix[9]=R[6];
  // matrix[10]=R[10];
  // matrix[11]=0;
  // matrix[12]=pos[0];
  // matrix[13]=pos[1];
  // matrix[14]=pos[2];
  // matrix[15]=1;

  // 右手系と左手系は転置
  matrix[0]=R[0];
  matrix[1]=R[1];
  matrix[2]=R[2];
  matrix[3]=0;
  matrix[4]=R[4];
  matrix[5]=R[5];
  matrix[6]=R[6];
  matrix[7]=0;
  matrix[8]=R[8];
  matrix[9]=R[9];
  matrix[10]=R[10];
  matrix[11]=0;
  matrix[12]=pos[1];
  matrix[13]=pos[2];
  matrix[14]=-pos[0];
  matrix[15]=1;
  m.setM(matrix);
  return m;
}


// 影を作るための変換行列
void Misc::setShadowTransform()
{
    float matrix[16];
    matrix4 m;

	// 指向性光源(Directional light)から地面への射影変換
	// ここでは光源の高さを1とした。
	// drawstuffの座標系：光ベクトル(-LIGHTX, -LIGHTY, 1)
	// irrlichtの座標系：光ベクトル(-LIGHTY, 1, LIGHTX)
	for (int i=0; i<16; i++) matrix[i] = 0;
	matrix[0] = 1;
	matrix[4] = - sim->vision->getLightX();
	matrix[6] = - sim->vision->getLightZ();
	matrix[10] = 1;
	matrix[15] = 1;  // Height

	m.setM(matrix);
	m = idriver->getTransform(ETS_WORLD) * m;
	//ids->driver->setTransform(ETS_WORLD, core::matrix4());
	idriver->setTransform(ETS_WORLD, m);

    //return m;

}


// ***********************************
// 091227 hosokawa
// make a bitmap texture file which color is argument (r,g,b)
// ビットマップ形式のテクスチャファイルを生成．色は引数(r,g,b)
// ***********************************
void Misc::makeTexture(char *file_name, float r, float g, float b)
{
    FILE *fp;
    int red   = (int)(r*255)*0x10000;
    int green = (int)(g*255)*0x100;
    int blue  = (int)(b*255);
    unsigned int color = red + green + blue;
    unsigned char bmpHeader[54] =
    {
        'B', 'M',     // bfType; ファイルタイプ
        0,  0, 0, 0,  // bfSize; ファイルサイズ
        0,  0, 0, 0,  // bfReserved1,2; 予約
        54, 0, 0, 0,  // bfOffBits; DIB形式ファイルの先頭からピクセルデータ領域の先頭までのオフセット
        40, 0, 0, 0,  // biSize; ここから始まるヘッダのサイズ
        1,  0, 0, 0,  // biWidth; ビットマップの幅
        1,  0, 0, 0,  // biHeight; ビットマップの高さ
        0x01, 1,      // biPlanes; プレーン数
        32, 0,        // biBitCount; 1ピクセルあたりのビット数
        0,  0, 0, 0,  // biCompression; 圧縮タイプ BI_RGB, BI_RLE8, BI_RLE4のいずれか
        0,  0, 0, 0,  // biSizeImage; イメージの全バイト数
        0,  0, 0, 0,  // biXPelsPerMeter; 水平解像度(px/m)
        0,  0, 0, 0,  // biYPelsPerMeter; 垂直解像度(px/m)
        0,  0, 0, 0,  // biClrUsed; カラーインデックス数
        0,  0, 0, 0,  // biClrImportant; 重要なカラーインデックス数
    };

    if ((fp = fopen(file_name, "wb+"))== NULL)
    {
        cout << "Cannot open a draw texture file" << endl;
        exit(1);
    }

    fseek(fp, 0L, SEEK_SET);
    fwrite(bmpHeader, 1, 54, fp);
    fwrite(&color, 4, 100, fp);

    fclose(fp);
}

// drawstuff to irrDrawstuff
void Misc::dsToiDS(const double ode_pos[3], const double ode_R[12], double pos[3], vector3df *Rot)
{
    pos[0] =   ode_pos[1] + 1;
	pos[1] =   ode_pos[2];
	pos[2] = - ode_pos[0];


	// set rottation
    dQuaternion result;

    // rotating matrix to quaternion
    dRtoQ((const dReal *) ode_R, (dReal *) result);

    // ODE to irrlicht (quaternion)
    core::quaternion quat;

    quat.W = (f32)  -result[0];   /* right-hand system to left-hand system　*/
    quat.X = (f32)  result[2];   /*       ODE ==　right-hand system　       */
    quat.Y = (f32)  result[3];   /*       irrlicht == left-hand system         */
    quat.Z = (f32) -result[1];

    // rotating matrix(irrlicht)
    //core::vector3df Rot;

    // quaternion to Euler angle
    quat.toEuler(*Rot);
    double roll, pitch, yaw;

    // get Euler angle
    roll  =  Rot->X;
    pitch =  Rot->Y;
    yaw   =  Rot->Z;

    Rot->X = (irr::f32) (roll  * RADTODEG64);
    Rot->Y = (irr::f32) (pitch * RADTODEG64);
    Rot->Z = (irr::f32) (yaw   * RADTODEG64);
}

void Misc::updateForDraw(scene::ISceneNode* irrSceneNode,
                    const double pos[3], const double R[12])
{
    // set position
    irrSceneNode->setPosition(core::vector3df((f32)pos[1],(f32)pos[2],(f32)-pos[0]));

    // set rottation
    dQuaternion result;

    // rotating matrix to quaternion
    dRtoQ((const dReal *) R, (dReal *) result);

    // ODE to irrlicht (quaternion)
    core::quaternion quat;

    quat.W = (f32)  -result[0];   /* right-hand system to left-hand system　*/
    quat.X = (f32)  result[2];   /*       ODE ->　right-hand system　       */
    quat.Y = (f32)  result[3];   /*       ODE -> left-hand system         */
    quat.Z = (f32) -result[1];

    // rotating matrix(irrlicht)
    core::vector3df Rot;

    // quaternion to Euler angle
    quat.toEuler(Rot);
    double roll, pitch, yaw;

    // get Euler angle
    roll  =  Rot.X;
    pitch =  Rot.Y;
    yaw   =  Rot.Z;

    Rot.X =  (irr::f32) (roll  * RADTODEG64);
    Rot.Y =  (irr::f32) (pitch * RADTODEG64);
    Rot.Z =  (irr::f32) (yaw   * RADTODEG64);

    // set rotation
    irrSceneNode->setRotation(Rot);
}

void Misc::odeToIrrRotation(const double R[12])
{
   // 姿勢の設定
    dQuaternion result;

    // 回転行列からクオータニオンへ変更
    dRtoQ((const dReal *) R, (dReal *) result);

    // ODEからirrlichtのクオータニオンへ変更
    core::quaternion quat;
    quat.W = (irr::f32) -result[0];   /* right-hand system to left-hand system　*/
    quat.X = (irr::f32)  result[2];   /*       ODE ->　right-hand system　       */
    quat.Y = (irr::f32)  result[3];   /*       ODE -> left-hand system         */
    quat.Z = (irr::f32) -result[1];

    // irrlichtの回転行列
    core::vector3df Rot;

    // クオータニオンからオイラー角へ
    quat.toEuler(Rot);
    double roll, pitch, yaw;

    // オイラー角の取得
    roll  =  Rot.X;
    pitch =  Rot.Y;
    yaw   =  Rot.Z;

    Rot.X =  (irr::f32) roll  * RADTODEG64;
    Rot.Y =  (irr::f32) pitch * RADTODEG64;
    Rot.Z =  (irr::f32) yaw   * RADTODEG64;

	matrix4 matrix;
 	matrix.setRotationDegrees(vector3df(Rot.X,Rot.Y,Rot.Z));// 回転 rotation
	idriver->setTransform(ETS_WORLD, matrix);// ワールドに反映
}

void Misc::odeToIrrTransform(const double pos[3],const double R[12])
{
   // 姿勢の設定
    dQuaternion result;

    // 回転行列からクオータニオンへ変更
    dRtoQ((const dReal *) R, (dReal *) result);

    // ODEからirrlichtのクオータニオンへ変更
    core::quaternion quat;
    quat.W = (irr::f32) -result[0];   /* right-hand system to left-hand system　*/
    quat.X = (irr::f32)  result[2];   /*       ODE ->　right-hand system　       */
    quat.Y = (irr::f32)  result[3];   /*       ODE -> left-hand system         */
    quat.Z = (irr::f32) -result[1];

    // irrlichtの回転行列
    core::vector3df Rot;

    // クオータニオンからオイラー角へ
    quat.toEuler(Rot);
    double roll, pitch, yaw;

    // オイラー角の取得
    roll  =  Rot.X;
    pitch =  Rot.Y;
    yaw   =  Rot.Z;

    Rot.X =  (irr::f32) roll  * RADTODEG64;
    Rot.Y =  (irr::f32) pitch * RADTODEG64;
    Rot.Z =  (irr::f32) yaw   * RADTODEG64;

	matrix4 matrix;
    matrix.setTranslation(vector3df(pos[1],pos[2],-pos[0]));//右に移動
	matrix.setRotationDegrees(vector3df(Rot.X,Rot.Y,Rot.Z));// 回転 rotation
	idriver->setTransform(ETS_WORLD, matrix);// ワールドに反映
}

void Misc::odeToIrrTransform(const double pos[3], const double R[12], float size)
{
   // 姿勢の設定
    dQuaternion result;

    // 回転行列からクオータニオンへ変更
    dRtoQ((const dReal *) R, (dReal *) result);

    // ODEからirrlichtのクオータニオンへ変更
    core::quaternion quat;
    quat.W = (irr::f32) -result[0];   /* right-hand system to left-hand system　*/
    quat.X = (irr::f32)  result[2];   /*       ODE ->　right-hand system　       */
    quat.Y = (irr::f32)  result[3];   /*       ODE -> left-hand system         */
    quat.Z = (irr::f32) -result[1];

    // irrlichtの回転行列
    core::vector3df Rot;

    // クオータニオンからオイラー角へ
    quat.toEuler(Rot);
    double roll, pitch, yaw;

    // オイラー角の取得
    roll  =  Rot.X;
    pitch =  Rot.Y;
    yaw   =  Rot.Z;

    Rot.X =  (irr::f32) roll  * RADTODEG64;
    Rot.Y =  (irr::f32) pitch * RADTODEG64;
    Rot.Z =  (irr::f32) yaw   * RADTODEG64;

	matrix4 matrix;
    matrix.setTranslation(vector3df(pos[1],pos[2],-pos[0]));//右に移動
	matrix.setRotationDegrees(vector3df(Rot.X,Rot.Y,Rot.Z));// 回転 rotation
	matrix.setScale(vector3df(size, size, size));
	idriver->setTransform(ETS_WORLD, matrix);// ワールドに反映
}

/*
From Irrlicht Tutorial 9: Mesh Viewer
The second function loadModel() loads a model and displays it using an
addAnimatedMeshSceneNode and the scene manager. Nothing difficult. It also
displays a short message box, if the model could not be loaded.
*/
void Misc::loadModel(const c8* fn)
{
	// modify the name if it a .pk3 file
	core::stringc filename(fn);
	core::stringc extension;
	core::getFileNameExtension(extension, filename);
	extension.make_lower();

	// if a texture is loaded apply it to the current model..
	if (extension == ".jpg" || extension == ".pcx" ||
		extension == ".png" || extension == ".ppm" ||
		extension == ".pgm" || extension == ".pbm" ||
		extension == ".psd" || extension == ".tga" ||
		extension == ".bmp" || extension == ".wal")
	{
		video::ITexture *texture = idevice->getVideoDriver()->getTexture( filename );
		if ( texture && model )
		{
			// always reload texture
			idevice->getVideoDriver()->removeTexture(texture);
			texture = idevice->getVideoDriver()->getTexture( filename );
			model->setMaterialTexture(0, texture);
		}
		return;
	}
	// if an archive is loaded add it to the FileArchive..
	else if (extension == ".pk3" || extension == ".zip")
	{
		idevice->getFileSystem()->addZipFileArchive(filename.c_str());
		return;
	}
	else if (extension == ".pak")
	{
		idevice->getFileSystem()->addPakFileArchive(filename.c_str());
		return;
	}

	// load a model into the engine
	if (model) model->remove();

	model = 0;
	scene::IAnimatedMesh* m = idevice->getSceneManager()->getMesh( filename.c_str() );

	if (!m)
	{
		// model could not be loaded
		if (startUpModelFile != filename)
			idevice->getGUIEnvironment()->addMessageBox(
			caption.c_str(), L"The model could not be loaded. " \
			L"Maybe it is not a supported file format.");
		return;
	}

	// set default material properties
	if (octree)
		model = idevice->getSceneManager()->addOctreeSceneNode(m->getMesh(0));
	else
	{
		scene::IAnimatedMeshSceneNode* animModel = idevice->getSceneManager()->addAnimatedMeshSceneNode(m);

		animModel->setAnimationSpeed(0);
		model = animModel;
	}
	float s0 = 0.04, s1 = 0.04, s2 = 0.04;
	float pos[]= {1, 1, 1};
	//model->setScale(vector3df(s0,s1,s2)); // モデルのスケール
	model->setPosition(vector3df(pos[0],pos[1],pos[2])); // モデルの位置
	//model->setMaterialFlag(video::EMF_LIGHTING, useLight);
	//model->setMaterialFlag(video::EMF_NORMALIZE_NORMALS, useLight);
	//model->setDebugDataVisible(scene::EDS_OFF);
}

void diSetTextures(int a)
{
    sim->setUseTextures((bool) (a != 0));
}

extern "C" void diStop()
{
#ifdef WIN32
    if (main_window) PostMessage (main_window,WM_QUIT,0,0);
#endif
}


extern "C" double diElapsedTime()
{
    static double prev=0.0;
    double currentTime;
    double curr;

#ifdef WIN32
    currentTime = timeGetTime();
    curr = currentTime/1000.0;
#else
    timeval tv ;
    gettimeofday(&tv, 0);
    curr = tv.tv_sec + (double) tv.tv_usec / 1000000.0 ;
#endif

    if (!prev) prev = curr;
    double retval = curr-prev;
    prev=curr;
    if (retval>1.0) retval=1.0;
    if (retval< DBL_EPSILON) retval = DBL_EPSILON;
    return retval;
}

extern "C" void diSetTexture(int texture_number)
{
    assert(! (sim->getCurrentState() != 2) && "drawing function called outside simulation loop");
    sim->pri->setTextureNum(texture_number);
}


