﻿/*
*Copyright(c) Live2D Inc. All rights reserved.
*このソースコードの使用には、"http://www.live2d.com/eula/live2d-open-software-license-agreement_jp.html"ファイルにあるLive2D Open Software ライセンスが適用されます。
*Use of this source code is governed by a Live2D Open Software license that can be found in the "http://www.live2d.com/eula/live2d-open-software-license-agreement_en.html" file.
*/

using System.Collections.Generic;
using System.IO;
using UnityEngine;
using UnityEngine.UI;
using UnityEditor;
using live2d.Mecanim.EditorExtensions;

[CustomEditor(typeof(EuclidModel))]
public class EuclidModelEditor : Editor
{
    private delegate void Action();

    private Action textureAction = delegate {};

    public override void OnInspectorGUI()
    {
        var model = target as EuclidModel;
        var asset = (TextAsset)EditorGUILayout.ObjectField(
            "Euclid Model", model.configure.EuclidFile, typeof(TextAsset), false);

        // インスペクタ上でアセットが変化していたらテクスチャの再読込を行う
        if (asset != model.configure.EuclidFile)
        {
            textureAction = delegate {};
            if (ReloadTextureList(model.configure, asset))
            {
                Undo.RecordObject(target, "Change Euclid Model");
                model.configure.EuclidFile = asset;
            }
        }

        textureAction();

        if (model.configure.EuclidFile != null)
        {
            DrawModelInspector(model);
            GenerateParameterScript(model);
        }
    }

    private bool ReloadTextureList(live2d.Euclid.EuclidModelConfig configure, TextAsset asset)
    {
        configure.TextureList.Clear();

        // アセットは削除されてもよい
        if (asset == null) { return true; }

        // FIXME: Resources.LoadAllを使っているのでAssets/Resources下からしか引っ張ってこれない制約がある
        var prefix = "Assets/Resources/";

        var apath = AssetDatabase.GetAssetPath(asset);
        if (!apath.StartsWith(prefix))
        {
            textureAction = new Action(() =>
                EditorGUILayout.HelpBox(
                    "The model asset should be reachable from Assets/Resources folder.\n" +
                    "And the textures should be contained at folder that is silbing of " +
                    "the model file and named as same.\n" +
                    "e.g.\n" +
                    "   Assets/Resources/\n" +
                    "       Charactor.bytes\n" +
                    "       Charactor/texture_01.png\n" +
                    "       Charactor/texture_02.png\n" +
                    "       ...",
                    MessageType.Error, true));
            return false;
        }

        var relpath = apath.Substring(prefix.Length);

        var file = new FileInfo(apath);
        var targetDir = relpath.Substring(0, relpath.Length - file.Extension.Length);

        var candidates = Resources.LoadAll<Texture2D>(targetDir);
        if (candidates != null)
        {
            configure.TextureList.AddRange(candidates);
            textureAction = new Action(() =>
                EditorGUILayout.HelpBox("Lookup textures from " + targetDir, MessageType.Info, true));
            return true;
        }

        textureAction = new Action(() =>
            EditorGUILayout.HelpBox("No eligible texture found.", MessageType.Error, true));
        return false;
    }

    private void DrawModelInspector(EuclidModel model)
    {
        // TODO: デバッグモードだといろいろ表示できたりするやつを作る
        DrawDefaultInspector();
    }

    private void GenerateParameterScript(EuclidModel model)
    {
        if (GUILayout.Button("Generate Parameter Script"))
        {
            var info = model.getParametersInfo();

            var path = AssetDatabase.GetAssetPath(model.configure.EuclidFile);
            var outPath = GenerateParametersScriptUi.DefaultScriptPath(path);

            GenerateParametersScriptUi.GenerateParametersScript(info, outPath);
        }
    }
}
