﻿/*
*Copyright(c) Live2D Inc. All rights reserved.
*このソースコードの使用には、"http://www.live2d.com/eula/live2d-open-software-license-agreement_jp.html"ファイルにあるLive2D Open Software ライセンスが適用されます。
*Use of this source code is governed by a Live2D Open Software license that can be found in the "http://www.live2d.com/eula/live2d-open-software-license-agreement_en.html" file.
*/

using System;
using System.Collections.Generic;
using UnityEngine;
using live2d.Euclid;

public class EuclidModel : MonoBehaviour
{
    [SerializeField]
    public EuclidModelConfig configure = new EuclidModelConfig();

    public List<EuclidModelBackend> backends = new List<EuclidModelBackend>();

    #region Unity Functions
    private void Awake()
    {
        SetupShaders();
    }

    private void Update()
    {
        transform.position = configure.RootJoint.transform.position;
    }

    private void OnDestroy()
    {
        backends.ForEach(m => m.Destroy());
        backends.Clear();
    }
    #endregion

    private EuclidModelBackend CreateOrphanBackend()
    {
        SetupShaders();
        return new EuclidModelBackend(configure, transform);
    }

    private void LoadShader(ref Shader shader, string name)
    {
        if (shader != null) { return; }
        shader = Shader.Find(name);
    }

    private void SetupShaders()
    {
        ///Shaderの設定はここで行います。
        ///自前のシェーダを使用する場合はここを変更してください。
        LoadShader(ref configure.ModelShader,         "Euclid/EuclidModelShader");
        LoadShader(ref configure.ClippingBaseShader,  "Euclid/EuclidClippingBaseShader");
        LoadShader(ref configure.ClippingModelShader, "Euclid/EuclidClippingModelShader");
        LoadShader(ref configure.ClippingMaskShader,  "Euclid/EuclidClippingMaskShader");
    }

    public EuclidModelBackend AddBackend()
    {
        var model = CreateOrphanBackend();
        backends.Add(model);
        return model;
    }

    public ParameterInfo[] getParametersInfo()
    {
        if (backends.Count > 0)
        {
            return backends[0].getParametersInfo();
        }
        return EuclidModelBackend.GetParametersInfo(configure);
    }

    public float getParameterValue(string parameterName)
    {
        if (string.IsNullOrEmpty(parameterName))
        {
            throw new ArgumentNullException();
        }
        // FIXME: ビュー依存パラメータは無視しないといけない
        if (backends.Count > 0)
        {
            return backends[0].getParameterValue(parameterName);
        }
        else
        {
            // 本来は、ここに来るようなものはないはず
            var model = CreateOrphanBackend();
            model.Initialize(null, false);
            var info = model.getParameterValue(parameterName);
            model.Destroy();
            return info;
        }
    }

    public void setParameterValue(string parameterName, float value)
    {
        if (string.IsNullOrEmpty(parameterName))
        {
            throw new ArgumentNullException();
        }
        // FIXME: ビュー依存パラメータは無視しないといけない
        for (int i = 0; i < backends.Count; i++)
        {
            backends[i].setParameterValue(parameterName, value);
        }
    }
}
