﻿/*
*Copyright(c) Live2D Inc. All rights reserved.
*このソースコードの使用には、"http://www.live2d.com/eula/live2d-open-software-license-agreement_jp.html"ファイルにあるLive2D Open Software ライセンスが適用されます。
*Use of this source code is governed by a Live2D Open Software license that can be found in the "http://www.live2d.com/eula/live2d-open-software-license-agreement_en.html" file.
*/

using System;
using System.IO;
using System.Linq;
using System.Text;
using UnityEditor;
using UnityEngine;
using ParameterInfo =  live2d.Euclid.ParameterInfo;

namespace live2d.Mecanim.EditorExtensions
{
    public class GenerateParametersScriptUi : EditorWindow
    {
        private static ParameterInfo[] GetParametersInfo(string moeBytesPath)
        {
            // Load the text asset at the given path.
            var moeTextAsset = AssetDatabase.LoadAssetAtPath<TextAsset>(moeBytesPath);

            if (moeTextAsset == null)
            {
                throw new IOException("Failed to load moe asset at path " + moeBytesPath);
            }


            // Create a temporary object for extracting parameters information from it.
            var tempModelObject = new GameObject();
            var model = tempModelObject.AddComponent<EuclidModel>();


            model.configure.EuclidFile = moeTextAsset;


            // Extract parameters from model.
            var parametersInfo = model.getParametersInfo();


            // Clean up.
            DestroyImmediate(tempModelObject);


            return parametersInfo;
        }

        internal static string DefaultScriptPath(string moeBytesPath)
        {
            return moeBytesPath.Replace(".bytes", "Parameters.cs");
        }

        private void ShowDragAndDropAreaAndHandleMoeDrops()
        {
            var currentEvent = Event.current;
            var dropArea = EditorGUILayout.GetControlRect(GUILayout.Height(256.0f));


            EditorGUI.HelpBox(dropArea, "Drop moe (*.bytes) file here", MessageType.Info);


            switch (currentEvent.type)
            {
                case EventType.DragUpdated:
                case EventType.DragPerform:
                    {
                        if (!dropArea.Contains(currentEvent.mousePosition))
                        {
                            return;
                        }


                        DragAndDrop.visualMode = DragAndDropVisualMode.Copy;


                        if (currentEvent.type == EventType.DragPerform)
                        {
                            foreach (var moeBytesPath in DragAndDrop.paths.Where(p => p.EndsWith(".bytes")))
                            {
                                var scriptPath = _scriptOutputPath;


                                // Generate an output path if necessary.
                                if (string.IsNullOrEmpty(scriptPath))
                                {
                                    scriptPath = DefaultScriptPath(moeBytesPath);
                                }


                                GenerateParametersScript(moeBytesPath, scriptPath);


                                DragAndDrop.AcceptDrag();


                                return;
                            }
                        }


                        break;
                    }
                default:
                    {
                        break;
                    }
            }
        }

        internal static void GenerateParametersScript(ParameterInfo[] paramInfo, string scriptPath)
        {
            // Do checks.
            if (paramInfo == null || string.IsNullOrEmpty(scriptPath))
            {
                throw new ArgumentNullException();
            }

            // Generate code.
            var code = new ParametersScriptGenerator()
                .SetClassName(Path.GetFileNameWithoutExtension(scriptPath))
                .SetParametersInfo(paramInfo)
                .GenerateCode();


            // Write the code and import it as a script.
            var absoluteScriptPath = Application.dataPath.Substring(0, Application.dataPath.Length - "Assets".Length) + "/" + scriptPath;


            File.WriteAllText(absoluteScriptPath, code, Encoding.UTF8);
            AssetDatabase.ImportAsset(scriptPath);
        }

        private static void GenerateParametersScript(string moeBytesPath, string scriptPath)
        {
            // Do checks.
            if (string.IsNullOrEmpty(moeBytesPath))
            {
                throw new ArgumentNullException();
            }

            GenerateParametersScript(GetParametersInfo(moeBytesPath), scriptPath);
        }

        #region Unity editor integration

        [MenuItem("Assets/Generate/Model Parameters Script")]
        private static void ShowUi()
        {
            GetWindow<GenerateParametersScriptUi>();
        }

        #endregion

        #region Unity event handling

        private string _scriptOutputPath;

        private void OnGUI()
        {
            // Allow the user to set an output path.
            _scriptOutputPath = EditorGUILayout.TextField("Script Output Path [optional]", _scriptOutputPath);


            // Show drag-and-drop area.
            ShowDragAndDropAreaAndHandleMoeDrops();
        }

        #endregion
    }
}
