﻿//-------------------------------------------------------------------------------------------------
// File : a3dUtil.h
// Desc : Utility Module.
// Copyright(c) Project Asura. All right reserved.
//-------------------------------------------------------------------------------------------------
#pragma once


namespace a3d {

//-------------------------------------------------------------------------------------------------
//! @brief      ネイティブ形式に変換します.
//!
//! @param[in]      format      A3D形式です.
//! @return     ネイティブ形式に変換したフォーマットを返却します.
//-------------------------------------------------------------------------------------------------
DXGI_FORMAT ToNativeFormat(RESOURCE_FORMAT format);

//-------------------------------------------------------------------------------------------------
//! @brief      タイプレスのネイティブ形式に変換します.
//!
//! @param[in]      format      A3D形式です.
//! @param[in]      isStencil   ステンシルテクスチャとする場合はtrueを指定.
//! @return     タイプレスのネイティブ形式に変換したフォーマットを返却します. 
//!             タイプレス形式が無い場合は該当するDXGI_FORMATを返却します.
//-------------------------------------------------------------------------------------------------
DXGI_FORMAT ToNativeTypelessFormat(RESOURCE_FORMAT format);

//-------------------------------------------------------------------------------------------------
//! @brief      ビューフォーマットに変換します.
//!
//! @param[in]      format      A3D形式です.
//! @param[in]      isStencil   ステンシルテクスチャとする場合はtrueを指定.
//! @return     ビューフォーマットに適した形式のフォーマットを返却します. 
//-------------------------------------------------------------------------------------------------
DXGI_FORMAT ToNativeViewFormat(RESOURCE_FORMAT format, bool isStencil);

//-------------------------------------------------------------------------------------------------
//! @brief      ネイティブ形式からA3D形式に変換します.
//!
//! @param[in]      format      DXGI形式です.
//! @return     ネイティブ形式からA3D形式に変換したフォーマットを返却します.
//-------------------------------------------------------------------------------------------------
RESOURCE_FORMAT ToWrapFormat(DXGI_FORMAT format);

//-------------------------------------------------------------------------------------------------
//! @brief      リソースフォーマットのビット数に変換します.
//!
//! @param[in]      format      リソースフォーマットです.
//! @return     リソースフォーマットのビット数を返却します.
//-------------------------------------------------------------------------------------------------
uint32_t ToBits(RESOURCE_FORMAT format);

//-------------------------------------------------------------------------------------------------
//! @brief      リソースフォーマットのバイト数に変換します.
//!
//! @param[in]      format      リソースフォーマットです.
//! @return     リソースフォーマットのビット数を返却します.
//-------------------------------------------------------------------------------------------------
uint32_t ToByte(RESOURCE_FORMAT format);

//-------------------------------------------------------------------------------------------------
//! @brief      ネイティブ形式に変換します.
//!
//! @param[in]      dimension       A3D形式です.
//! @return     ネイティブ形式に変換したリソースフラグを返却します.
//-------------------------------------------------------------------------------------------------
D3D11_COMPARISON_FUNC ToNativeComparisonFunc(COMPARE_OP operation);

//-------------------------------------------------------------------------------------------------
//! @brief      ネイティブ形式に変換します.
//!
//! @param[in]      topology        A3D形式です.
//! @param[in]      controlCount    制御パッチ数です.
//! @return     ネイティブ形式に変換したプリミティブトポロジーを返却します.
//-------------------------------------------------------------------------------------------------
D3D_PRIMITIVE_TOPOLOGY ToNativePrimitive(PRIMITIVE_TOPOLOGY topology, uint32_t controlCount);

//-------------------------------------------------------------------------------------------------
//! @brief      レンダーターゲットビューディメンションに変換します.
//!
//! @param[in]      value       A3D形式です.
//! @return     変換したレンダーターゲットビューディメンションを返却します.
//-------------------------------------------------------------------------------------------------
D3D11_RTV_DIMENSION ToNativeRTVDimension(VIEW_DIMENSION value);

//-------------------------------------------------------------------------------------------------
//! @brief      深度ステンシルビューディメンションに変換します.
//!
//! @param[in]      value       A3D形式です.
//! @return     変換した深度ステンシルビューディメンションを返却します.
//-------------------------------------------------------------------------------------------------
D3D11_DSV_DIMENSION ToNativeDSVDimension(VIEW_DIMENSION value);

//-------------------------------------------------------------------------------------------------
//! @brief      シェーダリソースビューに変換します.
//!
//! @param[in]      value       A3D形式です.
//! @return     変換したシェーダリソースビューディメンションを返却します.
//-------------------------------------------------------------------------------------------------
D3D11_SRV_DIMENSION ToNativeSRVDimension(VIEW_DIMENSION value);

//-------------------------------------------------------------------------------------------------
//! @brief      アンオーダードアクセスビューに変換します.
//!
//! @param[in]      value       A3D形式です.
//! @return     変換したアンオーダードアクセスビューディメンションを返却します.
//-------------------------------------------------------------------------------------------------
D3D11_UAV_DIMENSION ToNativeUAVDimension(VIEW_DIMENSION value);

//-------------------------------------------------------------------------------------------------
//! @brief      使用用途を取得します.
//!
//! @param[in]      type        ヒープタイプ.
//! @return     使用用途を返却します.
//-------------------------------------------------------------------------------------------------
D3D11_USAGE ToNativeUsage(HEAP_TYPE type);

//-------------------------------------------------------------------------------------------------
//! @brief      CPUアクセスフラグを取得します.
//!
//! @param[in]      prop        CPUページプロパティです.
//! @return     CPUアクセスフラグを返却します.
//-------------------------------------------------------------------------------------------------
uint32_t ToNativeCpuAccessFlags(HEAP_TYPE type, CPU_PAGE_PROPERTY prop);

//-------------------------------------------------------------------------------------------------
//! @brief      バインドフラグを取得します.
//-------------------------------------------------------------------------------------------------
uint32_t ToNativeBindFlags(uint32_t usage);

#if defined(A3D_FOR_WINDOWS10)
//-------------------------------------------------------------------------------------------------
//! @brief      カラースペースタイプを変換します.
//!
//! @param[in]      value       A3D形式です.
//! @return     変換したカラースペースタイプを返却します.
//-------------------------------------------------------------------------------------------------
DXGI_COLOR_SPACE_TYPE ToNativeColorSpace(COLOR_SPACE_TYPE value);
#endif//defined(A3D_FOR_WINDOWS10)

//-------------------------------------------------------------------------------------------------
//! @brief      サブリソース番号を計算します.
//!
//! @param[in]      mipSlice        ミップ番号.
//! @param[in]      arraySlice      配列番号.
//! @param[in]      planeSlice      平面スライス
//! @param[in]      mipLevels       ミップレベル数.
//! @param[in]      arraySize       配列サイズ.
//! @return     サブリソース番号を返却します.
//-------------------------------------------------------------------------------------------------
uint32_t CalcSubresource(
    uint32_t mipSlice,
    uint32_t arraySlice,
    uint32_t planeSlice,
    uint32_t mipLevels,
    uint32_t arraySize);

//-------------------------------------------------------------------------------------------------
//! @brief      サブリソース番号からミップ番号，配列番号を求めます.
//!
//! @param[in]      subresouce      サブリソース番号です.
//! @param[in]      mipLevels       ミップレベル数です.
//! @param[in]      arraySize       配列数です.
//! @param[out]     mipSlice        ミップ番号です.
//! @param[out]     arraySlice      配列番号です
//! @param[out]     planeSlice      平面スライスです.
//-------------------------------------------------------------------------------------------------
void DecomposeSubresource(
    uint32_t subresource,
    uint32_t mipLevels,
    uint32_t arraySize,
    uint32_t& mipSlice,
    uint32_t& arraySlice,
    uint32_t& planeSlice);

//-------------------------------------------------------------------------------------------------
//! @brief      サブリソースレイアウトを求めます.
//!
//! @param[in]      subresource     サブリソース番号です.
//! @param[in]      format          リソースフォーマットです.
//! @param[in]      width           横幅です.
//! @param[in]      height          縦幅です.
//! @param[in]      depth           奥行です.
//! @return     サブリソースレイアウトを返却します.
//-------------------------------------------------------------------------------------------------
SubresourceLayout CalcSubresourceLayout(
    uint32_t        subresource,
    RESOURCE_FORMAT format,
    uint32_t        width,
    uint32_t        height,
    uint32_t        depth
);


} // namespace a3d
