﻿//-------------------------------------------------------------------------------------------------
// File : rsHeap.h
// Desc : Memory Heap.
// Copyright(c) Project Asura. All right reserved.
//-------------------------------------------------------------------------------------------------
#pragma once

//-------------------------------------------------------------------------------------------------
// Includes
//-------------------------------------------------------------------------------------------------
#include <a3d.h>
#include <mutex>


namespace rs {

///////////////////////////////////////////////////////////////////////////////////////////////////
// Heap class
///////////////////////////////////////////////////////////////////////////////////////////////////
class Heap : public a3d::IAllocator
{
    //=============================================================================================
    // list of friend classes
    //=============================================================================================
    /* NOTHING */

public:
    struct Block;

    ///////////////////////////////////////////////////////////////////////////////////////////////
    // BlockHeader structure
    ///////////////////////////////////////////////////////////////////////////////////////////////
    struct BlockHeader
    {
        uint32_t    Signature;      //!< メモリサイン.
        uint64_t    Size;           //!< メモリブロックのサイズです(バイト単位).
        Block*      pNext;          //!< 次のメモリブロックへのポインタです.
        Block*      pPrev;          //!< 前のメモリブロックへのポインタです.
    };

    ///////////////////////////////////////////////////////////////////////////////////////////////
    // Block structure
    ///////////////////////////////////////////////////////////////////////////////////////////////
    struct Block
    {
        BlockHeader Header;     //!< ヘッダ情報です.
        uint8_t*    pBuffer;    //!< メモリブロックです.
    };

    ///////////////////////////////////////////////////////////////////////////////////////////////
    // Info structure
    ///////////////////////////////////////////////////////////////////////////////////////////////
    struct Info
    {
        uint64_t    Count;      //!< アロケートカウンターです.
        uint64_t    Size;       //!< メモリサイズです(バイト単位).
    };

    //=============================================================================================
    // public variables.
    //=============================================================================================
    /* NOTHING */

    //=============================================================================================
    // public methods.
    //=============================================================================================

    //---------------------------------------------------------------------------------------------
    //! @brief      コンストラクタです.
    //---------------------------------------------------------------------------------------------
    Heap();

    //---------------------------------------------------------------------------------------------
    //! @brief      デストラクタです.
    //---------------------------------------------------------------------------------------------
    ~Heap();

    //---------------------------------------------------------------------------------------------
    //! @brief      初期化処理を行います.
    //!
    //! @param[in]      size        確保するメモリサイズです(バイト単位).
    //! @retval true    初期化に成功.
    //! @retval false   初期化に失敗.
    //---------------------------------------------------------------------------------------------
    bool Init(uint64_t size);

    //---------------------------------------------------------------------------------------------
    //! @brief      終了処理を行います.
    //---------------------------------------------------------------------------------------------
    void Term();

    //---------------------------------------------------------------------------------------------
    //! @brief      メモリを確保します.
    //!
    //! @param[in]      size        確保するメモリサイズです.
    //! @param[in]      alignment   メモリアライメントです.
    //! @return     確保したメモリへのポインタを返却します. 確保に失敗した場合は nullptr が返却されます.
    //---------------------------------------------------------------------------------------------
    void* Alloc(size_t size, size_t alignment) noexcept override;

    //---------------------------------------------------------------------------------------------
    //! @brief      メモリを再確保します.
    //!
    //! @param[in]      ptr         確保済みメモリへのポインタ.
    //! @param[in]      size        確保するメモリサイズ.
    //! @param[in]      alignment   メモリアライメントです.
    //! @return     確保したメモリへのポインタを返却します. 確保に失敗した場合は nullptr が返却されます.
    //---------------------------------------------------------------------------------------------
    void* Realloc(void* ptr, size_t size, size_t alignment) noexcept override;

    //---------------------------------------------------------------------------------------------
    //! @brief      メモリを解放します.
    //!
    //! @param[in]      ptr     解放するメモリへのポインタ.
    //---------------------------------------------------------------------------------------------
    void Free(void* ptr) noexcept override;

    //---------------------------------------------------------------------------------------------
    //! @brief      メモリマーカーを取得します.
    //!
    //! @return     メモリマーカーを返却します.
    //---------------------------------------------------------------------------------------------
    Info GetMarker();

    //---------------------------------------------------------------------------------------------
    //! @brief      メモリリークをチェックします.
    //!
    //! @param[in]      marker      メモリマーカーです.
    //! @retval true    メモリリークがあります
    //! @retval false   メモリリークはありません.
    //---------------------------------------------------------------------------------------------
    bool IsLeak(const Info& maker);

    //---------------------------------------------------------------------------------------------
    //! @brief      メモリリークをチェックします.
    //!
    //! @param[in]      marker      メモリマーカーです.
    //! @param[out]     pLeakInfo   メモリリーク情報の格納先です.
    //! @retval true    メモリリークがあります.
    //! @retval false   メモリリークはありません.
    //---------------------------------------------------------------------------------------------
    bool IsLeak(const Info& marker, Info* pLeakInfo);

    //---------------------------------------------------------------------------------------------
    //! @brief      メモリがいっぱいであるかどうかチェックします.
    //!
    //! @retval true    使用可能なメモリ領域がありません.
    //! @retval false   使用可能なメモリ領域があります.
    //---------------------------------------------------------------------------------------------
    bool IsFull();

private:
    //=============================================================================================
    // private variables.
    //=============================================================================================
    uint64_t        m_Size;             //!< ヒープサイズです(バイト単位).
    uint64_t        m_BlockCount;       //!< メモリブロック数です.
    uint64_t        m_UsedSize;         //!< 使用済みメモリサイズです(バイト単位).
    uint64_t        m_Offset;           //!< メモリオフセットです.
    uint8_t*        m_pBuffer;          //!< ヒープメモリです.
    Block*          m_pHead;            //!< 先頭ブロックです.
    Block*          m_pTail;            //!< 末端ブロックです.
    std::mutex      m_Mutex;            //!< ロックオブジェクトです

    //=============================================================================================
    // private methods.
    //=============================================================================================

    //---------------------------------------------------------------------------------------------
    //! @brief      メモリブロックを確保します.
    //!
    //! @param[in]      size        確保するメモリサイズ.
    //! @return     確保したメモリへのポインタを返却します.
    //---------------------------------------------------------------------------------------------
    void* AllocBlock(uint64_t size);

    //---------------------------------------------------------------------------------------------
    //! @brief      メモリブロックを解放します.
    //!
    //! @param[in]      ptr         解放するメモリ.
    //---------------------------------------------------------------------------------------------
    void FreeBlock(void* ptr);

    //---------------------------------------------------------------------------------------------
    //! @brief      メモリアライメント処理を行います.
    //!
    //! @param[in]      ptr         アライメントするポインタ
    //! @param[in]      alignment   メモリアライメント.
    //! @return     メモリアライメントしたポインタを返却します.
    //---------------------------------------------------------------------------------------------
    void* MemoryAlign(void* ptr, size_t alignment);

    //---------------------------------------------------------------------------------------------
    //! @brief      逆メモリアライメント処理を行います.
    //!
    //! @param[in       ptr         逆メモリアライメントするポインタ.
    //! @return     逆メモリアライメントしたポインタを返却します.
    //---------------------------------------------------------------------------------------------
    void* MemoryUnalign(void* ptr);

    Heap             (const Heap&) = delete;
    Heap& operator = (const Heap&) = delete;
};

} // namespace rs
