/*-------------------------------------------------------------------------
 *
 * Shared Disk File EXclusiveness Control Program(SF-EX)
 *
 * sfex_stat.c --- Display lock status. This is a part of the SF-EX.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  
 * 02110-1301, USA.
 *
 * Copyright (c) 2007 NIPPON TELEGRAPH AND TELEPHONE CORPORATION
 *
 * $Id$
 *
 *-------------------------------------------------------------------------
 *
 * sfex_stat [-i <index>] <device>
 *
 * -i <index> --- The index is number of the resource that display the lock.
 * This number is specified by the integer of one or more. When two or more 
 * resources are exclusively controlled by one meta-data, this option is used. 
 * Default is 1.
 *
 * <device> --- This is file path which stored meta-data. It is usually 
 * expressed in "/dev/...", because it is partition on the shared disk.
 *
 * exit code --- 0 - Normal end. Own node is holding lock. 2 - Normal 
 * end. Own node does not hold a lock. 3 - Error occurs while processing 
 * it. The content of the error is displayed into stderr. 4 - The mistake 
 * is found in the command line parameter.
 *
 *-------------------------------------------------------------------------*/

#if HAVE_CONFIG_H
#  include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <errno.h>
#include <string.h>
#include <limits.h>
#if HAVE_UNISTD_H
#  include <unistd.h>
#endif

#include "sfex.h"
#include "lib.h"

const char *progname;
const char *nodename;

/*
 * usage --- display command line syntax
 *
 * display command line syntax. By the purpose, it can specify destination 
 * stream. stdout or stderr is set usually.
 *
 * dist --- destination stream of the command line syntax, such as stderr.
 *
 * retrun value --- void
 */
void usage(FILE *dist) {
  fprintf(dist, "usage: %s [-i <index>] <device>\n", progname);
}

/*
 * main --- main function
 *
 * entry point of sfex_stat command.
 *
 * exit code --- 0 - Normal end. Own node is holding lock. 2 - Normal 
 * end. Own node dose not hold a lock. 3 - Error occurs while processing 
 * it. The content of the error is displayed into stderr. 4 - The mistake 
 * is found in the command line parameter.
 */
int
main(int argc, char *argv[]) {
  sfex_controldata cdata;
  sfex_lockdata ldata;

  /* command line parameter */
  int index = 1;		/* default 1st lock */
  const char *device;

  /*
   * startup process
   */

  /* get a program name */
  progname = get_progname(argv[0]);

  /* read command line option */
  opterr = 0;
  while (1) {
    int c = getopt(argc, argv, "hi:");
    if (c == -1)
      break;
    switch (c) {
    case 'h':			/* help */
      usage(stdout);
      exit(0);
    case 'i':			/* -i <index> */
      {
	unsigned long l = strtoul(optarg, NULL, 10);
	if (l < SFEX_MIN_NUMLOCKS || l > SFEX_MAX_NUMLOCKS) {
	  fprintf(stderr,
		  "%s: ERROR: index %s is out of range or invalid. it must be integer value between %lu and %lu.\n",
		  progname, optarg,
		  (unsigned long)SFEX_MIN_NUMLOCKS,
		  (unsigned long)SFEX_MAX_NUMLOCKS);
	  exit(4);
	}
	index = l;
      }
      break;
    case '?':			/* error */
      usage(stderr);
      exit(4);
    }
  }

  /* check parameter except the option */
  if (optind >= argc) {
    fprintf(stderr, "%s: ERROR: no device specified.\n", progname);
    usage(stderr);
    exit(4);
  } else if (optind + 1 < argc) {
    fprintf(stderr, "%s: ERROR: too many arguments.\n", progname);
    usage(stderr);
    exit(4);
  }
  device = argv[optind];

  /*
   * main processes start 
   */

  /* get a node name */
  nodename = get_nodename();

  /* read control data */
  read_controldata(&cdata, device);

  /* check whether to exceed the number of lock data that the number
     of index has registered. */
  if (index > cdata.numlocks) {
    fprintf(stderr, "%s: ERROR: index %d is too large. %d locks are stored.\n",
	    progname, index, cdata.numlocks);
    exit(4);
  }

  /* read lock data */
  read_lockdata(&cdata, &ldata, device, index);

  /* display status */
  print_controldata(&cdata);
  print_lockdata(&ldata, index);

  /* check current lock status */
  if (ldata.status != SFEX_STATUS_LOCK || strcmp(ldata.nodename, nodename)) {
    fprintf(stdout, "%s: status is UNLOCKED.\n", progname);
    exit(2);
  } else {
    fprintf(stdout, "%s: status is LOCKED.\n", progname);
    exit(0);
  }
}
