/*-------------------------------------------------------------------------
 * 
 * Shared Disk File EXclusiveness Control Program(SF-EX)
 *
 * lib.c --- Libraries for other SF-EX modules.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  
 * 02110-1301, USA.
 *
 * Copyright (c) 2007 NIPPON TELEGRAPH AND TELEPHONE CORPORATION
 *
 * $Id$
 *
 *-------------------------------------------------------------------------*/

#if HAVE_CONFIG_H
#  include "config.h"
#endif

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#include <string.h>
#if HAVE_UNISTD_H
#  include <unistd.h>
#endif
#if HAVE_SYS_UTSNAME_H
#  include <sys/utsname.h>
#endif

#include "sfex.h"
#include "lib.h"

/*
 * get_progname --- a program name
 *
 * We get program name from directory path. It does not include delimiter 
 * characters. Return value is pointer that point string of program name. 
 * We assume delimiter is '/'.
 */
const char *
get_progname(const char *argv0) {
  char *p;

  p = strrchr(argv0, '/');
  if (p)
    return p + 1;
  else
    return argv0;
}

/*
 * get_nodename --- get a node name(hostname)
 *
 * We get a node name by using uname(2) and return pointer of it.
 * The error checks are done in this function. The caller does not have 
 * to check return value.
 */
char *
get_nodename(void) {
  struct utsname u;
  char *n;

  if (uname(&u)) {
    fprintf(stderr, "%s: ERROR: %s\n", progname, strerror(errno));
    exit(3);
  }
  if (strlen(u.nodename) > SFEX_MAX_NODENAME) {
    fprintf(stderr, "%s: ERROR: nodename %s is too long. must be less than %d byte.\n",
	    progname, u.nodename, SFEX_MAX_NODENAME);
    exit(3);
  }
  n = strdup(&u.nodename[0]);
  if (!n) {
    fprintf(stderr, "%s: ERROR: %s\n", progname, strerror(errno));
    exit(3);
  }
  return n;
}

/*
 * init_controldata --- initialize control data
 *
 * We initialize each member of sfex_controldata structure.
 */
void
init_controldata(sfex_controldata *cdata, size_t blocksize, int numlocks) {
  memcpy(cdata->magic, SFEX_MAGIC, sizeof(cdata->magic));
  cdata->version = SFEX_VERSION;
  cdata->revision = SFEX_REVISION;
  cdata->blocksize = blocksize;
  cdata->numlocks = numlocks;
}

/*
 * init_lockdata --- initialize lock data
 *
 * We initialize each member of sfex_lockdata structure.
 */
void
init_lockdata(sfex_lockdata *ldata) {
  ldata->status = SFEX_STATUS_UNLOCK;
  ldata->count = 0;
  ldata->nodename[0] = 0;
}

/*
 * write_controldata --- write control data into file
 *
 * We write sfex_controldata struct into file. We open a file with 
 * synchronization mode and write out control data.
 *
 * cdata --- pointer of control data
 *
 * device --- name of target file.
 */
void
write_controldata(const sfex_controldata *cdata, const char *device) {
  char *block;
  int fd;

  /* prepare buffer */
  block = (char *)malloc(cdata->blocksize);
  if (!block) {
    fprintf(stderr, "%s: ERROR: %s\n", progname, strerror(errno));
    exit(3);
  }

  /* We write control data into the buffer with given format. */
  /* We write the offset value of each field of the control data directly.
   * Because a point using this value is limited to two places, we do not 
   * use macro. If you change the following offset values, you must change 
   * values in the read_controldata() function.
   */
  memset(block, 0, cdata->blocksize);
  memcpy(block, cdata->magic, 4);
  sprintf(block + 4, "%d", cdata->version);
  sprintf(block + 8, "%d", cdata->revision);
  sprintf(block + 12, "%d", cdata->blocksize);
  sprintf(block + 20, "%d", cdata->numlocks);

  /*  file open */
  do {
    fd = open(device, O_WRONLY | O_DSYNC);
    if (fd == -1) {
      if (errno == EINTR || errno == EAGAIN)
	continue;
      fprintf(stderr, "%s: ERROR: can't open device %s: %s\n",
	      progname, device, strerror(errno));
      exit(3);
    }
    break;
  } while (1);

  /* write buffer into a file  */
  {
    ssize_t wrote_size = 0;
    do {
      ssize_t s = write(fd, block + wrote_size, cdata->blocksize - wrote_size);
      if (s == -1) {
	if (errno == EINTR || errno == EAGAIN)
	  continue;
	fprintf(stderr, "%s: ERROR: can't write meta-data: %s\n", progname, strerror(errno));
	exit(3);
      }
      wrote_size += s;
    } while (wrote_size < cdata->blocksize);
  }

  /* file close */
  close(fd);

  /* release buffer and exit */
  free(block);
}

/*
 * write_lockdata --- write lock data into file
 *
 * We write sfex_lockdata into file and seek file pointer to the given
 * position of lock data.
 *
 * cdata --- pointer for control data
 *
 * ldata --- pointer for lock data
 *
 * device --- file name for write
 *
 * index --- index number for lock data. 1 origine.
 */
void
write_lockdata(const sfex_controldata *cdata, const sfex_lockdata *ldata, const char *device, int index) {
  char *alloc, *block;
  int fd;

  /* prepare buffer ** */
  /* The buffer address is aligned by cdata->blocksize. This is in preparation 
     for the case to use direct I/O. */
  alloc = (char *)malloc(2 * cdata->blocksize);
  if (!alloc) {
    fprintf(stderr, "%s: ERROR: %s\n", progname, strerror(errno));
    exit(3);
  }
  block = (char *)((((long)alloc + cdata->blocksize - 1) / cdata->blocksize)
		   * cdata->blocksize);

  /* We write lock data into buffer with given format */
  /* We write the offset value of each field of the control data directly.
   * Because a point using this value is limited to two places, we do not 
   * use macro. If you chage the following offset values, you must change 
   * values in the read_lockdata() function.
   */
  memset(block, 0, cdata->blocksize);
  block[0] = ldata->status;
  sprintf(block + 1, "%d", ldata->count);
  strcpy(block + 5, ldata->nodename);

  /* file open */
  do {
    fd = open(device, O_WRONLY | SFEX_O_DIRECT);
    if (fd == -1) {
      if (errno == EINTR || errno == EAGAIN)
	continue;
      fprintf(stderr, "%s: ERROR: can't open device %s: %s\n",
	      progname, device, strerror(errno));
      exit(3);
    }
    break;
  } while (1);

  /* seek a file pointer to given position */
  if (lseek(fd, cdata->blocksize * index, SEEK_SET) == -1) {
    fprintf(stderr, "%s: ERROR: can't seek file pointer: %s\n", progname, strerror(errno));
    exit(3);
  }

  /* write buffer into file */
  do {
    ssize_t s = write(fd, block, cdata->blocksize);
    if (s == -1) {
      if (errno == EINTR || errno == EAGAIN)
	continue;
      fprintf(stderr, "%s: ERROR: can't write meta-data: %s\n", progname, strerror(errno));
      exit(3);
    } else if (s != cdata->blocksize) {
      /* if writing atomically failed, this process is error */
      fprintf(stderr, "%s: ERROR: can't write meta-data atomically.\n", progname);
      exit(3);
    }
    break;
  } while (1);

  /* file close */
  close(fd);

  /* release buffer and exit */
  free(alloc);
}

/*
 * read_controldata --- read control data from file
 *
 * read sfex_controldata structure from file.
 *
 * cdata --- pointer for control data. 
 *
 * device --- file name for reading
 */
void
read_controldata(sfex_controldata *cdata, const char *device) {
  char *block;
  int fd;

  /* prepare buffer */
  block = (char *)malloc(SFEX_CONTROLBLOCK_SIZE);
  if (!block) {
    fprintf(stderr, "%s: ERROR: %s\n", progname, strerror(errno));
    exit(3);
  }

  /* file open */
  do {
    fd = open(device, O_RDONLY | O_DSYNC | O_RSYNC);
    if (fd == -1) {
      if (errno == EINTR || errno == EAGAIN)
	continue;
      fprintf(stderr, "%s: ERROR: can't open device %s: %s\n",
	      progname, device, strerror(errno));
      exit(3);
    }
    break;
  } while (1);

  /* read data from file */
  {
    ssize_t read_size = 0;
    do {
      ssize_t s = read(fd, block + read_size, SFEX_CONTROLBLOCK_SIZE - read_size);
      if (s == -1) {
	if (errno == EINTR || errno == EAGAIN)
	  continue;
	fprintf(stderr, "%s: ERROR: can't read meta-data: %s\n", progname, strerror(errno));
	exit(3);
      } else if (s == 0)
	break;
      read_size += s;
    } while (read_size < SFEX_CONTROLBLOCK_SIZE);
    if (read_size < SFEX_CONTROLBLOCK_SIZE) {
      fprintf(stderr, "%s: ERROR: unexpected eof while reading meta-data\n", progname);
      exit(3);
    }
  }

  /* file close */
  close(fd);

  /* read control data from buffer */
  /* 1. check the magic number.  2. check null terminator of each field 
     3. check the version number.  4. Unmuch of revision number is allowed  */
  /* We write the offset value of each field of the control data directly.
   * Because a point using this value is limited to two places, we do not 
   * use macro. If you chage the following offset values, you must change 
   * values in the write_controldata() function.
   */
  memcpy(cdata->magic, block, 4);
  if (memcmp(cdata->magic, SFEX_MAGIC, sizeof(cdata->magic))) {
    fprintf(stderr, "%s: ERROR: magic number mismatched.\n", progname);
    exit(3);
  }
  if (block[7] || block[11] || block[19] || block[23]) {
    fprintf(stderr, "%s: ERROR: control data format error.\n", progname);
    exit(3);
  }
  cdata->version = atoi(block + 4);
  if (cdata->version != SFEX_VERSION) {
    fprintf(stderr, "%s: ERROR: version number mismatched. program is %d, data is %d.\n",
	    progname, SFEX_VERSION, cdata->version);
    exit(3);
  }
  cdata->revision = atoi(block + 8);
  cdata->blocksize = atoi(block + 12);
  cdata->numlocks = atoi(block + 20);

  /* release buffer and exit */
  free(block);
}

/*
 * read_lockdata --- read lock data from file
 *
 * read sfex_lockdata from file and seek file pointer to head position of the 
 * file.
 *
 * cdata --- pointer for control data
 *
 * ldata --- pointer for lock data. Read lock data are stored into this 
 * pointed area.
 *
 * device --- file name of source file
 *
 * index --- index number. 1 origin.
 */
void
read_lockdata(const sfex_controldata *cdata, sfex_lockdata *ldata, const char *device, int index) {
  char *alloc, *block;
  int fd;

  /* prepare buffer */
  /* The buffer address is aligned by cdata->blocksize. This is in preparation 
     for the case to use direct I/O. */
  alloc = (char *)malloc(2 * cdata->blocksize);
  if (!alloc) {
    fprintf(stderr, "%s: ERROR: %s\n", progname, strerror(errno));
    exit(3);
  }
  block = (char *)((((long)alloc + cdata->blocksize - 1) / cdata->blocksize)
		   * cdata->blocksize);

  /* file open */
  do {
    fd = open(device, O_RDONLY | SFEX_O_DIRECT);
    if (fd == -1) {
      if (errno == EINTR || errno == EAGAIN)
	continue;
      fprintf(stderr, "%s: ERROR: can't open device %s: %s\n",
	      progname, device, strerror(errno));
      exit(3);
    }
    break;
  } while (1);

  /* seek a file pointer to given position */
  if (lseek(fd, cdata->blocksize * index, SEEK_SET) == -1) {
    fprintf(stderr, "%s: ERROR: can't seek file pointer: %s\n", progname, strerror(errno));
    exit(3);
  }

  /* read from file */
  do {
    ssize_t s = read(fd, block, cdata->blocksize);
    if (s == -1) {
      if (errno == EINTR || errno == EAGAIN)
	continue;
      fprintf(stderr, "%s: ERROR: can't read meta-data: %s\n", progname, strerror(errno));
      exit(3);
    } else if (s != cdata->blocksize) {
      /* if writing atomically failed, this process is error */
      fprintf(stderr, "%s: ERROR: can't read meta-data atomically.\n", progname);
      exit(3);
    }
    break;
  } while (1);

  /* file close */
  close(fd);

  /* read control data form buffer */
  /* 1. check null terminator of each field 2. check the status */
  /* We write the offset value of each field of the control data directly.
   * Because a point using this value is limited to two places, we do not 
   * use macro. If you chage the following offset values, you must change 
   * values in the write_lockdata() function.
   */
  if (block[4] || block[260]) {
    fprintf(stderr, "%s: ERROR: lock data format error.\n", progname);
    exit(3);
  }
  ldata->status = block[0];
  if (ldata->status != SFEX_STATUS_UNLOCK && ldata->status != SFEX_STATUS_LOCK) {
    fprintf(stderr, "%s: ERROR: lock data format error.\n", progname);
    exit(3);
  }
  ldata->count = atoi(block + 1);
  strcpy(ldata->nodename, block + 5);

  /* release buffer and exit */
  free(alloc);
}

void print_controldata(const sfex_controldata *cdata) {
  printf("control data:\n");
  printf("  magic: 0x%02x, 0x%02x, 0x%02x, 0x%02x\n",
	 cdata->magic[0], cdata->magic[1], cdata->magic[2], cdata->magic[3]);
  printf("  version: %d\n", cdata->version);
  printf("  revision: %d\n", cdata->revision);
  printf("  blocksize: %d\n", (int)cdata->blocksize);
  printf("  numlocks: %d\n", cdata->numlocks);
}

void print_lockdata(const sfex_lockdata *ldata, int index) {
  printf("lock data #%d:\n", index);
  printf("  status: %s\n", ldata->status == SFEX_STATUS_UNLOCK ? "unlock" : "lock");
  printf("  count: %d\n", ldata->count);
  printf("  nodename: %s\n",ldata->nodename);
}
