/*-------------------------------------------------------------------------
 *
 * Shared Disk File EXclusiveness Control Program(SF-EX)
 *
 * sfex_lock.c --- Acquire a lock. This is a part of the SF-EX.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  
 * 02110-1301, USA.
 *
 * Copyright (c) 2007 NIPPON TELEGRAPH AND TELEPHONE CORPORATION
 *
 * $Id$
 *
 *-------------------------------------------------------------------------
 *
 * sfex_lock [-i <index>] [-c <collision_timeout>] [-t <lock_timeout>] <device>
 *
 * -i <index> --- The index is number of the resource that acquire the lock.
 * This number is specified by the integer of one or more. When two or more 
 * resources are exclusively controlled by one meta-data, this option is used. 
 * Default is 1.
 *
 * -c <collision_timeout> --- The waiting time to detect the collision of
 * the lock with other nodes is specified. Time that is very longer than 
 * "once synchronous read from device which stored meta-data + once 
 * synchronous write" is specified usually. Default is 1 second.
 * This value need not be changed by using this option usually.  because 
 * it is not thought to take one second or more to synchronous read and write.
 *
 * -t <lock_timeout> --- This specifies the validity term of lock. The 
 * unit is a second. This timer prevents the resource being locked for 
 * a long time when node crashes with the lock acquired. Therefore, 
 * the lock holding node must update lock data at intervals that are 
 * shorter than this timer. The sfex_update command is used for updating 
 * lock. Default is 60 seconds.
 *
 * <device> --- This is file path which stored meta-data. It is usually 
 * expressed in "/dev/...", because it is partition on the shared disk.
 *
 * exit code --- 0 - Acquire a lock from unlock status. 1 - Acquire a lock
 * from lock timeout status. 2 - Lock acquisition failed. 3 - Error occurs 
 * while processing it. The content of the error is displayed into 
 * stderr. 4 - The mistake is found in the command line parameter.
 *
 *-------------------------------------------------------------------------*/

#if HAVE_CONFIG_H
#  include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <errno.h>
#include <string.h>
#include <limits.h>
#if HAVE_UNISTD_H
#  include <unistd.h>
#endif

#include "sfex.h"
#include "lib.h"

const char *progname;
const char *nodename;

/*
 * usage --- display command line syntax
 *
 * display command line syntax. By the purpose, it can specify destination 
 * stream. stdout or stderr is set usually.
 *
 * dist --- destination stream of the command line syntax, such as stderr.
 *
 * return value --- void
 */
void usage(FILE *dist) {
  fprintf(dist, "usage: %s [-i <index>] [-c <collision_timeout>] [-t <lock_timeout>] <device>\n", progname);
}

/*
 * main --- main function
 *
 * entry point of sfex_lock command.
 *
 * exit code --- 0 - Acquire a lock from unlock status. 1 - Acquire a lock
 * from lock timeout status. 2 - Lock acquisition failed. 3 - Error occurs 
 * while processing it. The content of the error is displayed into 
 * stderr. 4 - The mistake is found in the command line parameter.
 */
int
main(int argc, char *argv[]) {
  sfex_controldata cdata;
  sfex_lockdata ldata;
  int is_clean = TRUE;

  /* command line parameter */
  int index = 1;		/* default 1st lock */
  time_t collision_timeout = 1;	/* default 1 sec */
  time_t lock_timeout = 60;	/* default 60 sec */
  const char *device;

  /*
   * startup process
   */

  /* get a program name */
  progname = get_progname(argv[0]);

  /* read command line option */
  opterr = 0;
  while (1) {
    int c = getopt(argc, argv, "hi:c:t:");
    if (c == -1)
      break;
    switch (c) {
    case 'h':			/* help*/
      usage(stdout);
      exit(0);
    case 'i':			/* -i <index> */
      {
	unsigned long l = strtoul(optarg, NULL, 10);
	if (l < SFEX_MIN_NUMLOCKS || l > SFEX_MAX_NUMLOCKS) {
	  fprintf(stderr,
		  "%s: ERROR: index %s is out of range or invalid. it must be integer value between %lu and %lu.\n",
		  progname, optarg,
		  (unsigned long)SFEX_MIN_NUMLOCKS,
		  (unsigned long)SFEX_MAX_NUMLOCKS);
	  exit(4);
	}
	index = l;
      }
      break;
    case 'c':			/* -c <collision_timeout> */
      {
	unsigned long l = strtoul(optarg, NULL, 10);
	if (l < 1 || l > INT_MAX) {
	  fprintf(stderr,
		  "%s: ERROR: collision_timeout %s is out of range or invalid. it must be integer value between %lu and %lu.\n",
		  progname, optarg,
		  (unsigned long)1,
		  (unsigned long)INT_MAX);
	  exit(4);
	}
	collision_timeout = l;
      }
      break;
    case 't':			/* -t <lock_timeout> */
      {
	unsigned long l = strtoul(optarg, NULL, 10);
	if (l < 1 || l > INT_MAX) {
	  fprintf(stderr,
		  "%s: ERROR: lock_timeout %s is out of range or invalid. it must be integer value between %lu and %lu.\n",
		  progname, optarg,
		  (unsigned long)1,
		  (unsigned long)INT_MAX);
	  exit(4);
	}
	lock_timeout = l;
      }
      break;
    case '?':			/* error */
      usage(stderr);
      exit(4);
    }
  }

  /* check parameter except the option */
  if (optind >= argc) {
    fprintf(stderr, "%s: ERROR: no device specified.\n", progname);
    usage(stderr);
    exit(4);
  } else if (optind + 1 < argc) {
    fprintf(stderr, "%s: ERROR: too many arguments.\n", progname);
    usage(stderr);
    exit(4);
  }
  device = argv[optind];

  /*
   * main process
   */

  /* get a node name */
  nodename = get_nodename();

  /* read control data */
  read_controldata(&cdata, device);

  /* check whether to exceed the number of lock data that the number
     of index has registered. */
  if (index > cdata.numlocks) {
    fprintf(stderr, "%s: ERROR: index %d is too large. %d locks are stored.\n",
	    progname, index, cdata.numlocks);
    exit(4);
  }

  /* read lock data */
  read_lockdata(&cdata, &ldata, device, index);

  /* check current lock status */
  /* It waits only at the lock_timeout seconds, check whether the lock is 
     updated between those time if it is locking. The lock with another node 
     become effective, and own node fails if updated. */
  if ((ldata.status == SFEX_STATUS_LOCK) && (strcmp(nodename, ldata.nodename))) {
    sfex_lockdata ldata_new;
    unsigned int t = lock_timeout;
    while (t > 0)
      t = sleep(t);
    read_lockdata(&cdata, &ldata_new, device, index);
    if (ldata.count != ldata_new.count) {
      fprintf(stdout, "can't acquire lock.\n");
      exit(2);
    }
    /* The lock acquisition is possible because it was not updated. */
    is_clean = FALSE;
  }

  /* reserve lock */
  ldata.status = SFEX_STATUS_LOCK;
  ldata.count = SFEX_NEXT_COUNT(ldata.count);
  strcpy(ldata.nodename, nodename);
  write_lockdata(&cdata, &ldata, device, index);

  /* detect the collision of lock */
  /* The collision occurs when two or more nodes do the reservation 
     processing of the lock at the same time. It waits for collision_timeout 
     seconds to detect this,and whether the superscription of lock data by 
     another node is done is checked. If the superscription was done by 
     another node, the lock acquisition with the own node is given up.  
  */
  {
    sfex_lockdata ldata_new;
    unsigned int t = collision_timeout;
    while (t > 0)
      t = sleep(t);
    read_lockdata(&cdata, &ldata_new, device, index);
    if (strcmp(ldata.nodename, ldata_new.nodename)) {
      fprintf(stdout, "can't acquire lock.\n");
      exit(2);
    }
  }

  /* extension of lock */
  /* Validly time of the lock is extended. It is because of spending at 
     the collision_timeout seconds to detect the collision. */
  ldata.count = SFEX_NEXT_COUNT(ldata.count);
  write_lockdata(&cdata, &ldata, device, index);

  /* lock acquisition completion */
  fprintf(stdout, "acquired lock successfully.\n");
  exit(is_clean ? 0 : 1);
}
