﻿/*
 * VsqFile.cs
 * Copyright (c) 2008-2009 kbinani
 *
 * This file is part of Boare.Lib.Vsq.
 *
 * Boare.Lib.Vsq is free software; you can redistribute it and/or
 * modify it under the terms of the BSD License.
 *
 * Boare.Lib.Vsq is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */
using System;
using System.Collections.Generic;
using System.IO;

namespace Boare.Lib.Vsq {

    [Serializable]
    public class VsqFile : IDisposable, ICloneable {
        protected List<VsqTrack> m_tracks;
        protected List<TempoTableEntry> m_tempo_table;
        protected List<TimeSigTableEntry> m_timesig_table;
        protected int m_tpq;
        //double tpq_sec = 480.0 * 1000000.0;
        protected ulong m_total_clocks = 0;
        protected int m_base_tempo;
        protected VsqMaster m_master;  // VsqMaster, VsqMixerは通常，最初の非Master Trackに記述されるが，可搬性のため，
        protected VsqMixer m_mixer;    // ここではVsqFileに直属するものとして取り扱う．
        protected int m_premeasure_clocks;

        static readonly byte[] _MTRK = new byte[] { 0x4d, 0x54, 0x72, 0x6b };
        static readonly byte[] _MTHD = new byte[] { 0x4d, 0x54, 0x68, 0x64 };
        static readonly byte[] _MASTER_TRACK = new byte[] { 0x4D, 0x61, 0x73, 0x74, 0x65, 0x72, 0x20, 0x54, 0x72, 0x61, 0x63, 0x6B, };
        static readonly string[] _CURVES = new string[] { "VEL", "BRE", "BRI", "CLE", "OPE", "GEN", "POR", "PIT", "PBS" };

        public VsqCommand Execute( VsqCommand command ) {
#if DEBUG
            Console.WriteLine( "VsqFile.Execute(VsqCommand)" );
            //for( int i = 1; i < Tracks.Count; i++ ){
            //    Console.WriteLine( "    EditedStart,EditedEnd=" + Tracks[i].EditedStart + "," + Tracks[i].EditedEnd );
            //}
            Console.WriteLine( "    type=" + command.Type );
#endif
            VsqCommandType type = command.Type;
            if ( type == VsqCommandType.ChangePreMeasure ) {
                #region ChangePreMeasure
                VsqCommand ret = VsqCommand.GCommandChangePreMeasure( Master.PreMeasure );
                int value = (int)command.Args[0];
                Master.PreMeasure = value;
                UpdateTimesigInfo();
                return ret;
                #endregion
            } else if ( type == VsqCommandType.AddTrack ) {
                #region AddTrack
#if DEBUG
                System.Diagnostics.Debug.WriteLine( "    AddTrack" );
#endif
                VsqTrack track = (VsqTrack)command.Args[0];
                VsqMixerEntry mixer = (VsqMixerEntry)command.Args[1];
                int position = (int)command.Args[2];
                VsqCommand ret = VsqCommand.GCommandDeleteTrack( position );
                if ( Tracks.Count <= 17 ) {
                    Tracks.Insert( position, (VsqTrack)track.Clone() );
                    Mixer.Slave.Add( (VsqMixerEntry)mixer.Clone() );
                    //Mixer.Tracks = Mixer.Slave.Count;
                    return ret;
                } else {
                    return null;
                }
                #endregion
            } else if ( type == VsqCommandType.DeleteTrack ) {
                #region DeleteTrack
                int track = (int)command.Args[0];
                VsqCommand ret = VsqCommand.GCommandAddTrack( Tracks[track], Mixer.Slave[track - 1], track );
                Tracks.RemoveAt( track );
                Mixer.Slave.RemoveAt( track - 1 );
                //Mixer.Tracks = Mixer.Slave.Count;
                UpdateTotalClocks();
                return ret;
                #endregion
            } else if ( type == VsqCommandType.UpdateTempo ) {
                #region UpdateTempo
                int clock = (int)command.Args[0];
                int tempo = (int)command.Args[1];
                int new_clock = (int)command.Args[2];

                int index = -1;
                for ( int i = 0; i < TempoTable.Count; i++ ) {
                    if ( TempoTable[i].Clock == clock ) {
                        index = i;
                        break;
                    }
                }
                VsqCommand ret = null;
                if ( index >= 0 ) {
                    if ( tempo <= 0 ) {
                        ret = VsqCommand.GCommandUpdateTempo( clock, clock, TempoTable[index].Tempo );
                        TempoTable.RemoveAt( index );
                    } else {
                        ret = VsqCommand.GCommandUpdateTempo( new_clock, clock, TempoTable[index].Tempo );
                        TempoTable[index].Tempo = tempo;
                        TempoTable[index].Clock = new_clock;
                    }
                } else {
                    ret = VsqCommand.GCommandUpdateTempo( clock, clock, -1 );
                    TempoTable.Add( new TempoTableEntry( new_clock, tempo, 0.0 ) );
                }
                UpdateTempoInfo();
                UpdateTotalClocks();

                // 編集領域を更新
                int affected_clock = Math.Min( clock, new_clock );
                for ( int i = 1; i < Tracks.Count; i++ ) {
                    if ( affected_clock < Tracks[i].EditedStart ) {
                        Tracks[i].EditedStart = affected_clock;
                    }
                    Tracks[i].EditedEnd = (int)m_total_clocks;
                }
                return ret;
                #endregion
            } else if ( type == VsqCommandType.UpdateTempoRange ) {
                #region UpdateTempoRange
                int[] clocks = (int[])command.Args[0];
                int[] tempos = (int[])command.Args[1];
                int[] new_clocks = (int[])command.Args[2];
                int[] new_tempos = new int[tempos.Length];
                int affected_clock = int.MaxValue;
                for ( int i = 0; i < clocks.Length; i++ ) {
                    int index = -1;
                    affected_clock = Math.Min( affected_clock, clocks[i] );
                    affected_clock = Math.Min( affected_clock, new_clocks[i] );
                    for ( int j = 0; j < TempoTable.Count; j++ ) {
                        if ( TempoTable[j].Clock == clocks[i] ) {
                            index = j;
                            break;
                        }
                    }
                    if ( index >= 0 ) {
                        new_tempos[i] = TempoTable[index].Tempo;
                        if ( tempos[i] <= 0 ) {
                            TempoTable.RemoveAt( index );
                        } else {
                            TempoTable[index].Tempo = tempos[i];
                            TempoTable[index].Clock = new_clocks[i];
                        }
                    } else {
                        new_tempos[i] = -1;
                        TempoTable.Add( new TempoTableEntry( new_clocks[i], tempos[i], 0.0 ) );
                    }
                }
                UpdateTempoInfo();
                UpdateTotalClocks();
                for ( int i = 1; i < Tracks.Count; i++ ) {
                    if ( affected_clock < Tracks[i].EditedStart ) {
                        Tracks[i].EditedStart = affected_clock;
                    }
                    Tracks[i].EditedEnd = (int)m_total_clocks;
                }
                return VsqCommand.GCommandUpdateTempoRange( new_clocks, clocks, new_tempos );
                #endregion
            } else if ( type == VsqCommandType.UpdateTimesig ) {
                #region UpdateTimesig
                int barcount = (int)command.Args[0];
                int numerator = (int)command.Args[1];
                int denominator = (int)command.Args[2];
                int new_barcount = (int)command.Args[3];
                int index = -1;
                for ( int i = 0; i < TimeSigTable.Count; i++ ) {
                    if ( barcount == TimeSigTable[i].BarCount ) {
                        index = i;
                        break;
                    }
                }
                VsqCommand ret = null;
                if ( index >= 0 ) {
                    if ( numerator <= 0 ) {
                        ret = VsqCommand.GCommandUpdateTimesig( barcount, barcount, TimeSigTable[index].Numerator, TimeSigTable[index].Denominator );
                        TimeSigTable.RemoveAt( index );
                    } else {
                        ret = VsqCommand.GCommandUpdateTimesig( new_barcount, barcount, TimeSigTable[index].Numerator, TimeSigTable[index].Denominator );
                        TimeSigTable[index].BarCount = new_barcount;
                        TimeSigTable[index].Numerator = numerator;
                        TimeSigTable[index].Denominator = denominator;
                    }
                } else {
                    ret = VsqCommand.GCommandUpdateTimesig( new_barcount, new_barcount, -1, -1 );
                    TimeSigTable.Add( new TimeSigTableEntry( 0, numerator, denominator, new_barcount ) );
                }
                UpdateTimesigInfo();
                UpdateTotalClocks();
                return ret;
                #endregion
            } else if ( type == VsqCommandType.UpdateTimesigRange ) {
                #region UpdateTimesigRange
                int[] barcounts = (int[])command.Args[0];
                int[] numerators = (int[])command.Args[1];
                int[] denominators = (int[])command.Args[2];
                int[] new_barcounts = (int[])command.Args[3];
                int[] new_numerators = new int[numerators.Length];
                int[] new_denominators = new int[denominators.Length];
                for ( int i = 0; i < barcounts.Length; i++ ) {
                    int index = -1;
                    for ( int j = 0; j < TimeSigTable.Count; j++ ) {
                        if ( TimeSigTable[j].BarCount == barcounts[i] ) {
                            index = j;
                            break;
                        }
                    }
                    if ( index >= 0 ) {
                        new_numerators[i] = TimeSigTable[index].Numerator;
                        new_denominators[i] = TimeSigTable[index].Denominator;
                        if ( numerators[i] <= 0 ) {
                            TimeSigTable.RemoveAt( index );
                        } else {
                            TimeSigTable[index].BarCount = new_barcounts[i];
                            TimeSigTable[index].Numerator = numerators[i];
                            TimeSigTable[index].Denominator = denominators[i];
                        }
                    } else {
                        new_numerators[i] = -1;
                        new_denominators[i] = -1;
                        TimeSigTable.Add( new TimeSigTableEntry( 0, numerators[i], denominators[i], new_barcounts[i] ) );
                    }
                }
                UpdateTimesigInfo();
                UpdateTotalClocks();
                return VsqCommand.GCommandUpdateTimesigRange( new_barcounts, barcounts, new_numerators, new_denominators );
                #endregion
            } else if ( type == VsqCommandType.Replace ) {
                #region Replace
                VsqFile vsq = (VsqFile)command.Args[0];
                VsqFile inv = (VsqFile)this.Clone();
                m_tracks.Clear();
                for ( int i = 0; i < vsq.m_tracks.Count; i++ ) {
                    m_tracks.Add( (VsqTrack)vsq.m_tracks[i].Clone() );
                }
                m_tempo_table.Clear();
                for ( int i = 0; i < vsq.m_tempo_table.Count; i++ ) {
                    m_tempo_table.Add( (TempoTableEntry)vsq.m_tempo_table[i].Clone() );
                }
                m_timesig_table.Clear();
                for ( int i = 0; i < vsq.m_timesig_table.Count; i++ ) {
                    m_timesig_table.Add( (TimeSigTableEntry)vsq.m_timesig_table[i].Clone() );
                }
                m_tpq = vsq.m_tpq;
                m_total_clocks = vsq.m_total_clocks;
                m_base_tempo = vsq.m_base_tempo;
                m_master = (VsqMaster)vsq.m_master.Clone();
                m_mixer = (VsqMixer)vsq.m_mixer.Clone();
                m_premeasure_clocks = vsq.m_premeasure_clocks;
                UpdateTotalClocks();
                return VsqCommand.GCommandReplace( inv );
                #endregion
            } else if ( type == VsqCommandType.EventAdd ) {
                #region TrackAddNote
                int track = (int)command.Args[0];
                VsqEvent item = (VsqEvent)command.Args[1];
                //int key = this.Tracks[track].GetNextId( 0 );
                //item.InternalID = key;
                Tracks[track].Events.Add( item );
                VsqCommand ret = VsqCommand.GCommandEventDelete( track, item.InternalID );
                //this.Tracks[track].Events.Sort();
                UpdateTotalClocks();
                if ( item.Clock < Tracks[track].EditedStart ) {
                    Tracks[track].EditedStart = item.Clock;
                }
                if ( Tracks[track].EditedEnd < item.Clock + item.ID.Length ) {
                    Tracks[track].EditedEnd = item.Clock + item.ID.Length;
                }
                return ret;
                #endregion
            } else if ( type == VsqCommandType.EventAddRange ) {
                #region TrackAddNoteRange
#if DEBUG
                Console.WriteLine( "    TrackAddNoteRange" );
#endif
                int track = (int)command.Args[0];
                VsqEvent[] items = (VsqEvent[])command.Args[1];
                List<int> inv_ids = new List<int>();
                int min_clock = (int)m_total_clocks;
                int max_clock = 0;
                for ( int i = 0; i < items.Length; i++ ) {
                    VsqEvent item = (VsqEvent)items[i].Clone();
                    min_clock = Math.Min( min_clock, item.Clock );
                    max_clock = Math.Max( max_clock, item.Clock + item.ID.Length );
                    //int key = Tracks[track].GetNextId( i );
                    //item.InternalID = key;
#if DEBUG
                    Console.Write( "        i=" + i + "; item.InternalID=" + item.InternalID );
#endif
                    Tracks[track].Events.Add( item );
                    inv_ids.Add( item.InternalID );
#if DEBUG
                    Console.WriteLine( " => " + item.InternalID );
#endif
                }
                //Tracks[track].Events.Sort();
                UpdateTotalClocks();
                if ( min_clock < Tracks[track].EditedStart ) {
                    Tracks[track].EditedStart = min_clock;
                }
                if ( Tracks[track].EditedEnd < max_clock ) {
                    Tracks[track].EditedEnd = max_clock;
                }
                return VsqCommand.GCommandEventDeleteRange( track, inv_ids.ToArray() );
                #endregion
            } else if ( type == VsqCommandType.EventDelete ) {
                #region TrackDeleteNote
                int internal_id = (int)command.Args[0];
                int track = (int)command.Args[1];
                VsqEvent[] original = new VsqEvent[1];
                foreach ( VsqEvent item in Tracks[track].Events ) {
                    if ( item.InternalID == internal_id ) {
                        original[0] = (VsqEvent)item.Clone();
                        break;
                    }
                }
                if ( original[0].Clock < Tracks[track].EditedStart ) {
                    Tracks[track].EditedStart = original[0].Clock;
                }
                if ( Tracks[track].EditedEnd < original[0].Clock + original[0].ID.Length ) {
                    Tracks[track].EditedEnd = original[0].Clock + original[0].ID.Length;
                }
                VsqCommand ret = VsqCommand.GCommandEventAddRange( track, original );
                for ( int i = 0; i < this.Tracks[track].Events.Count; i++ ) {
                    if ( this.Tracks[track].Events[i].InternalID == internal_id ) {
                        Tracks[track].Events.RemoveAt( i );
                        break;
                    }
                }
                UpdateTotalClocks();
                return ret;
                #endregion
            } else if ( type == VsqCommandType.EventDeleteRange ) {
                #region TrackDeleteNoteRange
                int[] internal_ids = (int[])command.Args[0];
                int track = (int)command.Args[1];
                List<VsqEvent> inv = new List<VsqEvent>();
                int min_clock = int.MaxValue;
                int max_clock = int.MinValue;
                for ( int j = 0; j < internal_ids.Length; j++ ) {
                    for ( int i = 0; i < Tracks[track].Events.Count; i++ ) {
                        if ( internal_ids[j] == Tracks[track].Events[i].InternalID ) {
                            inv.Add( (VsqEvent)Tracks[track].Events[i].Clone() );
                            min_clock = Math.Min( min_clock, Tracks[track].Events[i].Clock );
                            max_clock = Math.Max( max_clock, Tracks[track].Events[i].Clock + Tracks[track].Events[i].ID.Length );
                            Tracks[track].Events.RemoveAt( i );
                            break;
                        }
                    }
                }
                UpdateTotalClocks();
                Tracks[track].EditedStart = min_clock;
                Tracks[track].EditedEnd = max_clock;
                return VsqCommand.GCommandEventAddRange( track, inv.ToArray() );
                #endregion
            } else if ( type == VsqCommandType.EventChangeClock ) {
                #region TrackChangeClock
                int track = (int)command.Args[0];
                int internal_id = (int)command.Args[1];
                int value = (int)command.Args[2];
                foreach ( VsqEvent item in this.Tracks[track].Events ) {
                    if ( item.InternalID == internal_id ) {
                        VsqCommand ret = VsqCommand.GCommandEventChangeClock( track, internal_id, item.Clock );
                        int min = Math.Min( item.Clock, value );
                        int max = Math.Max( item.Clock + item.ID.Length, value + item.ID.Length );
                        Tracks[track].EditedStart = min;
                        Tracks[track].EditedEnd = max;
                        item.Clock = value;
                        //this.Tracks[track].Events.Sort();
                        UpdateTotalClocks();
                        return ret;
                    }
                }
                return null;
                #endregion
            } else if ( type == VsqCommandType.EventChangeLyric ) {
                #region TrackChangeLyric
                int track = (int)command.Args[0];
                int internal_id = (int)command.Args[1];
                string phrase = (string)command.Args[2];
                string phonetic_symbol = (string)command.Args[3];
                bool protect_symbol = (bool)command.Args[4];
                foreach ( VsqEvent item in this.Tracks[track].Events ) {
                    if ( item.InternalID == internal_id ) {
                        if ( item.ID.type == VsqIDType.Anote ) {
                            VsqCommand ret = VsqCommand.GCommandEventChangeLyric( track, internal_id, item.ID.LyricHandle.L0.Phrase, item.ID.LyricHandle.L0.PhoneticSymbol, item.ID.LyricHandle.L0.PhoneticSymbolProtected );
                            item.ID.LyricHandle.L0.Phrase = phrase;
                            item.ID.LyricHandle.L0.PhoneticSymbol = phonetic_symbol;
                            item.ID.LyricHandle.L0.PhoneticSymbolProtected = protect_symbol;
                            Tracks[track].EditedStart = item.Clock;
                            Tracks[track].EditedEnd = item.Clock + item.ID.Length;
                            UpdateTotalClocks();
                            return ret;
                        }
                    }
                }
                return null;
                #endregion
            } else if ( type == VsqCommandType.EventChangeNote ) {
                #region TrackChangeNote
                int track = (int)command.Args[0];
                int internal_id = (int)command.Args[1];
                int note = (int)command.Args[2];
                foreach ( VsqEvent item in this.Tracks[track].Events ) {
                    if ( item.InternalID == internal_id ) {
                        VsqCommand ret = VsqCommand.GCommandEventChangeNote( track, internal_id, item.ID.Note );
                        item.ID.Note = note;
                        UpdateTotalClocks();
                        Tracks[track].EditedStart = item.Clock;
                        Tracks[track].EditedEnd = item.Clock + item.ID.Length;
                        return ret;
                    }
                }
                return null;
                #endregion
            } else if ( type == VsqCommandType.EventChangeClockAndNote ) {
                #region TrackChangeClockAndNote
                int track = (int)command.Args[0];
                int internal_id = (int)command.Args[1];
                int clock = (int)command.Args[2];
                int note = (int)command.Args[3];
                foreach ( VsqEvent item in this.Tracks[track].Events ) {
                    if ( item.InternalID == internal_id ) {
                        VsqCommand ret = VsqCommand.GCommandEventChangeClockAndNote( track, internal_id, item.Clock, item.ID.Note );
                        int min = Math.Min( item.Clock, clock );
                        int max = Math.Max( item.Clock + item.ID.Length, clock + item.ID.Length );
                        Tracks[track].EditedStart = min;
                        Tracks[track].EditedEnd = max;
                        item.Clock = clock;
                        item.ID.Note = note;
                        //this.Tracks[track].Events.Sort();
                        UpdateTotalClocks();
                        return ret;
                    }
                }
                return null;
                #endregion
            } else if ( type == VsqCommandType.TrackEditCurve ) {
                #region TrackEditCurve
                int track = (int)command.Args[0];
                string curve = (string)command.Args[1];
                List<BPPair> com = (List<BPPair>)command.Args[2];

                VsqCommand inv = null;
                List<BPPair> edit = new List<BPPair>();
                if ( com != null ) {
                    if ( com.Count > 0 ) {
                        int start_clock = com[0].Clock;
                        int end_clock = com[0].Clock;
                        foreach ( BPPair item in com ) {
                            start_clock = Math.Min( start_clock, item.Clock );
                            end_clock = Math.Max( end_clock, item.Clock );
                        }
                        Tracks[track].EditedStart = start_clock;
                        Tracks[track].EditedEnd = end_clock;
                        int start_value = Tracks[track][curve][start_clock];
                        int end_value = Tracks[track][curve][end_clock];
                        foreach ( int clock in Tracks[track][curve].GetKeyClockEnumerator() ) {
                            if ( start_clock <= clock && clock <= end_clock ) {
                                edit.Add( new BPPair( clock, Tracks[track][curve][clock] ) );
                            }
                        }
                        bool start_found = false;
                        bool end_found = false;
                        for ( int i = 0; i < edit.Count; i++ ) {
                            if ( edit[i].Clock == start_clock ) {
                                start_found = true;
                                edit[i].Value = start_value;
                                if ( start_found && end_found ) {
                                    break;
                                }
                            }
                            if ( edit[i].Clock == end_clock ) {
                                end_found = true;
                                edit[i].Value = end_value;
                                if ( start_found && end_found ) {
                                    break;
                                }
                            }
                        }
                        if ( !start_found ) {
                            edit.Add( new BPPair( start_clock, start_value ) );
                        }
                        if ( !end_found ) {
                            edit.Add( new BPPair( end_clock, end_value ) );
                        }

                        // 並べ替え
                        edit.Sort();
                        inv = VsqCommand.GCommandTrackEditCurve( track, curve, edit );
                    } else if ( com.Count == 0 ) {
                        inv = VsqCommand.GCommandTrackEditCurve( track, curve, new List<BPPair>() );
                    }
                }

                UpdateTotalClocks();
                if ( com.Count == 0 ) {
                    return inv;
                } else if ( com.Count == 1 ) {
                    bool found = false;
                    foreach ( int clock in Tracks[track][curve].GetKeyClockEnumerator() ) {
                        if ( clock == com[0].Clock ) {
                            found = true;
                            Tracks[track][curve].Add( clock, com[0].Value );
                            break;
                        }
                    }
                    if ( !found ) {
                        Tracks[track][curve].Add( com[0].Clock, com[0].Value );
                    }
                } else {
                    int start_clock = com[0].Clock;
                    int end_clock = com[com.Count - 1].Clock;
                    bool removed = true;
                    while ( removed ) {
                        removed = false;
                        foreach ( int clock in Tracks[track][curve].GetKeyClockEnumerator() ) {
                            if ( start_clock <= clock && clock <= end_clock ) {
                                Tracks[track][curve].Remove( clock );
                                removed = true;
                                break;
                            }
                        }
                    }
                    foreach ( BPPair item in com ) {
                        Tracks[track][curve].Add( item.Clock, item.Value );
                    }
                }
                return inv;
                #endregion
            } else if ( type == VsqCommandType.TrackEditCurveRange ) {
                #region TrackEditCurveRange
                int track = (int)command.Args[0];
                string[] curves = (string[])command.Args[1];
                List<BPPair>[] coms = (List<BPPair>[])command.Args[2];
                List<BPPair>[] inv_coms = new List<BPPair>[curves.Length];
                VsqCommand inv = null;

                for ( int k = 0; k < curves.Length; k++ ) {
                    string curve = curves[k];
                    List<BPPair> com = coms[k];
                    //SortedList<int, int> list = Tracks[track][curve].List;
                    List<BPPair> edit = new List<BPPair>();
                    if ( com != null ) {
                        if ( com.Count > 0 ) {
                            int start_clock = com[0].Clock;
                            int end_clock = com[0].Clock;
                            foreach ( BPPair item in com ) {
                                start_clock = Math.Min( start_clock, item.Clock );
                                end_clock = Math.Max( end_clock, item.Clock );
                            }
                            Tracks[track].EditedStart = start_clock;
                            Tracks[track].EditedEnd = end_clock;
                            int start_value = Tracks[track][curve][start_clock];
                            int end_value = Tracks[track][curve][end_clock];
                            foreach ( int clock in Tracks[track][curve].GetKeyClockEnumerator() ) {
                                if ( start_clock <= clock && clock <= end_clock ) {
                                    edit.Add( new BPPair( clock, Tracks[track][curve][clock] ) );
                                }
                            }
                            bool start_found = false;
                            bool end_found = false;
                            for ( int i = 0; i < edit.Count; i++ ) {
                                if ( edit[i].Clock == start_clock ) {
                                    start_found = true;
                                    edit[i].Value = start_value;
                                    if ( start_found && end_found ) {
                                        break;
                                    }
                                }
                                if ( edit[i].Clock == end_clock ) {
                                    end_found = true;
                                    edit[i].Value = end_value;
                                    if ( start_found && end_found ) {
                                        break;
                                    }
                                }
                            }
                            if ( !start_found ) {
                                edit.Add( new BPPair( start_clock, start_value ) );
                            }
                            if ( !end_found ) {
                                edit.Add( new BPPair( end_clock, end_value ) );
                            }

                            // 並べ替え
                            edit.Sort();
                            inv_coms[k] = edit;
                            //inv = GCommandTrackEditCurve( track, curve, edit );
                        } else if ( com.Count == 0 ) {
                            //inv = GCommandTrackEditCurve( track, curve, new List<BPPair>() );
                            inv_coms[k] = new List<BPPair>();
                        }
                    }

                    UpdateTotalClocks();
                    if ( com.Count == 0 ) {
                        return inv;
                    } else if ( com.Count == 1 ) {
                        bool found = false;
                        foreach ( int clock in Tracks[track][curve].GetKeyClockEnumerator() ) {
                            if ( clock == com[0].Clock ) {
                                found = true;
                                Tracks[track][curve].Add( clock, com[0].Value );
                                break;
                            }
                        }
                        if ( !found ) {
                            Tracks[track][curve].Add( com[0].Clock, com[0].Value );
                        }
                    } else {
                        int start_clock = com[0].Clock;
                        int end_clock = com[com.Count - 1].Clock;
                        bool removed = true;
                        while ( removed ) {
                            removed = false;
                            foreach ( int clock in Tracks[track][curve].GetKeyClockEnumerator() ) {
                                if ( start_clock <= clock && clock <= end_clock ) {
                                    Tracks[track][curve].Remove( clock );
                                    removed = true;
                                    break;
                                }
                            }
                        }
                        foreach ( BPPair item in com ) {
                            Tracks[track][curve].Add( item.Clock, item.Value );
                        }
                    }
                }
                return VsqCommand.GCommandTrackEditCurveRange( track, curves, inv_coms );
                #endregion
            } else if ( type == VsqCommandType.EventChangeVelocity ) {
                #region TrackChangeVelocity
                int track = (int)command.Args[0];
                List<KeyValuePair<int, int>> veloc = (List<KeyValuePair<int, int>>)command.Args[1];
                List<KeyValuePair<int, int>> inv = new List<KeyValuePair<int, int>>();
                foreach ( VsqEvent ev in Tracks[track].Events ) {
                    foreach ( KeyValuePair<int, int> add in veloc ) {
                        if ( ev.InternalID == add.Key ) {
                            inv.Add( new KeyValuePair<int, int>( ev.InternalID, ev.ID.Dynamics ) );
                            ev.ID.Dynamics = add.Value;
                            Tracks[track].EditedStart = ev.Clock;
                            Tracks[track].EditedEnd = ev.Clock + ev.ID.Length;
                            break;
                        }
                    }
                }
                return VsqCommand.GCommandEventChangeVelocity( track, inv );
                #endregion
            } else if ( type == VsqCommandType.EventChangeLength ) {
                #region TrackChangeLength
                int track = (int)command.Args[0];
                int internal_id = (int)command.Args[1];
                int new_length = (int)command.Args[2];
                foreach ( VsqEvent item in this.Tracks[track].Events ) {
                    if ( item.InternalID == internal_id ) {
                        VsqCommand ret = VsqCommand.GCommandEventChangeLength( track, internal_id, item.ID.Length );
                        Tracks[track].EditedStart = item.Clock;
                        int max = Math.Max( item.Clock + item.ID.Length, item.Clock + new_length );
                        Tracks[track].EditedEnd = max;
                        item.ID.Length = new_length;
                        UpdateTotalClocks();
                        return ret;
                    }
                }
                return null;
                #endregion
            } else if ( type == VsqCommandType.EventChangeClockAndLength ) {
                #region TrackChangeClockAndLength
                int track = (int)command.Args[0];
                int internal_id = (int)command.Args[1];
                int new_clock = (int)command.Args[2];
                int new_length = (int)command.Args[3];
                foreach ( VsqEvent item in this.Tracks[track].Events ) {
                    if ( item.InternalID == internal_id ) {
                        VsqCommand ret = VsqCommand.GCommandEventChangeClockAndLength( track, internal_id, item.Clock, item.ID.Length );
                        int min = Math.Min( item.Clock, new_clock );
                        int max_length = Math.Max( item.ID.Length, new_length );
                        int max = Math.Max( item.Clock + max_length, new_clock + max_length );
                        Tracks[track].EditedStart = min;
                        Tracks[track].EditedEnd = max;
                        item.ID.Length = new_length;
                        item.Clock = new_clock;
                        UpdateTotalClocks();
                        return ret;
                    }
                }
                return null;
                #endregion
            } else if ( type == VsqCommandType.EventChangeIDContaints ) {
                #region TrackChangeIDContaints
                int track = (int)command.Args[0];
                int internal_id = (int)command.Args[1];
                VsqID value = (VsqID)command.Args[2];
                foreach ( VsqEvent item in Tracks[track].Events ) {
                    if ( item.InternalID == internal_id ) {
                        VsqCommand ret = VsqCommand.GCommandEventChangeIDContaints( track, internal_id, item.ID );
                        int max_length = Math.Max( item.ID.Length, value.Length );
                        Tracks[track].EditedStart = item.Clock;
                        Tracks[track].EditedEnd = item.Clock + max_length;
                        item.ID = (VsqID)value.Clone();
                        UpdateTotalClocks();
                        return ret;
                    }
                }
                return null;
                #endregion
            } else if ( type == VsqCommandType.EventChangeIDContaintsRange ) {
                #region TrackChangeIDContaintsRange
                int track = (int)command.Args[0];
                int[] internal_ids = (int[])command.Args[1];
                VsqID[] values = (VsqID[])command.Args[2];
                VsqID[] inv_values = new VsqID[values.Length];
                for ( int i = 0; i < internal_ids.Length; i++ ) {
                    foreach ( VsqEvent item in Tracks[track].Events ) {
                        if ( item.InternalID == internal_ids[i] ) {
                            inv_values[i] = (VsqID)item.ID.Clone();
                            int max_length = Math.Max( item.ID.Length, values[i].Length );
                            Tracks[track].EditedStart = item.Clock;
                            Tracks[track].EditedEnd = item.Clock + max_length;
                            item.ID = (VsqID)values[i].Clone();
                            break;
                        }
                    }
                }
                UpdateTotalClocks();
                return VsqCommand.GCommandEventChangeIDContaintsRange( track, internal_ids, inv_values );
                #endregion
            } else if ( type == VsqCommandType.EventChangeClockAndIDContaints ) {
                #region TrackChangeClockAndIDContaints
                int track = (int)command.Args[0];
                int internal_id = (int)command.Args[1];
                int new_clock = (int)command.Args[2];
                VsqID value = (VsqID)command.Args[3];
                foreach ( VsqEvent item in Tracks[track].Events ) {
                    if ( item.InternalID == internal_id ) {
                        VsqCommand ret = VsqCommand.GCommandEventChangeClockAndIDContaints( track, internal_id, item.Clock, item.ID );
                        int max_length = Math.Max( item.ID.Length, value.Length );
                        int min = Math.Min( item.Clock, new_clock );
                        int max = Math.Max( item.Clock + max_length, new_clock + max_length );
                        item.ID = (VsqID)value.Clone();
                        item.Clock = new_clock;
                        Tracks[track].EditedStart = min;
                        Tracks[track].EditedEnd = max;
                        UpdateTotalClocks();
                        return ret;
                    }
                }
                return null;
                #endregion
            } else if ( type == VsqCommandType.EventChangeClockAndIDContaintsRange ) {
                #region TrackChangeClockAndIDContaintsRange
                int track = (int)command.Args[0];
                int[] internal_ids = (int[])command.Args[1];
                int[] clocks = (int[])command.Args[2];
                VsqID[] values = (VsqID[])command.Args[3];
                List<VsqID> inv_id = new List<VsqID>();
                List<int> inv_clock = new List<int>();
                for ( int i = 0; i < internal_ids.Length; i++ ) {
                    foreach ( VsqEvent item in Tracks[track].Events ) {
                        if ( item.InternalID == internal_ids[i] ) {
                            inv_id.Add( (VsqID)item.ID.Clone() );
                            inv_clock.Add( item.Clock );
                            int max_length = Math.Max( item.ID.Length, values[i].Length );
                            int min = Math.Min( item.Clock, clocks[i] );
                            int max = Math.Max( item.Clock + max_length, clocks[i] + max_length );
                            Tracks[track].EditedStart = min;
                            Tracks[track].EditedEnd = max;
                            item.ID = (VsqID)values[i].Clone();
                            item.Clock = clocks[i];
                            break;
                        }
                    }
                }
                UpdateTotalClocks();
                return VsqCommand.GCommandEventChangeClockAndIDContaintsRange(
                    track,
                    internal_ids,
                    inv_clock.ToArray(),
                    inv_id.ToArray() );
#if DEBUG
                Console.WriteLine( "    TrackChangeClockAndIDContaintsRange" );
                Console.WriteLine( "    track=" + track );
                for ( int i = 0; i < internal_ids.Length; i++ ) {
                    Console.WriteLine( "    id=" + internal_ids[i] + "; clock=" + clocks[i] + "; ID=" + values[i].ToString() );
                }
#endif
                #endregion
            } else if ( type == VsqCommandType.TrackChangeName ) {
                #region TrackCangeName
                int track = (int)command.Args[0];
                string new_name = (string)command.Args[1];
                VsqCommand ret = VsqCommand.GCommandTrackChangeName( track, Tracks[track].Name );
                Tracks[track].Name = new_name;
                return ret;
                #endregion
            } else if ( type == VsqCommandType.TrackReplace ) {
                #region TrackReplace
                int track = (int)command.Args[0];
                VsqTrack item = (VsqTrack)command.Args[1];
                VsqCommand ret = VsqCommand.GCommandTrackReplace( track, (VsqTrack)Tracks[track].Clone() );
                Tracks[track] = item;
                UpdateTotalClocks();
                return ret;
                #endregion
            /*} else if ( type == VsqCommandType.TrackAddSinger ) {
                #region TrackAddSinger
                int track = (int)command.Args[0];
                VsqEvent item = (VsqEvent)command.Args[1];
                int id = Tracks[track].GetNextIdForAnotherEvent( 0 );
                item.InternalID = id;
                Tracks[track].AnotherEvents.Add( item );
                VsqCommand ret = GCommandTrackDeleteSinger( track, id );
                return ret;
                UpdateTotalClocks();
                #endregion
            } else if ( type == VsqCommandType.TrackDeleteSinger ) {
                #region TrackDeleteSinger
                int track = (int)command.Args[0];
                int id = (int)command.Args[1];
                for ( int i = 0; i < Tracks[track].AnotherEvents.Count; i++ ) {
                    if ( Tracks[track].AnotherEvents[i].InternalID == id ) {
                        VsqCommand ret = GCommandTrackAddSinger( track, Tracks[track].AnotherEvents[i] );
                        Tracks[track].AnotherEvents.RemoveAt( i );
                        return ret;
                    }
                }
                #endregion
            } else if ( type == VsqCommandType.TrackChangeSingerClock ) {
                #region TrackChangeSingerClock
                int track = (int)command.Args[0];
                int id = (int)command.Args[1];
                int clock = (int)command.Args[2];
                for ( int i = 0; i < Tracks[track].AnotherEvents.Count; i++ ) {
                    if ( Tracks[track].AnotherEvents[i].InternalID == id ) {
                        VsqCommand ret = GCommandTrackChangeSingerClock( track, id, Tracks[track].AnotherEvents[i].Clock );
                        Tracks[track].AnotherEvents[i].Clock = clock;
                        return ret;
                    }
                }
                #endregion
            } else if ( type == VsqCommandType.TrackChangeSinger ) {
                #region TrackChangeSinger
                int track = (int)command.Args[0];
                int id = (int)command.Args[1];
                VsqEvent item = (VsqEvent)command.Args[2];
                for ( int i = 0; i < Tracks[track].AnotherEvents.Count; i++ ) {
                    if ( Tracks[track].AnotherEvents[i].InternalID == id ) {
                        VsqCommand ret = GCommandTrackChangeSinger( track, id, Tracks[track].AnotherEvents[i] );
                        Tracks[track].AnotherEvents[i] = item;
                        return ret;
                    }
                }
                #endregion*/
            }

            return null;
        }

        public object Clone() {
            VsqFile ret = new VsqFile();
            ret.m_tracks = new List<VsqTrack>();
            for ( int i = 0; i < m_tracks.Count; i++ ) {
                ret.m_tracks.Add( (VsqTrack)m_tracks[i].Clone() );
            }
            ret.m_tempo_table = new List<TempoTableEntry>();
            for ( int i = 0; i < m_tempo_table.Count; i++ ) {
                ret.m_tempo_table.Add( (TempoTableEntry)m_tempo_table[i].Clone() );
            }
            ret.m_timesig_table = new List<TimeSigTableEntry>();
            for ( int i = 0; i < m_timesig_table.Count; i++ ) {
                ret.m_timesig_table.Add( (TimeSigTableEntry)m_timesig_table[i].Clone() );
            }
            ret.m_tpq = m_tpq;
            ret.m_total_clocks = m_total_clocks;
            ret.m_base_tempo = m_base_tempo;
            ret.m_master = (VsqMaster)m_master.Clone();
            ret.m_mixer = (VsqMixer)m_mixer.Clone();
            ret.m_premeasure_clocks = m_premeasure_clocks;
            return ret;
        }

        private VsqFile() {
            VsqUtil.LoadSingerConfigs();
        }
        
        public VsqMaster Master {
            get {
                return m_master;
            }
            set {
                m_master = value;
            }
        }

        public VsqMixer Mixer {
            get {
                return m_mixer;
            }
            set {
                m_mixer = value;
            }
        }

        /// <summary>
        /// 小節の区切りを順次返すEnumerator。
        /// </summary>
        /// <returns></returns>
        public IEnumerable<VsqBarLineType> GetEnumerator( int end_clock ) {
            int local_denominator;
            int local_numerator;
            int local_clock;
            int local_bar_count;
            int clock_step;
            for ( int i = 0; i < TimeSigTable.Count; i++ ) {
                local_denominator = TimeSigTable[i].Denominator;
                local_numerator = TimeSigTable[i].Numerator;
                local_clock = TimeSigTable[i].Clock;
                local_bar_count = TimeSigTable[i].BarCount;
                clock_step = 480 * 4 / local_denominator;
                int mod = clock_step * local_numerator;
                int bar_counter = local_bar_count - 1;
                int t_end = end_clock;
                if ( i + 1 < TimeSigTable.Count ) {
                    t_end = TimeSigTable[i + 1].Clock;
                }
                for ( int clock = local_clock; clock < t_end; clock += clock_step ) {
                    if ( (clock - local_clock) % mod == 0 ) {
                        bar_counter++;
                        yield return new VsqBarLineType( clock, true, local_denominator, local_numerator, bar_counter );
                    } else {
                        yield return new VsqBarLineType( clock, false, local_denominator, local_numerator, bar_counter );
                    }
                }
            }
            yield break;
        }

        /// <summary>
        /// 基本テンポ値を取得します
        /// </summary>
        public int BaseTempo {
            get {
                return m_base_tempo;
            }
        }

        /// <summary>
        /// プリメジャー値を取得または設定します。
        /// </summary>
        public int PreMeasure {
            get {
                //return Tracks[1].MetaText.master.PreMeasure;
                return m_master.PreMeasure;
            }
            /*set {
                Tracks[1].MetaText.master.PreMeasure = value;
            }*/
        }

        public int PreMeasureClocks {
            get {
                return m_premeasure_clocks;
            }
        }

        /// <summary>
        /// プリメジャーの長さ(クロック)を計算します。
        /// </summary>
        private int GetPreMeasureInClock() {
            int pre_measure = m_master.PreMeasure;
            int last_bar_count = TimeSigTable[0].BarCount;
            int last_clock = TimeSigTable[0].Clock;
            int last_denominator = TimeSigTable[0].Denominator;
            int last_numerator = TimeSigTable[0].Numerator;
            for ( int i = 1; i < TimeSigTable.Count; i++ ) {
                if ( TimeSigTable[i].BarCount >= pre_measure ) {
                    break;
                } else {
                    last_bar_count = TimeSigTable[i].BarCount;
                    last_clock = TimeSigTable[i].Clock;
                    last_denominator = TimeSigTable[i].Denominator;
                    last_numerator = TimeSigTable[i].Numerator;
                }
            }

            int remained = pre_measure - last_bar_count;//プリメジャーの終わりまでの残り小節数
            return last_clock + remained * last_numerator * 480 * 4 / last_denominator;
        }

        /// <summary>
        /// 指定したクロックにおける、clock=0からの演奏経過時間(sec)
        /// </summary>
        /// <param name="clock"></param>
        /// <returns></returns>
        public double SecFromClock( int clock ) {
            for ( int i = TempoTable.Count - 1; i >= 0; i-- ) {
                if ( TempoTable[i].Clock < clock ) {
                    double init = TempoTable[i].Time;
                    int dclock = clock - TempoTable[i].Clock;
                    double sec_per_clock1 = TempoTable[i].Tempo * 1e-6 / 480.0;
                    return init + dclock * sec_per_clock1;
                }
            }

            double sec_per_clock = m_base_tempo * 1e-6 / 480.0;
            return clock * sec_per_clock;
        }

        /// <summary>
        /// 指定した時刻における、クロックを取得します
        /// </summary>
        /// <param name="time"></param>
        /// <returns></returns>
        public float ClockFromSec( float time ) {
            // timeにおけるテンポを取得
            int tempo = m_base_tempo;
            float base_clock = 0;
            float base_time = 0f;
            if ( m_tempo_table.Count == 0 ) {
                tempo = m_base_tempo;
                base_clock = 0;
                base_time = 0f;
            } else if ( m_tempo_table.Count == 1 ) {
                tempo = m_tempo_table[0].Tempo;
                base_clock = m_tempo_table[0].Clock;
                base_time = (float)m_tempo_table[0].Time;
            } else {
                for ( int i = m_tempo_table.Count - 1; i >= 0; i-- ) {
                    if ( m_tempo_table[i].Time < time ) {
                        return m_tempo_table[i].Clock + (time - (float)m_tempo_table[i].Time) * m_tpq * 1000000.0f / (float)m_tempo_table[i].Tempo;
                    }
                }
            }
            float dt = time - base_time;
            return base_clock + dt * m_tpq * 1000000.0f / (float)tempo;
        }

        /// <summary>
        /// 指定したクロックにおける拍子を取得します
        /// </summary>
        /// <param name="clock"></param>
        /// <param name="numerator"></param>
        /// <param name="denominator"></param>
        public void GetTimesigAt( int clock, out int numerator, out int denominator ) {
            int index = 0;
            for ( int i = TimeSigTable.Count - 1; i >= 0; i-- ) {
                index = i;
                if ( TimeSigTable[i].Clock <= clock ) {
                    break;
                }
            }
            numerator = TimeSigTable[index].Numerator;
            denominator = TimeSigTable[index].Denominator;
        }

        /// <summary>
        /// 指定したクロックにおけるテンポを取得します。
        /// </summary>
        /// <param name="clock"></param>
        /// <returns></returns>
        public int GetTempoAt( int clock ) {
            int index = 0;
            for ( int i = TempoTable.Count - 1; i >= 0; i-- ) {
                index = i;
                if ( TempoTable[i].Clock <= clock ) {
                    break;
                }
            }
            return TempoTable[index].Tempo;
        }

        /// <summary>
        /// 指定した小節の開始クロックを調べます。ここで使用する小節数は、プリメジャーを考慮しない。即ち、曲頭の小節が0である。
        /// </summary>
        /// <param name="bar_count"></param>
        /// <returns></returns>
        public int GetClockFromBarCount( int bar_count ) {
            int index = 0;
            for ( int i = TimeSigTable.Count - 1; i >= 0; i-- ) {
                index = i;
                if ( TimeSigTable[i].BarCount <= bar_count ) {
                    break;
                }
            }
            int numerator = TimeSigTable[index].Numerator;
            int denominator = TimeSigTable[index].Denominator;
            int init_clock = TimeSigTable[index].Clock;
            int init_bar_count = TimeSigTable[index].BarCount;
            int clock_per_bar = numerator * 480 * 4 / denominator;
            return init_clock + (bar_count - init_bar_count) * clock_per_bar;
        }

        /// <summary>
        /// 指定したクロックが、曲頭から何小節目に属しているかを調べます。ここで使用する小節数は、プリメジャーを考慮しない。即ち、曲頭の小節が0である。
        /// </summary>
        /// <param name="clock"></param>
        /// <returns></returns>
        public int GetBarCountFromClock( int clock ) {
            int index = 0;
            for ( int i = TimeSigTable.Count - 1; i >= 0; i-- ) {
                index = i;
                if ( TimeSigTable[i].Clock <= clock ) {
                    break;
                }
            }
            int bar_count = 0;
            if ( index >= 0 ) {
                int last_clock = TimeSigTable[index].Clock;
                int t_bar_count = TimeSigTable[index].BarCount;
                int numerator = TimeSigTable[index].Numerator;
                int denominator = TimeSigTable[index].Denominator;
                int clock_per_bar = numerator * 480 * 4 / denominator;
                bar_count = t_bar_count + (clock - last_clock) / clock_per_bar;
            }
            return bar_count;
        }

        /// <summary>
        /// 曲の長さを取得します。(クロック(4分音符は480クロック))
        /// </summary>
        public ulong TotalClocks {
            get {
                return m_total_clocks;
            }
            set {
                m_total_clocks = value;
            }
        }

        public int TickPerQuarter {
            get {
                return m_tpq;
            }
        }

        public List<TempoTableEntry> TempoTable {
            get {
                return m_tempo_table;
            }
        }

        public List<TimeSigTableEntry> TimeSigTable {
            get {
                return m_timesig_table;
            }
        }

        public List<VsqTrack> Tracks {
            get {
                return m_tracks;
            }
        }

        public void Dispose() {
            if ( m_tracks != null ) {
                m_tracks.Clear();
            }
            if ( m_tempo_table != null ) {
                m_tempo_table.Clear();
            }
            m_tracks = null;
            m_tempo_table = null;
        }

        /// <summary>
        /// 空のvsqファイルを構築します
        /// </summary>
        /// <param name="pre_measure"></param>
        /// <param name="numerator"></param>
        /// <param name="denominator"></param>
        /// <param name="tempo"></param>
        public VsqFile( int default_singer_program_change, int pre_measure, int numerator, int denominator, int tempo ) {
            VsqUtil.LoadSingerConfigs();
            m_total_clocks = (ulong)(pre_measure * 480 * 4 / denominator * numerator);
            m_tpq = 480;

            m_tracks = new List<VsqTrack>();
            m_tracks.Add( new VsqTrack( tempo, numerator, denominator ) );
            string singer = "Miku";
            SingerConfig sc = VsqUtil.GetSingerInfo( default_singer_program_change );
            if ( sc != null ) {
                singer = sc.VOICENAME;
            }
            m_tracks.Add( new VsqTrack( "Voice1", singer ) );
            m_master = new VsqMaster( pre_measure );
#if DEBUG
            Console.WriteLine( "VsqFile.ctor()" );
#endif
            m_mixer = new VsqMixer( 0, 0, 0, 0 );
            m_mixer.Slave.Add( new VsqMixerEntry( 0, 0, 0, 0 ) );
//            m_mixer.Slave.Add( new VsqMixerEntry( 0, 0, 0, 0 ) );
            //m_mixer.Tracks = 1;
            m_timesig_table = new List<TimeSigTableEntry>();
            m_timesig_table.Add( new TimeSigTableEntry( 0, numerator, denominator, 0 ) );
            m_tempo_table = new List<TempoTableEntry>();
            m_tempo_table.Add( new TempoTableEntry( 0, tempo, 0.0 ) );
            m_base_tempo = tempo;
            m_premeasure_clocks = GetPreMeasureInClock();
        }

        /// <summary>
        /// vsqファイルからのコンストラクタ
        /// </summary>
        /// <param name="_fpath"></param>
        public VsqFile( string _fpath ) {
            VsqUtil.LoadSingerConfigs();
            m_tempo_table = new List<TempoTableEntry>();
            m_timesig_table = new List<TimeSigTableEntry>();
            m_tpq = 480;
            //tpq_sec = 480.0 * 1000000.0;

            // SMFをコンバートしたテキストファイルを作成
            using ( TextMemoryStream tms = new TextMemoryStream( FileAccess.ReadWrite ) ) {
                using ( SMFReader reader = new SMFReader( _fpath ) ) {
                    foreach ( string line in reader.Lines ) {
                        tms.WriteLine( line );
                    }
                }

                // テキストファイルから、トラックごとに切り分け
                tms.Rewind();
                m_tracks = new List<VsqTrack>();

                // ヘッダからトラック数、TPQを取得
                string[] spl = tms.ReadLine().Split( " ".ToCharArray() );
                int trackNum = int.Parse( spl[2] );
                m_tpq = int.Parse( spl[3] );
                //tpq_sec = TickPerQuarter * 1000000.0;
                for ( int i = 0; i < trackNum; i++ ) {
                    List<string> lines = new List<string>();
                    string line;
                    while ( (line = tms.ReadLine()) != null ) {
                        if ( line.StartsWith( "TrkEnd" ) ) {
                            break;
                        }
                        if ( line != "MTrk" ) {
                            lines.Add( line );
                        }
                    }
                    m_tracks.Add( new VsqTrack( lines ) );
#if DEBUG
                    /*using ( StreamWriter sw = new StreamWriter( "meta_text_" + i + ".txt" ) ) {
                        foreach ( string s in lines ) {
                            sw.WriteLine( s );
                        }
                    }*/
#endif
                }
            }

            m_master = (VsqMaster)Tracks[1].Master.Clone();
            m_mixer = (VsqMixer)Tracks[1].Mixer.Clone();
            Tracks[1].Master = null;
            Tracks[1].Mixer = null;

#if DEBUG
            System.Diagnostics.Debug.WriteLine( "VsqFile.ctor()" );
#endif
            int master_track = -1;
            for ( int i = 0; i < Tracks.Count; i++ ) {
#if DEBUG
                System.Diagnostics.Debug.WriteLine( "    m_tracks[i].Name=" + m_tracks[i].Name );
#endif
                if ( m_tracks[i].Name == "Master Track" ) {
                    master_track = i;
                    break;
                }
            }

            int prev_tempo;
            int prev_index;
            double prev_time;
            if ( master_track >= 0 ) {
                #region TempoListの作成
                // MIDI event リストの取得
                List<MidiEvent> midi_event = Tracks[master_track].GetTempoList();
                // とりあえずtempo_tableに格納
                prev_tempo = midi_event[0].intValue[0];
                m_base_tempo = prev_tempo;
                prev_index = 0;
                TempoTable.Add( new TempoTableEntry( midi_event[0].index, midi_event[0].intValue[0], 0.0 ) );
                double thistime;
                prev_time = 0.0;
                for ( int j = 1; j < midi_event.Count; j++ ) {
                    int current_tempo = midi_event[j].intValue[0];
                    int current_index = midi_event[j].index;
                    thistime = prev_time + (double)(prev_tempo) * (double)(current_index - prev_index) / (m_tpq * 1000000.0);
                    TempoTable.Add( new TempoTableEntry( current_index, current_tempo, thistime ) );
                    prev_tempo = current_tempo;
                    prev_index = current_index;
                    prev_time = thistime;
                }
                TempoTable.Sort();
                #endregion

                #region TimeSigTableの作成
                List<MidiEvent> time_sigs = Tracks[master_track].GetTimeSigList();
                TimeSigTable.Add( new TimeSigTableEntry( 0, time_sigs[0].intValue[0], time_sigs[0].intValue[1], 0 ) );
                for ( int j = 1; j < time_sigs.Count; j++ ) {
                    int numerator = TimeSigTable[j - 1].Numerator;
                    int denominator = TimeSigTable[j - 1].Denominator;
                    int clock = TimeSigTable[j - 1].Clock;
                    int bar_count = TimeSigTable[j - 1].BarCount;

                    int dif = 480 * 4 / denominator * numerator;//1小節が何クロックか？
                    bar_count += (time_sigs[j].index - clock) / dif;
                    TimeSigTable.Add( new TimeSigTableEntry( time_sigs[j].index, time_sigs[j].intValue[0], time_sigs[j].intValue[1], bar_count ) );
                }
                #endregion
            }

            // 曲の長さを計算
            UpdateTempoInfo();
            UpdateTimesigInfo();
            m_premeasure_clocks = GetPreMeasureInClock();
            UpdateTotalClocks();
#if DEBUG
            System.Diagnostics.Debug.WriteLine( "    m_total_clocks=" + m_total_clocks );
#endif
        }

        /// <summary>
        /// TimeSigTableの[*].Clockの部分を更新します
        /// </summary>
        public void UpdateTimesigInfo() {
#if DEBUG
            Console.WriteLine( "VsqFile.UpdateTimesigInfo()" );
#endif
            if ( TimeSigTable[0].Clock != 0 ) {
                throw new ApplicationException( "initial timesig does not found" );
            }
            TimeSigTable[0].Clock = 0;
            TimeSigTable.Sort();
            for ( int j = 1; j < TimeSigTable.Count; j++ ) {
                int numerator = TimeSigTable[j - 1].Numerator;
                int denominator = TimeSigTable[j - 1].Denominator;
                int clock = TimeSigTable[j - 1].Clock;
                int bar_count = TimeSigTable[j - 1].BarCount;
                int dif = 480 * 4 / denominator * numerator;//1小節が何クロックか？
                clock += (TimeSigTable[j].BarCount - bar_count) * dif;
                TimeSigTable[j].Clock = clock;
            }
            m_premeasure_clocks = GetPreMeasureInClock();
#if DEBUG
            for ( int i = 0; i < TimeSigTable.Count; i++ ) {
                Console.WriteLine( "    clock=" + TimeSigTable[i].Clock + "; bar_count=" + TimeSigTable[i].BarCount + "; " + TimeSigTable[i].Numerator + "/" + TimeSigTable[i].Denominator );
            }
#endif
        }

        /// <summary>
        /// TempoTableの[*].Timeの部分を更新します
        /// </summary>
        public void UpdateTempoInfo() {
            if ( TempoTable.Count == 0 ) {
                TempoTable.Add( new TempoTableEntry( 0, BaseTempo, 0.0 ) );
            }
            TempoTable.Sort();
            if ( TempoTable[0].Clock != 0 ) {
                TempoTable[0].Time = (double)BaseTempo * (double)TempoTable[0].Clock / (TickPerQuarter * 1000000.0);
            } else {
                TempoTable[0].Time = 0.0;
            }
            double prev_time = TempoTable[0].Time;
            int prev_clock = TempoTable[0].Clock;
            int prev_tempo = TempoTable[0].Tempo;
            double inv_tpq_sec = 1.0 / (TickPerQuarter * 1000000.0);
            for ( int i = 1; i < TempoTable.Count; i++ ) {
                TempoTable[i].Time = prev_time + (double)prev_tempo * (double)(TempoTable[i].Clock - prev_clock) * inv_tpq_sec;
                prev_time = TempoTable[i].Time;
                prev_tempo = TempoTable[i].Tempo;
                prev_clock = TempoTable[i].Clock;
            }
        }

        /// <summary>
        /// VsqFile.Executeの実行直後などに、m_total_clocksの値を更新する
        /// </summary>
        public void UpdateTotalClocks() {
            ulong max = (ulong)m_premeasure_clocks;
            for( int i = 1; i < Tracks.Count; i++ ){
                VsqTrack track = Tracks[i];
                foreach ( VsqEvent ve in track.Events ) {
                    max = Math.Max( max, (ulong)(ve.Clock + ve.ID.Length) );
                }
                foreach ( string vct in _CURVES ) {
                    if ( vct == "VEL" ) {
                        continue;
                    }
                    if ( track[vct].Count > 0 ) {
                        int keys = track[vct].Count;
                        ulong last_key = (ulong)track[vct].Keys[keys - 1];
                        max = Math.Max( max, last_key );
                    }
                }
            }
            m_total_clocks = max;
        }

        /// <summary>
        /// 曲の長さを取得する。(sec)
        /// </summary>
        public double GetTotalSec() {
            return SecFromClock( (int)m_total_clocks );
        }

        /// <summary>
        /// 指定された番号のトラックに含まれる歌詞を指定されたファイルに出力します
        /// </summary>
        /// <param name="track"></param>
        /// <param name="fpath"></param>
        public void PrintLyricTable( int track, string fpath ) {
            using ( StreamWriter sw = new StreamWriter( fpath ) ) {
                for ( int i = 0; i < Tracks[track].Events.Count; i++ ) {
                    int Length;
                    // timesignal
                    int time_signal = Tracks[track].Events[i].Clock;
                    // イベントで指定されたIDがLyricであった場合
                    if ( Tracks[track].Events[i].ID.type == VsqIDType.Anote ) {
                        // 発音長を取得
                        Length = Tracks[track].Events[i].ID.Length;

                        // tempo_tableから、発音開始時のtempoを取得
                        int last = TempoTable.Count - 1;
                        int tempo = TempoTable[last].Tempo;
                        int prev_index = TempoTable[last].Clock;
                        double prev_time = TempoTable[last].Time;
                        for ( int j = 1; j < TempoTable.Count; j++ ) {
                            if ( TempoTable[j].Clock > time_signal ) {
                                tempo = TempoTable[j - 1].Tempo;
                                prev_index = TempoTable[j - 1].Clock;
                                prev_time = TempoTable[j - 1].Time;
                                break;
                            }
                        }
                        int current_index = Tracks[track].Events[i].Clock;
                        double start_time = prev_time + (double)(current_index - prev_index) * (double)tempo / (m_tpq * 1000000.0);
                        // TODO: 単純に + Lengthしただけではまずいはず。要検討
                        double end_time = start_time + ((double)Length) * ((double)tempo) / (m_tpq * 1000000.0);
                        sw.WriteLine( Tracks[track].Events[i].Clock + "," +
                                      start_time.ToString( "0.000000" ) + "," +
                                      end_time.ToString( "0.000000" ) + "," +
                                      Tracks[track].Events[i].ID.LyricHandle.L0.Phrase + "," +
                                      Tracks[track].Events[i].ID.LyricHandle.L0.PhoneticSymbol );
                    }

                }
            }
        }

        private void PrintTrack( VsqTrack item, FileStream fs, int msPreSend ) {
#if DEBUG
            Console.WriteLine( "PrintTrack" );
#endif
            //VsqTrack item = Tracks[track];
            string _NL = "" + (char)0x0a;
            //ヘッダ
            fs.Write( _MTRK, 0, 4 );
            //データ長。とりあえず0
            fs.Write( new byte[] { 0x00, 0x00, 0x00, 0x00 }, 0, 4 );
            long first_position = fs.Position;
            //トラック名
            WriteFlexibleLengthUnsignedLong( fs, 0x00 );//デルタタイム
            fs.WriteByte( 0xff );//ステータスタイプ
            fs.WriteByte( 0x03 );//イベントタイプSequence/Track Name
            char[] seq_name = Lyric.encode( item.Name );
            WriteFlexibleLengthUnsignedLong( fs, (ulong)seq_name.Length );//seq_nameの文字数
            WriteCharArray( fs, seq_name );
            //Meta Textを準備
            using ( TextMemoryStream sr = new TextMemoryStream( FileAccess.ReadWrite ) ) {
                item.PrintMetaText( sr, false, m_total_clocks + (ulong)120, GetPreMeasureInClock() );
                sr.Rewind();
                int line_count = -1;
                string tmp = "";
                if ( sr.Peek() >= 0 ) {
                    tmp = sr.ReadLine();
                    char[] line_char;
                    string line = "";
                    while ( sr.Peek() >= 0 ) {
                        line = sr.ReadLine();
                        tmp += _NL + line;
                        while ( (tmp + GetLinePrefix( line_count )).Length >= 127 ) {
                            line_count++;
                            tmp = GetLinePrefix( line_count ) + tmp;
                            string work = tmp.Substring( 0, 127 );
                            tmp = tmp.Substring( 127 );
                            line_char = work.ToCharArray();
                            WriteFlexibleLengthUnsignedLong( fs, 0x00 );
                            fs.WriteByte( 0xff );//ステータスタイプ
                            fs.WriteByte( 0x01 );//イベントタイプMeta Text
                            WriteFlexibleLengthUnsignedLong( fs, (ulong)line_char.Length );//データ長
                            WriteCharArray( fs, line_char );//メタテキスト本体
                        }
                    }
                    // 残りを出力
                    line_count++;
                    tmp = GetLinePrefix( line_count ) + tmp + _NL;
                    while ( tmp.Length > 127 ) {
                        string work = tmp.Substring( 0, 127 );
                        tmp = tmp.Substring( 127 );
                        line_char = work.ToCharArray();
                        WriteFlexibleLengthUnsignedLong( fs, 0x00 );
                        fs.WriteByte( 0xff );
                        fs.WriteByte( 0x01 );
                        WriteFlexibleLengthUnsignedLong( fs, (ulong)line_char.Length );
                        WriteCharArray( fs, line_char );
                        line_count++;
                        tmp = GetLinePrefix( line_count );
                    }
                    line_char = tmp.ToCharArray();
                    WriteFlexibleLengthUnsignedLong( fs, 0x00 );
                    fs.WriteByte( 0xff );
                    fs.WriteByte( 0x01 );
                    WriteFlexibleLengthUnsignedLong( fs, (ulong)line_char.Length );
                    WriteCharArray( fs, line_char );
                }
            }

            int last = 0;
            VsqNrpn[] data = GenerateNRPN( item, msPreSend );
#if DEBUG
            for ( int i = 0; i < data.Length; i++ ) {
                Console.WriteLine( "    " + data[i].Clock + " 0x" + Convert.ToString( data[i].Nrpn, 16 ) + " 0x" + Convert.ToString( data[i].DataMsb, 16 ) + " 0x" + Convert.ToString( data[i].DataLsb ) );
            }
#endif
            NrpnData[] nrpns = VsqNrpn.Convert( data );
            for ( int i = 0; i < nrpns.Length; i++ ) {
                WriteFlexibleLengthUnsignedLong( fs, (ulong)(nrpns[i].Clock - last) );
                fs.WriteByte( 0xb0 );
                fs.WriteByte( nrpns[i].Parameter );
                fs.WriteByte( nrpns[i].Value );
                last = nrpns[i].Clock;
            }

            //トラックエンド
            VsqEvent last_event = item.Events[item.Events.Count - 1];
            int last_clock = last_event.Clock + last_event.ID.Length;
            WriteFlexibleLengthUnsignedLong( fs, (ulong)last_clock );
            fs.WriteByte( 0xff );
            fs.WriteByte( 0x2f );
            fs.WriteByte( 0x00 );
            long pos = fs.Position;
            fs.Seek( first_position - 4, SeekOrigin.Begin );
            WriteUnsignedInt( fs, (uint)(pos - first_position) );
            fs.Seek( pos, SeekOrigin.Begin );
        }

        /// <summary>
        /// 指定したクロックにおけるプリセンド・クロックを取得します
        /// </summary>
        /// <param name="clock"></param>
        /// <param name="msPreSend"></param>
        /// <returns></returns>
        public int GetPresendClockAt( int clock, int msPreSend ) {
            double clock_msec = SecFromClock( clock ) * 1000.0;
            float draft_clock_sec = (float)(clock_msec - msPreSend) / 1000.0f;
            int draft_clock = (int)Math.Floor( ClockFromSec( draft_clock_sec ) );
            return clock - draft_clock;
        }

        /// <summary>
        /// 指定したトラックから、Expression(DYN)のNRPNリストを作成します
        /// </summary>
        /// <param name="track"></param>
        /// <param name="msPreSend"></param>
        /// <returns></returns>
        public VsqNrpn[] GenerateExpressionNRPN( VsqTrack track, int msPreSend ) {
            return GenerateExpressionNRPN( track, msPreSend, 0, (int)m_total_clocks, 0 );
        }

        public VsqNrpn[] GenerateExpressionNRPN( VsqTrack track, int msPreSend, int clStart, int clEnd, int t_temp_premeasure ) {
            List<VsqNrpn> ret = new List<VsqNrpn>();
            //SortedList<int, int> list = track[VsqCurveType.DYN].List;
            int count = track["DYN"].Count;
            byte delay0, delay1;
            GetMsbAndLsb( (ushort)msPreSend, out delay0, out delay1 );
            int start_clock = clStart < m_premeasure_clocks ? m_premeasure_clocks : clStart;
            int start_value = track["DYN"][start_clock];
            VsqNrpn add0 = new VsqNrpn( start_clock - GetPresendClockAt( start_clock, msPreSend ) - clStart + t_temp_premeasure, VsqNrpn.NRPN_CC_E_VESION_AND_DEVICE, 0x00, 0x00 );
            add0.Append( VsqNrpn.NRPN_CC_E_DELAY, delay0, delay1 );
            add0.Append( VsqNrpn.NRPN_CC_E_EXPRESSION, (byte)start_value );
            ret.Add( add0 );
            int[] keys = track["DYN"].Keys;
            for ( int i = 1; i < keys.Length; i++ ) {
                if ( keys[i] > clEnd ) {
                    break;
                }
                if ( keys[i] > start_clock ) {
                    VsqNrpn add = new VsqNrpn( keys[i] - GetPresendClockAt( keys[i], msPreSend ) - clStart + t_temp_premeasure, VsqNrpn.NRPN_CC_E_EXPRESSION, (byte)track["DYN"][keys[i]] );
                    ret.Add( add );
                }
            }
            return ret.ToArray();
        }

        /// <summary>
        /// 先頭に記録されるNRPNを作成します
        /// </summary>
        /// <returns></returns>
        public static VsqNrpn GenerateHeaderNRPN() {
            VsqNrpn ret = new VsqNrpn( 0, VsqNrpn.NRPN_CC_BS_VERSION_AND_DEVICE, 0x00, 0x00 );
            ret.Append( VsqNrpn.NRPN_CC_BS_DELAY, 0x00, 0x00 );
            ret.Append( VsqNrpn.NRPN_CC_BS_LANGUAGE_TYPE, 0x00 );
            return ret;
        }

        public VsqNrpn[] GenerateSingerNRPN( VsqEvent ve, int msPreSend, int clShift ) {
            int clock = ve.Clock;

            double clock_msec = SecFromClock( clock ) * 1000.0;

            int ttempo = GetTempoAt( clock );
            double tempo = 6e7 / ttempo;
            //double sStart = SecFromClock( ve.Clock );
            double msEnd = SecFromClock( ve.Clock + ve.ID.Length ) * 1000.0;
            ushort duration = (ushort)Math.Ceiling( msEnd - clock_msec );
#if DEBUG
            Console.WriteLine( "GenerateNoteNRPN" );
            Console.WriteLine( "    duration=" + duration );
#endif
            byte duration0, duration1;
            GetMsbAndLsb( duration, out duration0, out duration1 );
            byte delay0, delay1;
            GetMsbAndLsb( (ushort)msPreSend, out delay0, out delay1 );
            List<VsqNrpn> ret = new List<VsqNrpn>();

            int i = clock - GetPresendClockAt( clock, msPreSend ) - clShift;
            VsqNrpn add = new VsqNrpn( i, VsqNrpn.NRPN_CC_BS_VERSION_AND_DEVICE, 0x00, 0x00 );
            add.Append( VsqNrpn.NRPN_CC_BS_DELAY, delay0, delay1 );
            add.Append( VsqNrpn.NRPN_CC_BS_LANGUAGE_TYPE, (byte)ve.ID.IconHandle.Language );
            add.Append( VsqNrpn.NRPN_PC_VERSION_AND_DEVICE, 0x00, 0x00 );
            add.Append( VsqNrpn.NRPN_PC_VOICE_TYPE, (byte)ve.ID.IconHandle.Program );
            return new VsqNrpn[] { add };
        }

        /// <summary>
        /// 指定したVsqEventを表現するNRPNを作成します
        /// </summary>
        /// <param name="ve"></param>
        /// <param name="msPreSend"></param>
        /// <param name="note_loc"></param>
        /// <returns></returns>
        public VsqNrpn[] GenerateNoteNRPN( VsqEvent ve, int msPreSend, byte note_loc ) {
            return GenerateNoteNRPN( ve, msPreSend, note_loc, 0 );
        }

        public VsqNrpn[] GenerateNoteNRPN( VsqEvent ve, int msPreSend, byte note_loc, int clShift ) {
            int clock = ve.Clock;

            double clock_msec = SecFromClock( clock ) * 1000.0;

            int ttempo = GetTempoAt( clock );
            double tempo = 6e7 / ttempo;
            //double sStart = SecFromClock( ve.Clock );
            double msEnd = SecFromClock( ve.Clock + ve.ID.Length ) * 1000.0;
            ushort duration = (ushort)Math.Ceiling( msEnd - clock_msec );
#if DEBUG
            Console.WriteLine( "GenerateNoteNRPN" );
            Console.WriteLine( "    duration=" + duration );
#endif
            byte duration0, duration1;
            GetMsbAndLsb( duration, out duration0, out duration1 );
            byte delay0, delay1;
            GetMsbAndLsb( (ushort)msPreSend, out delay0, out delay1 );
            List<VsqNrpn> ret = new List<VsqNrpn>();

            VsqNrpn add = new VsqNrpn( clock - GetPresendClockAt( clock, msPreSend ) - clShift, VsqNrpn.NRPN_CVM_NM_DELAY, delay0, delay1 );
            add.Append( VsqNrpn.NRPN_CVM_NM_NOTE_NUMBER, (byte)ve.ID.Note ); // Note number
            add.Append( VsqNrpn.NRPN_CVM_NM_VELOCITY, (byte)ve.ID.Dynamics ); // Velocity
            add.Append( VsqNrpn.NRPN_CVM_NM_NOTE_DURATION, duration0, duration1 ); // Note duration
            add.Append( VsqNrpn.NRPN_CVM_NM_NOTE_LOCATION, note_loc ); // Note Location

            if ( ve.ID.VibratoHandle != null ) {
                add.Append( VsqNrpn.NRPN_CVM_NM_INDEX_OF_VIBRATO_DB, 0x00, 0x00 );
                int vibrato_type = (int)VibratoTypeUtil.FromIconID( ve.ID.VibratoHandle.IconID );
                int note_length = ve.ID.Length;
                int vibrato_delay = ve.ID.VibratoDelay;
                byte bVibratoDuration = (byte)((float)(note_length - vibrato_delay) / (float)note_length * 127);
                byte bVibratoDelay = (byte)(0x7f - bVibratoDuration);
                add.Append( VsqNrpn.NRPN_CVM_NM_VIBRATO_CONFIG, (byte)vibrato_type, bVibratoDuration );
                add.Append( VsqNrpn.NRPN_CVM_NM_VIBRATO_DELAY, bVibratoDelay );
            }

            string[] spl = ve.ID.LyricHandle.L0.PhoneticSymbolList;
            string s = "";
            for ( int j = 0; j < spl.Length; j++ ) {
                s += spl[j];
            }
            char[] symbols = s.ToCharArray();
            add.Append( VsqNrpn.NRPN_CVM_NM_PHONETIC_SYMBOL_BYTES, (byte)symbols.Length );// 0x12(Number of phonetic symbols in bytes)
            int count = -1;
            for ( int j = 0; j < spl.Length; j++ ) {
                char[] chars = spl[j].ToCharArray();
                for ( int k = 0; k < chars.Length; k++ ) {
                    count++;
                    if ( k == 0 ) {
                        add.Append( (ushort)((0x50 << 8) | (byte)(0x13 + count)), (byte)chars[k], (byte)ve.ID.LyricHandle.L0.ConsonantAdjustment[j] ); // Phonetic symbol j
                    } else {
                        add.Append( (ushort)((0x50 << 8) | (byte)(0x13 + count)), (byte)chars[k] ); // Phonetic symbol j
                    }
                }
            }
            add.Append( VsqNrpn.NRPN_CVM_NM_PHONETIC_SYMBOL_CONTINUATION, 0x7f ); // End of phonetic symbols
            add.Append( VsqNrpn.NRPN_CVM_NM_V1MEAN, 0x04 );// 0x50(v1mean)
            add.Append( VsqNrpn.NRPN_CVM_NM_D1MEAN, 0x08 );// 0x51(d1mean)
            add.Append( VsqNrpn.NRPN_CVM_NM_D1MEAN_FIRST_NOTE, 0x14 );// 0x52(d1meanFirstNote)
            add.Append( VsqNrpn.NRPN_CVM_NM_D2MEAN, 0x1c );// 0x53(d2mean)
            add.Append( VsqNrpn.NRPN_CVM_NM_D4MEAN, 0x18 );// 0x54(d4mean)
            add.Append( VsqNrpn.NRPN_CVM_NM_PMEAN_ONSET_FIRST_NOTE, 0x0a ); // 055(pMeanOnsetFirstNote)
            add.Append( VsqNrpn.NRPN_CVM_NM_VMEAN_NOTE_TRNSITION, 0x0c ); // 0x56(vMeanNoteTransition)
            add.Append( VsqNrpn.NRPN_CVM_NM_PMEAN_ENDING_NOTE, 0x0c );// 0x57(pMeanEndingNote)
            add.Append( VsqNrpn.NRPN_CVM_NM_ADD_PORTAMENTO, (byte)ve.ID.PMbPortamentoUse );// 0x58(AddScoopToUpInternals&AddPortamentoToDownIntervals)
            add.Append( VsqNrpn.NRPN_CVM_NM_CHANGE_AFTER_PEAK, 0x32 );// 0x59(changeAfterPeak)
            byte accent = (byte)(0x64 * ve.ID.DEMaccent / 100.0);
            add.Append( VsqNrpn.NRPN_CVM_NM_ACCENT, accent );// 0x5a(Accent)
            add.Append( VsqNrpn.NRPN_CVM_NM_NOTE_MESSAGE_CONTINUATION, 0x7f );// 0x7f(Note message continuation)
            ret.Add( add );
            return ret.ToArray();
        }

        /// <summary>
        /// 指定したトラックのデータから、NRPNのリストを作成します
        /// </summary>
        /// <param name="track"></param>
        /// <param name="msPreSend"></param>
        /// <returns></returns>
        public VsqNrpn[] GenerateNRPN( VsqTrack track, int msPreSend ) {
            return GenerateNRPN( track, msPreSend, 0, (int)m_total_clocks, 0 );
        }

        public VsqNrpn[] GenerateNRPN( VsqTrack track, int msPreSend, int clStart, int clEnd, int t_temp_premeasure ) {
#if DEBUG
            Console.WriteLine( "GenerateNRPN(VsqTrack,int,int,int,int)" );
            Console.WriteLine( "    clStart,clEnd=" + clStart + "," + clEnd );
#endif
            List<VsqNrpn> list = new List<VsqNrpn>();
            //list.Add( GenerateHeaderNRPN() );

            int count = track.Events.Count;
            int note_start = 0;
            int note_end = track.Events.Count - 1;
            for ( int i = 0; i < track.Events.Count; i++ ) {
                if ( clStart <= track.Events[i].Clock ) {
                    note_start = i;
                    break;
                }
                note_start = i;
            }
            for ( int i = track.Events.Count - 1; i >= 0; i-- ) {
                if ( track.Events[i].Clock <= clEnd ) {
                    note_end = i;
                    break;
                }
            }

            // 最初の歌手を決める
            int singer_event = -1;
            for ( int i = note_start; i >= 0; i-- ) {
                if ( track.Events[i].ID.type == VsqIDType.Singer ) {
                    singer_event = i;
                    break;
                }
            }
            if ( singer_event >= 0 ) { //見つかった場合
                list.AddRange( GenerateSingerNRPN( track.Events[singer_event], 0, 0 ) );
            } else {                   //多分ありえないと思うが、歌手が不明の場合。
                throw new ApplicationException( "first singer was not specified" );
                list.Add( new VsqNrpn( 0, VsqNrpn.NRPN_CC_BS_LANGUAGE_TYPE, 0 ) );
                list.Add( new VsqNrpn( 0, VsqNrpn.NRPN_PC_VOICE_TYPE, 0 ) );
            }
#if DEBUG
            Console.WriteLine( "    note_start,note_end=" + note_start + "," + note_end );
            Console.WriteLine( "    clStart - t_temp_premeasure=" + (clStart - t_temp_premeasure) );
#endif
            for ( int i = note_start; i <= note_end; i++ ) {
                byte note_loc = 0x00;
                if ( i == note_start ) {
                    if ( i == note_end ) {
                        note_loc = 0x03;
                    } else {
                        note_loc = 0x01;
                    }
                } else if ( i == note_end ) {
                    note_loc = 0x02;
                }

                if ( track.Events[i].ID.type == VsqIDType.Anote ) {
                    list.AddRange( GenerateNoteNRPN( track.Events[i],
                                                     msPreSend,
                                                     note_loc,
                                                     clStart - t_temp_premeasure ) );
                    list.AddRange( GenerateVibratoNRPN( track.Events[i],
                                                        msPreSend,
                                                        clStart - t_temp_premeasure ) );
                } else if( track.Events[i].ID.type == VsqIDType.Singer ) {
                    if ( i > note_start && i != singer_event) {
                        list.AddRange( GenerateSingerNRPN( track.Events[i], msPreSend, clStart - t_temp_premeasure ) );
                    }
                }
            }
            list.AddRange( GenerateVoiceChangeParameterNRPN( track, msPreSend, clStart, clEnd, t_temp_premeasure ) );
            list.AddRange( GenerateExpressionNRPN( track, msPreSend, clStart, clEnd, t_temp_premeasure ) );
            list.AddRange( GeneratePitchBendSensitivityNRPN( track, msPreSend, clStart, clEnd, t_temp_premeasure ) );
            list.AddRange( GeneratePitchBendNRPN( track, msPreSend, clStart, clEnd, t_temp_premeasure ) );

            list.Sort();
            List<VsqNrpn> merged = new List<VsqNrpn>();
            for ( int i = 0; i < list.Count; i++ ) {
                merged.AddRange( list[i].Expand() );
            }
            return merged.ToArray();
        }

        /// <summary>
        /// 指定したトラックから、PitchBendのNRPNを作成します
        /// </summary>
        /// <param name="track"></param>
        /// <param name="msPreSend"></param>
        /// <returns></returns>
        public VsqNrpn[] GeneratePitchBendNRPN( VsqTrack track, int msPreSend ) {
            return GeneratePitchBendNRPN( track, msPreSend, 0, (int)m_total_clocks, 0 );
        }

        public VsqNrpn[] GeneratePitchBendNRPN( VsqTrack track, int msPreSend, int clStart, int clEnd, int t_pre_measure ) {
            List<VsqNrpn> ret = new List<VsqNrpn>();
            //SortedList<int, int> list = track[VsqCurveType.PIT].List;
            int[] keys = track["PIT"].Keys;
            int count = keys.Length;
            byte delay0, delay1;
            GetMsbAndLsb( (ushort)msPreSend, out delay0, out delay1 );
            int start_clock = clStart < m_premeasure_clocks ? m_premeasure_clocks : clStart;
            VsqNrpn add0 = new VsqNrpn( start_clock - GetPresendClockAt( start_clock, msPreSend ) - clStart + t_pre_measure, VsqNrpn.NRPN_PB_VERSION_AND_DEVICE, 0x00, 0x00 );
            add0.Append( VsqNrpn.NRPN_PB_DELAY, delay0, delay1 );

            ushort start_value = (ushort)(track["PIT"][start_clock] + 0x2000);
            byte start_value0, start_value1;
            GetMsbAndLsb( start_value, out start_value0, out start_value1 );
            add0.Append( VsqNrpn.NRPN_PB_PITCH_BEND, start_value0, start_value1 );
            ret.Add( add0 );
            
            for ( int i = 1; i < keys.Length; i++ ) {
                if ( keys[i] > clEnd ) {
                    break;
                }
                if ( keys[i] > start_clock ) {
                    ushort value = (ushort)(track["PIT"][keys[i]] + 0x2000);
                    byte value0, value1;
                    GetMsbAndLsb( value, out value0, out value1 );
                    VsqNrpn add = new VsqNrpn( keys[i] - GetPresendClockAt( keys[i], msPreSend ) - clStart + t_pre_measure, VsqNrpn.NRPN_PB_PITCH_BEND, value0, value1 );
                    ret.Add( add );
                }
            }
            return ret.ToArray();
        }

        /// <summary>
        /// 指定したトラックからPitchBendSensitivityのNRPNを作成します
        /// </summary>
        /// <param name="track"></param>
        /// <param name="msPreSend"></param>
        /// <returns></returns>
        public VsqNrpn[] GeneratePitchBendSensitivityNRPN( VsqTrack track, int msPreSend ) {
            return GeneratePitchBendSensitivityNRPN( track, msPreSend, 0, (int)m_total_clocks, 0 );
        }

        public VsqNrpn[] GeneratePitchBendSensitivityNRPN( VsqTrack track, int msPreSend, int clStart, int clEnd, int t_pre_measure ) {
            List<VsqNrpn> ret = new List<VsqNrpn>();
            //SortedList<int, int> list = track[VsqCurveType.PBS].List;
            int[] keys = track["PBS"].Keys;
            int count = keys.Length;
            byte delay0, delay1;
            GetMsbAndLsb( (ushort)msPreSend, out delay0, out delay1 );
            int start_clock = clStart < m_premeasure_clocks ? m_premeasure_clocks : clStart;
            int start_value = track["PBS"][start_clock];
            VsqNrpn add0 = new VsqNrpn( start_clock - GetPresendClockAt( start_clock, msPreSend ) - clStart + t_pre_measure, 
                                        VsqNrpn.NRPN_CC_PBS_VERSION_AND_DEVICE,
                                        0x00, 
                                        0x00 );
            add0.Append( VsqNrpn.NRPN_CC_PBS_DELAY, delay0, delay1 );
            add0.Append( VsqNrpn.NRPN_CC_PBS_PITCH_BEND_SENSITIVITY, (byte)start_value, 0x00 );
            ret.Add( add0 );
            for ( int i = 1; i < keys.Length; i++ ) {
                if ( keys[i] > clEnd ) {
                    break;
                }
                if ( keys[i] > start_clock ) {
                    VsqNrpn add = new VsqNrpn( keys[i] - GetPresendClockAt( keys[i], msPreSend ) - clStart + t_pre_measure, 
                                               VsqNrpn.NRPN_CC_PBS_PITCH_BEND_SENSITIVITY,
                                               (byte)track["PBS"][keys[i]],
                                               0x00 );
                    ret.Add( add );
                }
            }
            return ret.ToArray();
        }

        public VsqNrpn[] GenerateVibratoNRPN( VsqEvent ve, int msPreSend ) {
            return GenerateVibratoNRPN( ve, msPreSend, 0 );
        }

        public VsqNrpn[] GenerateVibratoNRPN( VsqEvent ve, int msPreSend, int clShift ) {
            List<VsqNrpn> ret = new List<VsqNrpn>();
            if ( ve.ID.VibratoHandle != null ){
                int vclock = ve.Clock + ve.ID.VibratoDelay;
                byte delay0, delay1;
                GetMsbAndLsb( (ushort)msPreSend, out delay0, out delay1 );
                VsqNrpn add2 = new VsqNrpn( vclock - GetPresendClockAt( vclock, msPreSend ) - clShift, 
                                            VsqNrpn.NRPN_CC_VD_VERSION_AND_DEVICE,
                                            0x00,
                                            0x00 );
                add2.Append( VsqNrpn.NRPN_CC_VD_DELAY, delay0, delay1 );
                add2.Append( VsqNrpn.NRPN_CC_VD_VIBRATO_DEPTH, (byte)ve.ID.VibratoHandle.StartDepth );
                add2.Append( VsqNrpn.NRPN_CC_VR_VIBRATO_RATE, (byte)ve.ID.VibratoHandle.StartRate );
                ret.Add( add2 );
                int vlength = ve.Clock + ve.ID.Length - ve.ID.VibratoDelay;
                if ( ve.ID.VibratoHandle.RateBPNum > 0 ) {
                    for ( int i = 0; i < ve.ID.VibratoHandle.RateBPX.Length; i++ ) {
                        float percent = ve.ID.VibratoHandle.RateBPX[i];
                        int cl = vclock + (int)(percent * vlength);
                        ret.Add( new VsqNrpn( cl, VsqNrpn.NRPN_CC_VR_VIBRATO_RATE, (byte)ve.ID.VibratoHandle.RateBPY[i] ) );
                    }
                }
                if ( ve.ID.VibratoHandle.DepthBPNum > 0 ) {
                    for ( int i = 0; i < ve.ID.VibratoHandle.DepthBPX.Length; i++ ) {
                        float percent = ve.ID.VibratoHandle.DepthBPX[i];
                        int cl = vclock + (int)(percent * vlength);
                        ret.Add( new VsqNrpn( cl, VsqNrpn.NRPN_CC_VD_VIBRATO_DEPTH, (byte)ve.ID.VibratoHandle.DepthBPY[i] ) );
                    }
                }
            }
            ret.Sort();
            return ret.ToArray();
        }

        /// <summary>
        /// 指定したトラックから、VoiceChangeParameterのNRPNのリストを作成します
        /// </summary>
        /// <param name="track"></param>
        /// <param name="msPreSend"></param>
        /// <returns></returns>
        public VsqNrpn[] GenerateVoiceChangeParameterNRPN( VsqTrack track, int msPreSend ) {
            return GenerateVoiceChangeParameterNRPN( track, msPreSend, 0, (int)m_total_clocks, 0 );
        }

        public VsqNrpn[] GenerateVoiceChangeParameterNRPN( VsqTrack track, int msPreSend, int clStart, int clEnd, int t_pre_measure ) {
            int premeasure_clock = m_premeasure_clocks;
            byte delay0, delay1;
            GetMsbAndLsb( (ushort)msPreSend, out delay0, out delay1 );
            List<VsqNrpn> res = new List<VsqNrpn>();
            int start_clock = (clStart < premeasure_clock) ? premeasure_clock : clStart;
            VsqNrpn ret = new VsqNrpn( start_clock - GetPresendClockAt( start_clock, msPreSend ) - clStart + t_pre_measure, 
                                       VsqNrpn.NRPN_VCP_VERSION_AND_DEVICE,
                                       0x00, 
                                       0x00 );
            ret.Append( VsqNrpn.NRPN_VCP_DELAY, delay0, delay1 );  // Voice Change Parameter delay

            ret.Append( VsqNrpn.NRPN_VCP_VOICE_CHANGE_PARAMETER_ID, 0x31 );   // BRE
            ret.Append( VsqNrpn.NRPN_VCP_VOICE_CHANGE_PARAMETER, (byte)track["BRE"][start_clock] );

            ret.Append( VsqNrpn.NRPN_VCP_VOICE_CHANGE_PARAMETER_ID, 0x32 );   // BRI
            ret.Append( VsqNrpn.NRPN_VCP_VOICE_CHANGE_PARAMETER, (byte)track["BRI"][start_clock] );

            ret.Append( VsqNrpn.NRPN_VCP_VOICE_CHANGE_PARAMETER_ID, 0x33 );   // CLE
            ret.Append( VsqNrpn.NRPN_VCP_VOICE_CHANGE_PARAMETER, (byte)track["CLE"][start_clock] );

            ret.Append( VsqNrpn.NRPN_VCP_VOICE_CHANGE_PARAMETER_ID, 0x34 );   // POR
            ret.Append( VsqNrpn.NRPN_VCP_VOICE_CHANGE_PARAMETER, (byte)track["POR"][start_clock] );

            ret.Append( VsqNrpn.NRPN_VCP_VOICE_CHANGE_PARAMETER_ID, 0x35 );   // OPE
            ret.Append( VsqNrpn.NRPN_VCP_VOICE_CHANGE_PARAMETER, (byte)track["OPE"][start_clock] );

            ret.Append( VsqNrpn.NRPN_VCP_VOICE_CHANGE_PARAMETER_ID, 0x70 );   // GEN
            ret.Append( VsqNrpn.NRPN_VCP_VOICE_CHANGE_PARAMETER, (byte)track["GEN"][start_clock] );
            res.Add( ret );

            foreach ( string vct in _CURVES ) {
                byte lsb = 0x31;
                switch ( vct ) {
                    case "DYN":
                    case "PBS":
                    case "PIT":
                    case "VEL":
                        continue;
                    case "BRE":
                        lsb = 0x31;
                        break;
                    case "BRI":
                        lsb = 0x32;
                        break;
                    case "CLE":
                        lsb = 0x33;
                        break;
                    case "POR":
                        lsb = 0x34;
                        break;
                    case "OPE":
                        lsb = 0x35;
                        break;
                    case "GEN":
                        lsb = 0x70;
                        break;
                }
                //SortedList<int, int> list = track[vct].List;
                int[] keys = track[vct].Keys;
                for ( int i = 0; i < keys.Length; i++ ) {
                    if ( keys[i] > clEnd ) {
                        break;
                    }
                    if ( keys[i] > start_clock ) {
                        VsqNrpn add = new VsqNrpn( keys[i] - GetPresendClockAt( keys[i], msPreSend ) - clStart + t_pre_measure,
                                                   VsqNrpn.NRPN_VCP_VOICE_CHANGE_PARAMETER_ID, 
                                                   lsb );
                        add.Append( VsqNrpn.NRPN_VCP_VOICE_CHANGE_PARAMETER, (byte)track[vct][keys[i]] );
                        res.Add( add );
                    }
                }
            }

            return res.ToArray();
        }

        private static void GetMsbAndLsb( ushort value, out byte msb, out byte lsb ) {
            msb = (byte)(value >> 7);
            lsb = (byte)(value - (msb << 7));
        }

        public void Write( string file ) {
            Write( file, 500 );
        }

        /// <summary>
        /// ファイルにこのVsqFileインスタンスの中身を出力します。
        /// </summary>
        /// <param name="file"></param>
        /// <param name="msPreSend">プリセンドタイム(msec)</param>
        public void Write( string file, int msPreSend ) {
#if DEBUG
            Console.WriteLine( "VsqFile.Write(string)" );
#endif
            int last_clock = 0;
            for ( int track = 1; track < Tracks.Count; track++ ) {
                if ( Tracks[track].Events.Count > 0 ) {
                    int index = Tracks[track].Events.Count - 1;
                    VsqEvent last = Tracks[track].Events[index];
                    last_clock = Math.Max( last_clock, last.Clock + last.ID.Length );
                }
            }

            //int delay = DelayFromPreSend( pre_send );
            using ( FileStream fs = new FileStream( file, FileMode.Create ) ) {
                long first_position;//チャンクの先頭のファイル位置

                #region  ヘッダ
                //チャンクタイプ
                fs.Write( _MTHD, 0, 4 );
                //データ長
                fs.WriteByte( 0x00 );
                fs.WriteByte( 0x00 );
                fs.WriteByte( 0x00 );
                fs.WriteByte( 0x06 );
                //フォーマット
                fs.WriteByte( 0x00 );
                fs.WriteByte( 0x01 );
                //トラック数
                WriteUnsignedShort( fs, (ushort)this.Tracks.Count );
                //時間単位
                fs.WriteByte( 0x01 );
                fs.WriteByte( 0xe0 );
                #endregion

                #region Master Track
                //チャンクタイプ
                fs.Write( _MTRK, 0, 4 );
                //データ長。とりあえず0を入れておく
                fs.Write( new byte[] { 0x00, 0x00, 0x00, 0x00 }, 0, 4 );
                first_position = fs.Position;
                //トラック名
                WriteFlexibleLengthUnsignedLong( fs, 0 );//デルタタイム
                fs.WriteByte( 0xff );//ステータスタイプ
                fs.WriteByte( 0x03 );//イベントタイプSequence/Track Name
                fs.WriteByte( (byte)_MASTER_TRACK.Length );//トラック名の文字数。これは固定
                fs.Write( _MASTER_TRACK, 0, _MASTER_TRACK.Length );

                List<MidiEvent> events = new List<MidiEvent>();
                foreach ( TimeSigTableEntry entry in TimeSigTable ) {
                    events.Add( MidiEvent.TimeSig( entry.Clock, entry.Numerator, entry.Denominator ) );
                    last_clock = Math.Max( last_clock, entry.Clock );
                }
                foreach ( TempoTableEntry entry in TempoTable ) {
                    events.Add( MidiEvent.TempoChange( entry.Clock, entry.Tempo ) );
                    last_clock = Math.Max( last_clock, entry.Clock );
                }
#if DEBUG
                Console.WriteLine( "    events.Count=" + events.Count );
#endif
                events.Sort();
                uint last = 0;
                foreach ( MidiEvent item in events ) {
                    switch ( item.type ) {
                        case MidiEventType.tempo:
#if DEBUG
                            Console.WriteLine( "    tempo: delta time=" + (item.index - last) );
#endif
                            WriteFlexibleLengthUnsignedLong( fs, (ulong)(item.index - last) );
                            last = (uint)item.index;
                            fs.WriteByte( 0xff );
                            fs.WriteByte( 0x51 );
                            fs.WriteByte( 0x03 );
                            WriteTempo( fs, (uint)item.intValue[0] );
                            break;
                        case MidiEventType.time_signal:
#if DEBUG
                            Console.WriteLine( "    time_signal: delta time=" + (item.index - last) );
#endif
                            int num = item.intValue[0];
                            int den = item.intValue[1];
                            WriteFlexibleLengthUnsignedLong( fs, (ulong)(item.index - last) );
                            last = (uint)item.index;
                            fs.WriteByte( 0xff );
                            fs.WriteByte( 0x58 );//イベントタイプTime Signature
                            fs.WriteByte( 0x04 );//データ長
                            fs.WriteByte( (byte)num );//分子
                            fs.WriteByte( (byte)Math.Log( den, 2 ) );//分母の2の負のべき乗数
                            fs.WriteByte( 0x18 );
                            fs.WriteByte( 0x08 );
                            break;
                    }
                }

                //WriteFlexibleLengthUnsignedLong( fs, (ulong)(last_clock + 120 - last) );
                WriteFlexibleLengthUnsignedLong( fs, 0 );
                fs.WriteByte( 0xff );
                fs.WriteByte( 0x2f );//イベントタイプEnd of Track
                fs.WriteByte( 0x00 );
                long pos = fs.Position;
                fs.Seek( first_position - 4, SeekOrigin.Begin );
                WriteUnsignedInt( fs, (uint)(pos - first_position) );
                fs.Seek( pos, SeekOrigin.Begin );
                #endregion

                #region トラック
                VsqTrack t_track = (VsqTrack)Tracks[1].Clone();
                t_track.Master = (VsqMaster)m_master.Clone();
                t_track.Mixer = (VsqMixer)m_mixer.Clone();
                PrintTrack( t_track, fs, msPreSend );
                for ( int track = 2; track < Tracks.Count; track++ ) {
                    PrintTrack( Tracks[track], fs, msPreSend );
                }
                #endregion

            }
        }

        /// <summary>
        /// メタテキストの行番号から、各行先頭のプレフィクス文字列("DM:0123:"等)を作成します
        /// </summary>
        /// <param name="count"></param>
        /// <returns></returns>
        public static string GetLinePrefix( int count ) {
            int digits = HowManyDigits( count );
            int c = (digits - 1) / 4 + 1;
            string format = "";
            for ( int i = 0; i < c; i++ ) {
                format += "0000";
            }
            return "DM:" + count.ToString( format ) + ":";
        }

        /// <summary>
        /// 数numberの桁数を調べます。（10進数のみ）
        /// </summary>
        /// <param name="number"></param>
        /// <returns></returns>
        private static int HowManyDigits( int number ) {
            int val;
            if ( number > 0 ) {
                val = number;
            } else {
                val = -number;
            }
            int i = 1;
            int digits = 1;
            while ( true ) {
                i++;
                digits *= 10;
                if ( val < digits ) {
                    return i - 1;
                }
            }
        }

        /// <summary>
        /// char[]を書き込む。
        /// </summary>
        /// <param name="fs"></param>
        /// <param name="item"></param>
        public void WriteCharArray( FileStream fs, char[] item ) {
            for ( int i = 0; i < item.Length; i++ ) {
                fs.WriteByte( (byte)item[i] );
            }
        }

        /// <summary>
        /// テンポ値をビッグエンディアンで書き込みます。
        /// テンポは3バイトで記入されるので、別関数を用意した。
        /// </summary>
        /// <param name="fs"></param>
        /// <param name="tempo"></param>
        public void WriteTempo( FileStream fs, uint tempo ) {
            byte[] dat = BitConverter.GetBytes( tempo );
            if ( BitConverter.IsLittleEndian ) {
                Array.Reverse( dat );
            }
            fs.WriteByte( dat[1] );
            fs.WriteByte( dat[2] );
            fs.WriteByte( dat[3] );
        }

        /// <summary>
        /// ushort値をビッグエンディアンでfsに書き込みます
        /// </summary>
        /// <param name="data"></param>
        public void WriteUnsignedShort( FileStream fs, ushort data ) {
            byte[] dat = BitConverter.GetBytes( data );
            if ( BitConverter.IsLittleEndian ) {
                Array.Reverse( dat );
            }
            fs.Write( dat, 0, dat.Length );
        }

        /// <summary>
        /// uint値をビッグエンディアンでfsに書き込みます
        /// </summary>
        /// <param name="data"></param>
        public void WriteUnsignedInt( FileStream fs, uint data ) {
            byte[] dat = BitConverter.GetBytes( data );
            if ( BitConverter.IsLittleEndian ) {
                Array.Reverse( dat );
            }
            fs.Write( dat, 0, dat.Length );
        }

        /// <summary>
        /// SMFの可変長数値表現を使って、ulongをbyte[]に変換します
        /// </summary>
        /// <param name="fs"></param>
        /// <param name="number"></param>
        public static byte[] GetBytesFlexibleLengthUnsignedLong( ulong number ) {
            bool[] bits = new bool[64];
            ulong val = 0x1;
            bits[0] = (number & val) == val;
            for ( int i = 1; i < 64; i++ ) {
                val = val << 1;
                bits[i] = (number & val) == val;
            }
            int first = 0;
            for ( int i = 63; i >= 0; i-- ) {
                if ( bits[i] ) {
                    first = i;
                    break;
                }
            }
            // 何バイト必要か？
            int bytes = first / 7 + 1;
            byte[] ret = new byte[bytes];
            for ( int i = 1; i <= bytes; i++ ) {
                uint num = 0;
                uint count = 0x80;
                for ( int j = (bytes - i + 1) * 7 - 1; j >= (bytes - i + 1) * 7 - 6 - 1; j-- ) {
                    count = count >> 1;
                    if ( bits[j] ) {
                        num += count;
                    }
                }
                if ( i != bytes ) {
                    num += 0x80;
                }
                ret[i - 1] = (byte)num;
            }
            return ret;
        }

        /// <summary>
        /// 整数を書き込む。フォーマットはSMFの可変長数値表現。
        /// </summary>
        /// <param name="fs"></param>
        /// <param name="number"></param>
        public static void WriteFlexibleLengthUnsignedLong( Stream fs, ulong number ) {
            byte[] bytes = GetBytesFlexibleLengthUnsignedLong( number );
            fs.Write( bytes, 0, bytes.Length );
        }
    }

}
