﻿/*
 * VsqTrack.cs
 * Copyright (c) 2008-2009 kbinani
 *
 * This file is part of Boare.Lib.Vsq.
 *
 * Boare.Lib.Vsq is free software; you can redistribute it and/or
 * modify it under the terms of the BSD License.
 *
 * Boare.Lib.Vsq is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */
using System;
using System.Collections.Generic;
using System.IO;

namespace Boare.Lib.Vsq {

    /// <summary>
    /// Stores the data of a vsq track.
    /// </summary>
    [Serializable]
    public partial class VsqTrack : ICloneable {
        private string m_name;
        private VsqMetaText m_meta_text;
        private List<MidiEvent> m_midi_event;
        private int m_edited_start = int.MaxValue;
        private int m_edited_end = int.MinValue;

        public IEnumerable<VsqEvent> GetSingerEventEnumerator() {
            for ( int i = 0; i < m_meta_text.Events.Count; i++ ) {
                if ( m_meta_text.Events[i].ID.type == VsqIDType.Singer ) {
                    yield return m_meta_text.Events[i];
                }
            }
        }

        public IEnumerable<VsqEvent> GetNoteEventEnumerator() {
            for ( int i = 0; i < m_meta_text.Events.Count; i++ ) {
                if ( m_meta_text.Events[i].ID.type == VsqIDType.Anote ) {
                    yield return m_meta_text.Events[i];
                }
            }
        }

        public void PrintMetaText( TextMemoryStream sw, bool encode, ulong eos, int start ) {
            m_meta_text.Print( sw, encode, eos, start );
        }

        public void PrintMetaText( string file ) {
            TextMemoryStream tms = new TextMemoryStream( FileAccess.ReadWrite );
            int count = m_meta_text.Events.Count;
            int clLast = m_meta_text.Events[count - 1].Clock + 480;
            m_meta_text.Print( tms, true, (ulong)clLast, 0 );
            using ( StreamWriter sw = new StreamWriter( file ) ) {
                tms.Rewind();
                while ( tms.Peek() >= 0 ) {
                    string line = tms.ReadLine();
                    sw.WriteLine( line );
                }
            }
        }

        public VsqMaster Master {
            get {
                return m_meta_text.master;
            }
            internal set {
                m_meta_text.master = value;
            }
        }

        public VsqMixer Mixer {
            get {
                return m_meta_text.mixer;
            }
            internal set {
                m_meta_text.mixer = value;
            }
        }

        /*public VsqBPList this[VsqCurveType curveType] {
            get {
                return m_meta_text[curveType];
            }
            set {
                m_meta_text[curveType] = value;
            }
        }*/

        public VsqBPList this[string curve] {
            get {
                return m_meta_text[curve];
            }
            set {
                m_meta_text[curve] = value;
            }
        }

        public VsqEventList Events{
            get {
                return m_meta_text.Events;
            }
        }

        public int EditedStart {
            get {
                return m_edited_start;
            }
            internal set {
                if ( value < m_edited_start ) {
                    m_edited_start = value;
                }
            }
        }

        public int EditedEnd {
            get {
                return m_edited_end;
            }
            internal set {
                if ( m_edited_end < value ) {
                    m_edited_end = value;
                }
            }
        }

        public void ResetEditedArea() {
            m_edited_start = int.MaxValue;
            m_edited_end = int.MinValue;
        }

        public object Clone() {
            VsqTrack res = new VsqTrack();
            res.m_name = m_name;
            if ( m_meta_text != null ) {
                res.m_meta_text = (VsqMetaText)m_meta_text.Clone();
            }
            if ( m_midi_event != null ) {
                res.m_midi_event = new List<MidiEvent>();
                foreach ( MidiEvent item in m_midi_event ) {
                    res.m_midi_event.Add( (MidiEvent)item.Clone() );
                }
            }
            res.m_edited_start = m_edited_start;
            res.m_edited_end = m_edited_end;
            return res;
        }

        private VsqTrack() {
        }

        /// <summary>
        /// Master Trackを構築
        /// </summary>
        /// <param name="tempo"></param>
        /// <param name="numerator"></param>
        /// <param name="denominator"></param>
        public VsqTrack( int tempo, int numerator, int denominator ) {
            this.m_name = "Master Track";
            this.m_meta_text = null;
            this.m_midi_event = new List<MidiEvent>();
            this.m_midi_event.Add( new MidiEvent( "0 Tempo " + tempo ) );
            this.m_midi_event.Add( new MidiEvent( "0 TimeSig " + numerator + "/" + denominator + " 24 8" ) );
        }

        /// <summary>
        /// Master Trackでないトラックを構築。
        /// </summary>
        /// <param name="name"></param>
        /// <param name="singer"></param>
        public VsqTrack( string name, string singer ) {
            m_name = name;
            m_meta_text = new VsqMetaText( name, singer );
            m_midi_event = new List<MidiEvent>();
        }

        /// <summary>
        /// midiイベントのリスト
        /// </summary>
        public List<MidiEvent> MidiEvent {
            get {
                return m_midi_event;
            }
        }

        /// <summary>
        /// メタテキスト。
        /// </summary>
        private VsqMetaText MetaText {
            get {
                return m_meta_text;
            }
        }

        /// <summary>
        /// トラックの名前。
        /// </summary>
        public string Name {
            get {
                return m_name;
            }
            set {
                m_name = value;
            }
        }

        /// <summary>
        /// 歌詞の文字数を調べます
        /// </summary>
        /// <returns></returns>
        public int LyricLength {
            get {
                int counter = 0;
                for ( int i = 0; i < MetaText.Events.Count; i++ ) {
                    if ( MetaText.Events[i].ID.type == VsqIDType.Anote ) {
                        counter++;
                    }
                }
                return counter;
            }
        }

        /// <summary>
        /// vsqファイルをmf2t形式にテキスト化されたファイルからコンストラクト。
        /// </summary>
        /// <param name="lines"></param>
        public VsqTrack( List<string> lines ) {
            m_midi_event = new List<MidiEvent>();
            m_name = "";
            string meta_text_path;
            meta_text_path = Path.GetTempFileName();

            using ( TextMemoryStream sw = new TextMemoryStream( FileAccess.Write ) ) {
                int signal;
                foreach ( string s in lines ) {
                    string line = s;
                    // signalを取得
                    int index = line.IndexOf( ' ' );
                    string str_signal = line.Substring( 0, index );
                    signal = int.Parse( str_signal );
                    string remain = line.Substring( index + 1 );

                    // イベントの種類で処理を分岐
                    string[] spl = remain.Split( new char[] { ' ' } );
                    if ( spl[0] == "Meta" && spl[1] == "Text" ) {
                        line = line.Replace( signal + " Meta Text \"", "" );
                        int second_colon = line.IndexOfAny( new char[] { ':' }, 3 );
                        line = line.Substring( second_colon + 1 );
                        line = line.Substring( 0, line.Length - 1 );
                        line = line.Replace( "\\n", Environment.NewLine );
                        sw.Write( line );
                    } else if ( spl[0] == "Meta" && (spl[1] == "TrkName" || spl[1] == "SeqName") ) {
                        m_name = spl[2];
                        for ( int i = 3; i < spl.Length; i++ ) {
                            m_name += " " + spl[i];
                        }
                        m_name = m_name.Replace( "\"", "" );
                        m_name = Lyric.decode( m_name );
                    } else {
                        MidiEvent.Add( new MidiEvent( line ) );
                    }
                }

                sw.Rewind();
                m_meta_text = new VsqMetaText( sw );
            }
#if !DEBUG
            File.Delete( meta_text_path );
#endif
        }

        /// <summary>
        /// MidiEventの中からテンポ情報を抽出します
        /// </summary>
        /// <returns></returns>
        public List<MidiEvent> GetTempoList() {
            List<MidiEvent> list = new List<MidiEvent>();
            for ( int i = 0; i < MidiEvent.Count; i++ ) {
                if ( MidiEvent[i].type == MidiEventType.tempo ) {
                    list.Add( MidiEvent[i] );
                }
            }
            list.Sort();
            return list;
        }

        /// <summary>
        /// MidiEventの中から拍子情報を抽出します
        /// </summary>
        /// <returns></returns>
        public List<MidiEvent> GetTimeSigList() {
            List<MidiEvent> list = new List<MidiEvent>();
            for ( int i = 0; i < MidiEvent.Count; i++ ) {
                if ( MidiEvent[i].type == MidiEventType.time_signal ) {
                    list.Add( MidiEvent[i] );
                }
            }
            list.Sort();
            return list;
        }
    }

}
