package org.lixm.core.common;


/**
 * <p>LIXMの処理中に発生したより特定可能な例外を表すクラスです。<p>
 *  
 *  <h3>LIXMの処理行程</h3>
 *  
 *  <p>LIXMには大きく分けて３つの処理が存在します。</p>
 * 
 *  
 *  <dl>
 *  <dt>プリプロセス段階</dt>
 *  <dd><p>
 *  モデライズ処理に必要な事前処理またはやっておきたい処理がこの段階に該当します。</p>
 *  
 *  <p>この段階の処理を行うかは任意ですが
 *  行う場合には必ずモデライズを開始する前でなければならず、
 *  かつこの段階が正常に終了しなければモデライズフェーズに移行してはいけません。</p>
 *  
 *  <p>この暖段階ではまだ{@link org.lixm.core.list.XMLModelList XMLModelList}
 *  は一切利用できません。</p></dd>
 * 
 *  
 *  <dt>モデライズ段階</dt>
 *  <dd><p>プリプロセスが完了しXMLソースをモデル化し
 *  {@link org.lixm.core.list.XMLModelList XMLModelList}
 * を作成する段階です。</p>
 *  
 *  <p>この段階の処理は必ず行わなければならず、正常に完了した場合に初めて<code>
 *  XMLModelList</code>を利用できます。</p></dd>
 *  
 *  <dt>その他の段階</dt>
 * 
 *  <dd><p>この段階は先の二つ以外にアプリケーション固有の処理が必要な場合が該当します。<br>
 *  プリプロセス前に発生するかモデライズ後に発生するか、
 *  LIXM仕様ではその順序は一切定められていません。</p></dd>
 *  </dl>
 *  
 *  <p>LIXM処理の詳細なシークエンスは以下の通りです</p>
 *  
 *  <pre>
 *  LIXM処理開始
 *              ↓
 *           （必要ならば）プリプロセスフェーズ開始→プリプロセスフェーズ実行中→プリプロセスフェーズ終了
 *                       ↓
 *                   モデライズフェーズ開始→モデライズフェーズ実行中→モデライズフェーズ終了
 *                               ↓
 *                          アプリケーションコードのXMLModelList利用
 *                                    ↓
 *                                 LIXM処理終了
 *  </pre>
 *  
 *  <p>縦方向が大きく分けた各フェーズ間の進行段階です。<br>
 * 
 *  横方向がその中でさらに分けた個々のフェーズ内の進行段階です。</p>
 *  
 *  <p>各フェーズの終了にはさらに二つの終了状態があり、
 *  それは<em>正常終了<em>と</em>正常ではない終了です</em>。<br>
 *  次のフェーズに進んで良いのは前のフェーズの最終処理が正常に完了した場合のみです。</p>
 *  
 *  <p>この時の各フェーズが正常に終了しなかった場合に発生するのが
 *  <code>LIXMPhaseException</code>となります。</p>
 *  
 *  @author tasogare
 *  @version 1.0M2
 * 
 */
public class LIXMPhaseException extends LIXMException {

    /**
	 * 
	 */
	private static final long serialVersionUID = 2037311115378486807L;

	/**
     * プリプロセスフェーズで例外が発生したことを表す定数です 
     */
    public static final byte  PHASE_PREPROCESS = 0;

    /**
     * モデライズフェーズで例外が発生したことを表す定数です
     */
    public static final byte  PHASE_MODELAIZE  = 2;

    /**
     * その他のフェーズで例外が発生したことを表す定数です
     */
    public static final byte  PHASE_OTHER      = 127;

    private byte              phaseType;

    private LIXMPhaseException() {
    }

    /**
     * フェーズ例外を直接的なメッセージと共に作ります。
     * 
     * @param message この例外の直接的なメッセージ
     * @param phase どのフェーズか表す定数
     * 
     * @see #PHASE_PREPROCESS
     * @see #PHASE_MODELAIZE
     * @see #PHASE_OTHER
     */
    public LIXMPhaseException(
        String message, byte phase)
    {
        super(message);
        phaseType = phase;
    }

    /**
     * 一般情報を持つフェーズ例外を作ります。
     * 
     * @param e この例外が持つ一般情報
     * @param phase どのフェーズか表す定数
     * 
     * @see #PHASE_PREPROCESS
     * @see #PHASE_MODELAIZE
     * @see #PHASE_OTHER
     */
    public LIXMPhaseException(
        Exception e, byte phase)
    {
        super(e);
        phaseType = phase;
    }

    /**
     * 一般情報を持つフェーズ例外を直接的なメッセージと共に作ります。
     *
     * @param message この例外の直接的なメッセージ
     * @param e この例外が持つ一般情報
     * @param phase どのフェーズか表す定数
     */
    public LIXMPhaseException(
        String message, Exception e, byte phase)
    {
        super(message, e);
        phaseType = phase;
    }

    /**
     * {@inheritDoc}
     */
    public String getMessage() {

        String phaseInfo = "";
        switch (phaseType) {
        case PHASE_PREPROCESS :
            phaseInfo = "プリプロセスの段階で問題が発生しました";
            break;

        case PHASE_MODELAIZE :
            phaseInfo = "モデルライズの段階で問題が発生しました";
            break;
        case PHASE_OTHER :
            phaseInfo = "固有の段階で問題が発生しました";
            break;
        default :
            break;
        }        
        return phaseInfo + " : " + super.getMessage( );
    }

    /**
     * <p>どのフェーズで発生した例外か特定します。<br>
     * このメソッドで得られる値はこの例外クラス内で完結される値で
     * その他の場所で用いられるべきではありません。</p>
     * 
     * <p>このメソッドで得られる値の利用は例外処理に留めてください。</p>
     * 
     * @see #PHASE_PREPROCESS
     * @see #PHASE_MODELAIZE
     * @see #PHASE_OTHER
     * 
     * @return フェーズを特定する値
     */
    public byte getPhaseType() {
        return phaseType;
    }
}
