package org.lixm.core.model;

import org.lixm.core.common.XMLType;

/**
 * <p>属性をモデル化します。</p>
 * @author tasogare
 * @version 1.1.1
 *
 */
public class AttributeModel extends AbstractModel {

    private String name;
    private String value;
    private String pref;
    private String id;

    public AttributeModel(
        String name, String value, String pref, String id)
    {

        super(XMLType.ATTRIBUTE);
        this.name = name;
        this.value = value;
        this.pref = pref;
        this.id = id;
    }

    /**
     * <p>実装依存の方法で属性名文字列を取得します。<br>
     * 主にデバッグやエンドユーザーメッセージに利用されるユーティリティです。</p>
     * 
     * 属性のローカル名や完全修飾名を取得したい場合は{@link #toLocalName()}か
     * {@link #toQName()}を利用して下さい。
     * 
     * @see #toLocalName()
     * @see #toQName()
     * 
     * @return 実装依存な属性名文字列
     */
    public String getName() {
        return name;
    }

    public void setName(String name) {
        this.name = name;
    }

    /**
     * <p>属性値を取得します。</p>
     * @return 属性値
     */
    public String getValue() {
        return value;
    }

    /**
     * 属性値を設定します。
     * @param value 新たな属性値
     */
    public void setValue(String value) {
        this.value = value;
    }

    /**
     * <p>名前空間プレフィックスを取得します。存在しない場合は<code>null</code>となります。</p>
     * @return 名前空間プレフィックス。存在しない場合は<code>null</code>
     */
    public String getPrefix() {
        return pref;
    }

    /**
     * <p>名前空間プレフィックスを設定します。存在しない場合は<code>null</code>となります。</p>
     * 
     * @param pref 新たな名前空間プレフィックス
     */
    public void setPrefix(String pref) {
        this.pref = pref;
    }

    /**
     * <p>名前空間URIを取得します。存在しない場合は<code>nullとなります。</code></p>
     * @return 名前空間URI。存在しない場合は<code>null</code>
     */
    public String getNamespace() {
        return id;
    }

    /**
     * <p>名前空間URIを設定します。存在しない場合は<code>nullとなります。</code></p>
     * 
     * @param namespace 新たな名前空間URI
     */
    public void setNamespace(String namespace) {
        this.id = namespace;
    }

    /**
     * <p>属性のローカル名を取得します。</p>
     * 
     * @return 属性のローカル名
     */
    public String toLocalName() {
        return getName( );
    }

    /**
     * <p>属性の完全修飾名を取得します。完全修飾名が存在しない場合必ず<code>null</code>
     * になります。</p>
     * 
     * @return 属性の完全修飾名。完全修飾名が存在しない場合は<code>null</code>。
     */
    public String toQName() {

        String qName = "";
        if (getPrefix( ) != null) {
            qName = getPrefix( ) + ":" + getName( );
        } else {
            qName = null;
        }
        return qName;
    }


    public String getLocalPair() {
        return this.toLocalName( ) + "=" + "'" + this.getValue( ) + "'";
    }

    public String getQPair() {
        if (toQName( ) != null) {
            return this.toQName( ) + "=" + "'" + this.getValue( ) + "'";
        } else {
            return this.getLocalPair( );
        }
    }

    /**
     * {@inheritDoc}
     */
    public String toString() {
        return getLocalPair( );
    }

    /**
     * <p>モデルの同一性を論理的に調べその結果を返します。</p>
     * @return 同一のモデルなら<code>true</code>、違うなら<code>false</code>。
     */
    public boolean equals(
        Object o)
    {
        if (!super.equals(o)) return false;
        return ( this.toString( ) == o.toString( ) );
    }

}
