/*
 * This file contains support for adding multiple namelists, mainly 
 * in order to obtain access to kernel type information.
 *
 * Copyright 1999 Silicon Graphics, Inc. All rights reserved.
 */
#include <klib.h>

/* The nmlist[] array contains information about various namelists
 * open for access. The namelist referenced by nmlist[0] is considered
 * the primary namelist. It is used in the setup of the global type 
 * information. The rest of the namelists can be used for accessing 
 * additional type information  At that, only those types which are 
 * not defined in the primary namelist get added to the type tree. 
 */
nmlist_t nmlist[MAXNMLIST];
int numnmlist = 0;
int curnmlist;

int __open_namelist(char *, int);
void init_struct_sizes(void);
int st_setup_typeinfo(void);


/* 
 * kl_set_curnmlist()
 */
void
kl_set_curnmlist(int index)
{
	/* Make sure that there is at least one nmlist record. If there
	 * isn't, then that's not too big a problem, as only some 
	 * functionalty depends upon kernel type information being 
	 * available.
	 */
	if ((index >= 0) && (index < numnmlist)) {
		curnmlist = index;
	}
}

/*
 * kl_open_namelist()
 */
int
kl_open_namelist(char *namelist, int flags)
{
	int saved_nmlist;

	if (numnmlist == 0) {
		/* This is the first one being opened. make sure we 
		 * zero out the table memory.
		 */
		bzero(nmlist, sizeof(nmlist));
	} else if (numnmlist == (MAXNMLIST - 1)) {
		/* There isn't room for this namelist in the table
	 	 */
		return(1);
	}

	/* Set up the current nmlist
	 */
	saved_nmlist = curnmlist;
	curnmlist = numnmlist;
	numnmlist++;
	nmlist[curnmlist].index = curnmlist;
	nmlist[curnmlist].namelist = (char *)kl_str_to_block(namelist, K_PERM);
	nmlist[curnmlist].stringtab = kl_init_string_table(K_PERM);

	/* Open up the namelist 
	 */
	if (__open_namelist(namelist, flags)) {
		numnmlist--;
		nmlist[curnmlist].index = 0;
		if (nmlist[curnmlist].namelist) {
			kl_free_block(nmlist[curnmlist].namelist);
			nmlist[curnmlist].namelist = 0;
		}
		if (nmlist[curnmlist].stringtab) {
			kl_free_string_table(nmlist[curnmlist].stringtab);
			nmlist[curnmlist].stringtab = (string_table_t *)NULL;
		}
		curnmlist = saved_nmlist;
		return(1);
	} 
	if (numnmlist == 1) {
		init_struct_sizes();
	}
	st_setup_typeinfo();
	return(0);
}
