/*-*-c++-*-
 * $Id: qlcrash.cpp,v 1.2 2002/04/19 15:24:17 felfert Exp $
 *
 * This file is part of qlcrash, a GUI for the dump-analysis tool lcrash.
 *
 * Copyright (C) 2001 IBM Deutschland Entwicklung GmbH, IBM Corporation
 *
 * Authors:
 * Michael Geselbracht (let@users.sourceforge.net)
 * Fritz Elfert (elfert@de.ibm.com)
 * Michael Holzheu (holzheu@de.ibm.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or
 * (at your option) any later version. See the file COPYING for more
 * information.
 */
// Qt includes
#include <qvbox.h>
#include <qaccel.h>
#include <qfontdialog.h>
#include <qtimer.h>
#include <qworkspace.h>

#include "qlcrashdoc.h"
#include "qlcrash.h"
#include "cmainwin.h"
#include "crashtypes.h"
#include "cconfigmanager.h"
#include "cpreferencesdialog.h"
#include "cprintexprdialog.h"
#include "cwalkexpressiondialog.h"
#include "cdumpview.h"
#include "ctaskbutton.h"
#include "cgraphobject.h"
#include "ctaskbar.h"
#include "chotkey.h"
#include "chelpabout.h"

#include "layout.xpm"
#include "console.xpm"
#include "tasklist.xpm"
#include "dump.xpm"
#include "print.xpm"
#include "s390dbf.xpm"
#include "trace.xpm"
#include "symbol.xpm"

// Testing
#include "chelpbrowser.h"
#include "cgraphmanager.h"
#include <qlabel.h>

#include "resourceadapter.h"

QLcrashApp::QLcrashApp(CConfigManager* cm)
	: oToolbarInfo(false)
{
  setCaption(tr("QLcrash") + " " + VERSION);
  setIcon(Resource::getBitmap(IDR_PNG7));
  setFocusPolicy(ClickFocus);

  // call inits to invoke all other construction parts
initDocument(cm);
  initView();
  initActions();
  initMenuBar();
  initToolBar();
  initStatusBar();

  connect(oDocument, SIGNAL(sigNewGraph(CListView*)), oMainWin, SLOT(slotAddTree(CListView*)));
  connect(oDocument, SIGNAL(sigTaskList(TaskListType*)), oMainWin, SLOT(slotTaskList(TaskListType*)));
  resize(800, 600);

  oDocument->loadConfig();
  restoreToolBars();
  oAction->loadConfig(oDocument->configManager());
  oDocument->runCrash();

  topLevelWidget()->installEventFilter(this);

  show();

  // Test
  /*
  CGraphManager* gm = new CGraphManager(0);
  gm->show();
  QLabel* l1 = new QLabel("Apple", this);
  QLabel* l2 = new QLabel("Pie", this);
  QLabel* l3 = new QLabel("Juice", this);

  l1->setFont(QFont("Helvetica", 14, QFont::Bold));
  l1->setAlignment(AlignCenter);
  l2->setAlignment(AlignCenter);
  l2->setFont(QFont("Helvetica", 14, QFont::Bold));
  l3->setAlignment(AlignCenter);
  l3->setFont(QFont("Helvetica", 14, QFont::Bold));

  CGraphObject *g1, *g2, *g3;
  g1 = gm->addWidget(l1, "Obj_1");
  g2 = gm->addWidget(l2, "Obj_2");
  g3 = gm->addWidget(l3, "Obj_3");

  gm->attach(g1, g2, "Bake");
  gm->attach(g1, g3, "Sqeeze");

  gm->layout();
  */
}

QLcrashApp::~QLcrashApp()
{
	storeToolBars();
	oAction->storeConfig(oDocument->configManager());
	oDocument->configManager()->saveConfig();
}

void
QLcrashApp::resizeEvent(QResizeEvent* e)
{	
	QMainWindow::resizeEvent(e);
	oTaskBar->resize(childToolbar->width() - oTaskBar->x(), oTaskBar->height());
}

bool
QLcrashApp::eventFilter(QObject* obj, QEvent* e)
{
	if (e->type() == QEvent::Accel) {
		QKeyEvent* ke = static_cast<QKeyEvent*>(e);
		
		switch (ke->key()) {
			case Key_Left:
				oMainWin->scrollLeft();
				break;
			case Key_Right:
				oMainWin->scrollRight();
				break;
			case Key_Up:
				oMainWin->scrollUp();
				break;
			case Key_Down:
				oMainWin->scrollDown();
				break;
		}
	}
	
	return QMainWindow::eventFilter(obj, e);
}

void QLcrashApp::initActions()
{
  // this one is initialized in initS390dbf()
  fileS390dbf = 0;

  oAction = new CHotKey(this);
  oAction->resize(300, 400);
  oAction->action(HK_toggleToolbar)->setToggleAction(true);
  oAction->action(HK_toggleStatusbar)->setToggleAction(true);
  oAction->action(HK_toggleTaskbar)->setToggleAction(true);
  oAction->action(HK_layout)->setEnabled(false);
  oAction->action(HK_findItem)->setEnabled(false);
  connect(oAction, SIGNAL(sigAction(int)), SLOT(slotActionHandler(int)));
/*
  connect(oAction->action(HK_toggleToolbar), SIGNAL(toggled(bool)), SLOT(slotViewToolBar(bool)));
  connect(oAction->action(HK_toggleStatusbar), SIGNAL(toggled(bool)), SLOT(slotViewStatusBar(bool)));
  connect(oAction->action(HK_toggleTaskbar), SIGNAL(toggled(bool)), SLOT(slotViewTaskBar(bool)));
*/
}

void QLcrashApp::initMenuBar()
{
  pFileMenu = new QPopupMenu();
  oAction->action(HK_newConsole)->addTo(pFileMenu);
  oAction->action(HK_taskList)->addTo(pFileMenu);
  oAction->action(HK_newDumpView)->addTo(pFileMenu);
  oAction->action(HK_graphPrint)->addTo(pFileMenu);
  oAction->action(HK_graphWalk)->addTo(pFileMenu);
  oAction->action(HK_graphTrace)->addTo(pFileMenu);
  oAction->action(HK_symbolList)->addTo(pFileMenu);
  oAction->action(HK_symbolListUpdate)->addTo(pFileMenu);
  pFileMenu->insertSeparator();
  oAction->action(HK_restart)->addTo(pFileMenu);
  pFileMenu->insertSeparator();
  oAction->action(HK_exit)->addTo(pFileMenu);

  pViewMenu = new QPopupMenu();
  oAction->action(HK_layout)->addTo(pViewMenu);
  oAction->action(HK_findItem)->addTo(pViewMenu);
  pViewMenu->insertSeparator();
  pViewMenu->setCheckable(true);
  oAction->action(HK_toggleToolbar)->addTo(pViewMenu);
  oAction->action(HK_toggleStatusbar)->addTo(pViewMenu);
  oAction->action(HK_toggleTaskbar)->addTo(pViewMenu);

  pOptionMenu = new QPopupMenu();
  oAction->action(HK_options)->addTo(pOptionMenu);
  oAction->action(HK_keyBindings)->addTo(pOptionMenu);

  pWindowMenu = new QPopupMenu(this);
  oAction->action(HK_windowTile)->addTo(pWindowMenu);
  oAction->action(HK_windowCascade)->addTo(pWindowMenu);
  pWindowMenu->insertSeparator();
//  pWindowMenu->setCheckable(true);
//  connect(pWindowMenu, SIGNAL(activated(int)), this, SLOT(slotWindowMenuActivated(int)));

  pHelpMenu = new QPopupMenu();
  oAction->action(HK_helpManual)->addTo(pHelpMenu);
  oAction->action(HK_helpAbout)->addTo(pHelpMenu);
  pHelpMenu->insertSeparator();
  // pHelpMenu->insertItem(tr("What's &This"), this, SLOT(whatsThis()), SHIFT+Key_F1);


  menuBar()->insertItem(tr("&File"), pFileMenu);
  menuBar()->insertItem(tr("&View"), pViewMenu);
  menuBar()->insertItem(tr("&Option"), pOptionMenu);
  menuBar()->insertItem(tr("&Window"), pWindowMenu);
  menuBar()->insertItem(tr("&Help"), pHelpMenu);
}

void QLcrashApp::initToolBar()
{
  fileToolbar = new QToolBar(this, "file operations");
  QPixmap pm;

  pm = QPixmap(tasklist_xpm);
  new QToolButton(
  	pm,
  	tr("Display task list"),
  	QString::null,
  	this,
  	SLOT(slotTaskList()),
  	fileToolbar
  );

  pm = QPixmap(layout_xpm);
  new QToolButton(
  	pm,
  	tr("Layout objects"),
  	QString::null,
  	this,
  	SLOT(slotViewLayout()),
  	fileToolbar
  );

  pm = QPixmap(console_xpm);
  new QToolButton(
  	pm,
  	tr("New console"),
  	QString::null,
  	oMainWin,
  	SLOT(slotNewConsole()),
  	fileToolbar
  );

  pm = QPixmap(dump_xpm);
  new QToolButton(
  	pm,
  	tr("Dump view"),
  	QString::null,
  	this,
  	SLOT(slotFileNewDump()),
  	fileToolbar
  );

  pm = QPixmap(print_xpm);
  new QToolButton(
  	pm,
  	tr("Graph print"),
  	QString::null,
  	this,
  	SLOT(slotGraphPrint()),
  	fileToolbar
  );

  pm = QPixmap(trace_xpm);
  new QToolButton(
  	pm,
  	tr("Stack trace"),
  	QString::null,
  	oMainWin,
  	SLOT(slotTraceList()),
  	fileToolbar
  );

  pm = QPixmap(symbol_xpm);
  new QToolButton(
  	pm,
  	tr("Global symbols"),
  	QString::null,
  	oMainWin,
  	SLOT(slotSymtabList()),
  	fileToolbar
  );

  /*
  fileToolbar->addSeparator();
  QWhatsThis::whatsThisButton(fileToolbar);
  */

  childToolbar = new QToolBar(tr("ChildBar"), this, Top, true);
  childToolbar->setHorizontalStretchable(true);
  setDockEnabled(childToolbar, Left, false);
  setDockEnabled(childToolbar, Right, false);
  oTaskBar = new CTaskBar(childToolbar);
  connect(oTaskBar, SIGNAL(sigWidgetRemoved(QWidget*)), SLOT(slotRemovedChildWin(QWidget*)));

  // hide all toolbars
  childToolbar->hide();
  fileToolbar->hide();
  statusBar()->hide();
}

void
QLcrashApp::initStatusBar()
{
  statusBar()->message(tr("Ready."));
}

void
QLcrashApp::initView()
{
	oMainWin = new CMainWin(this);
	setCentralWidget(oMainWin);
	
	connect(oMainWin, SIGNAL(sigNewChildWin(QWidget*)), SLOT(slotNewChildWin(QWidget*)));
	connect(oMainWin->workspace(), SIGNAL(windowActivated(QWidget*)), SLOT(slotWindowActivated(QWidget*)));
}

void
QLcrashApp::initDocument(CConfigManager* cm)
{
	oDocument = new QLcrashDoc(cm, this);
	connect(oDocument, SIGNAL(sigLCrashReady()), SLOT(slotLCrashReady()));
	connect(oDocument, SIGNAL(sigBusy(bool)), SLOT(slotBusy(bool)));
	connect(oDocument, SIGNAL(sigExit(bool)), SLOT(slotExit(bool)));
}

void
QLcrashApp::slotExit(bool ask)
{
	if (ask) {
		if (QMessageBox::information(0,
			    stdCaption(tr("Quit?")),
			    tr("Do you really want to quit?"),
			    tr("Yes"), tr("No")) == 1)
		    return;
	}
	qApp->quit();
}

void
QLcrashApp::slotFileQuit()
{ 
	statusBar()->message(tr("Exiting application..."));
	slotExit(true);
	statusBar()->message(tr("Ready."));
}

void
QLcrashApp::slotFileNewDump()
{
	if (!oDocument->waitForInit()) {
#ifdef WIN32
		CDumpView* dv = new CDumpView((UINT64)0x1000, this, 0, 0, WDestructiveClose);
#else
		CDumpView* dv = new CDumpView(0x1000llu, this, 0, 0, WDestructiveClose);
#endif
		slotNewChildWin(dv);
	}
}

void
QLcrashApp::slotViewToolBar(bool toggle)
{
  statusBar()->message(tr("Toggle toolbar..."));

  if (toggle== false)
  {
    fileToolbar->hide();
  }
  else
  {
    fileToolbar->show();
  };

  oDocument->configManager()->setItem(CFG_GRP_TOOLBAR, CFG_ATT_TOOLBAR_MAIN, (int) toggle);

  statusBar()->message(tr("Ready."));
}

void
QLcrashApp::slotViewStatusBar(bool toggle)
{
  statusBar()->message(tr("Toggle statusbar..."));
  ///////////////////////////////////////////////////////////////////
  //turn Statusbar on or off
  
  if (toggle == false)
  {
    statusBar()->hide();
  }
  else
  {
    statusBar()->show();
  }

  oDocument->configManager()->setItem(CFG_GRP_TOOLBAR, CFG_ATT_TOOLBAR_STATUS, (int) toggle);

  statusBar()->message(tr("Ready."));
}

void
QLcrashApp::slotViewTaskBar(bool toggle)
{
	statusBar()->message(tr("Toggle childbar..."));
	if (toggle == false) {
		childToolbar->hide();
	}
	else {
		childToolbar->show();
	}
	
	oDocument->configManager()->setItem(CFG_GRP_TOOLBAR, CFG_ATT_TOOLBAR_CHILD, (int) toggle);
	
	statusBar()->message(tr("Ready."));
}

void
QLcrashApp::slotViewLayout()
{
	oMainWin->layoutActiveWindow();
}

void
QLcrashApp::slotViewNextWindow()
{
	QList<QWidget> list = oMainWin->workspace()->windowList();
	QWidget* cur = oMainWin->workspace()->activeWindow();
	
	if (list.count() > 1 && cur != 0) {
		QWidget* walker = list.first();
		
		while (walker != 0) {
			if (walker == cur) {
				walker = list.next();
				if (walker == 0) {
					walker = list.first();
				}
				
				walker->setFocus();
				break;
			}
			
			walker = list.next();
		}
	}
}

void
QLcrashApp::slotViewPrevWindow()
{
	QList<QWidget> list = oMainWin->workspace()->windowList();
	QWidget* cur = oMainWin->workspace()->activeWindow();
	
	if (list.count() > 1 && cur != 0) {
		QWidget* walker = list.last();
		
		while (walker != 0) {
			if (walker == cur) {
				walker = list.prev();
				if (walker == 0) {
					walker = list.last();
				}
				
				walker->setFocus();
				break;
			}
			
			walker = list.prev();
		}
	}
}

void
QLcrashApp::slotViewLowerCurrent()
{
	QList<QWidget> list = oMainWin->workspace()->windowList();
	QWidget* w = oMainWin->workspace()->activeWindow();
	if (w != 0 && list.count() > 1) {
		list.find(w);
		assert(list.current() != 0);
		QWidget* walker = list.prev();
		
		while (walker != w) {
			if (walker == 0) {
				walker = list.last();
				if (walker == w) {
					break;
				}
			}
			walker->setFocus();
			walker = list.prev();
		}
	}
}

void
QLcrashApp::slotTaskList()
{
	oDocument->slotFromConsole(0, "graph task\n");
}

void
QLcrashApp::slotGraphPrint()
{
	oMainWin->graphPrintDialog();
}

void
QLcrashApp::slotGraphWalk()
{
	oMainWin->graphWalkDialog();
}

void
QLcrashApp::slotLCrashReady()
{
	if (fileS390dbf == 0) {
		initS390dbf();
	}
	
	// initialize command completion
	oMainWin->initCommandCompletion();
}

void
QLcrashApp::slotHelpAbout()
{
	CHelpAbout* ha = new CHelpAbout(this);
	ha->exec();
	delete ha;
}

void
QLcrashApp::slotStatusHelpMsg(const QString &text)
{
  statusBar()->message(text, 2000);
}

void
QLcrashApp::slotOptions()
{
	CPreferencesDialog* pd = new CPreferencesDialog(oDocument, this);
	int ret = pd->exec();
	
	if (ret == QDialog::Accepted) {
		if (pd->fontChanged()) {
			oMainWin->setConsoleFont(pd->font());
		}
	}
	
	delete pd;
}

void
QLcrashApp::slotKeyBindings()
{
	oAction->exec();
}

void
QLcrashApp::slotWindowTile()
{
	oMainWin->workspace()->tile();
}

void
QLcrashApp::slotWindowCascade()
{
	oMainWin->workspace()->cascade();
}

void
QLcrashApp::slotWindowMenuActivated(int id)
{
	QString str = pWindowMenu->text(id);
	QList<QWidget> list = oMainWin->workspace()->windowList();
	QWidget* walker = list.first();
	while (walker != 0) {
		if (walker->caption() == str) {
			walker->setFocus();
			break;
		}
		walker = list.next();
	}
}

void
QLcrashApp::showToolbar(QToolBar* bar, bool main)
{
	if (oToolbarInfo) {
		if (main) {
			moveToolBar(bar, oMainbarEdge, oMainbarNL, oMainbarIndex, oMainbarOffset);
		}
		else {
			moveToolBar(bar, oNavbarEdge, oNavbarNL, oNavbarIndex, oNavbarOffset);
		}
	}
	
	bar->show();
}

void
QLcrashApp::hideToolbar(QToolBar* bar, bool main)
{	
	if (main) {
		getLocation(bar, oMainbarEdge, oMainbarIndex, oMainbarNL, oMainbarOffset);
	}
	else {
		getLocation(bar, oNavbarEdge, oNavbarIndex, oNavbarNL, oNavbarOffset);
	}
	oToolbarInfo = true;
	bar->hide();
}

void
QLcrashApp::initS390dbf()
{
	if (oDocument->haveDebugFeature()) {
		fileS390dbf = new QAction(tr("Debug feature"), tr("Debug &feature"), 0, this);
		fileS390dbf->setStatusTip(tr("Display an s390dbf window"));
		fileS390dbf->setWhatsThis(tr("Debug Feature\n\nDisplays a file manager like s390dbf window"));
		connect(fileS390dbf, SIGNAL(activated()), oMainWin, SLOT(slotNewS390dbf()));
		
		// fileS390dbf->addTo(pFileMenu);
		
	  	QPixmap pm = QPixmap(s390dbf_xpm);
	  	new QToolButton(
	  		pm,
  			tr("Debug feature"),
  			QString::null,
	  		oMainWin,
  			SLOT(slotNewS390dbf()),
  			fileToolbar
	  	);
	}
}

void
QLcrashApp::slotHelpManual()
{
	CHelpBrowser* hb = new CHelpBrowser(oDocument, 0);
	hb->setFont(font());
	hb->show();
	hb->resize(800, 500);
}

void
QLcrashApp::slotNewChildWin(QWidget* child)
{
	oTaskBar->addWidget(child);
	pWindowMenu->insertItem(child->caption());
}

void
QLcrashApp::slotRemovedChildWin(QWidget* child)
{
	unsigned int count = pWindowMenu->count();
	for (unsigned int i = 0; i < count; i++) {
		if (child->caption() == pWindowMenu->text(pWindowMenu->idAt(i))) {
			pWindowMenu->removeItemAt(i);
			break;
		}
	}
	
	if (oMainWin->workspace()->activeWindow() == 0) {
		slotWindowActivated(0);
	}
}

void
QLcrashApp::restoreToolBars()
{
	ToolBarDock dock;
	bool nl;
	int index;
	int extraOffset;
	
	// get coords for main toolbar
	getLocation(fileToolbar, dock, index, nl, extraOffset);
	
	CConfigManager* cm = oDocument->configManager();
	moveToolBar(
		fileToolbar,
		(ToolBarDock) cm->item(CFG_GRP_TOOLBAR, CFG_ATT_TOOLBAR_MAINEDGE, dock),
		cm->item(CFG_GRP_TOOLBAR, CFG_ATT_TOOLBAR_MAINNL, nl),
		cm->item(CFG_GRP_TOOLBAR, CFG_ATT_TOOLBAR_MAININDEX, index),
		cm->item(CFG_GRP_TOOLBAR, CFG_ATT_TOOLBAR_MAINEXTRA, extraOffset)
	);
	
	// taskbar
	getLocation(childToolbar, dock, index, nl, extraOffset);
	moveToolBar(
		childToolbar,
		(ToolBarDock) cm->item(CFG_GRP_TOOLBAR, CFG_ATT_TOOLBAR_CHILDEDGE, dock),
		cm->item(CFG_GRP_TOOLBAR, CFG_ATT_TOOLBAR_CHILDNL, nl),
		cm->item(CFG_GRP_TOOLBAR, CFG_ATT_TOOLBAR_CHILDINDEX, index),
		cm->item(CFG_GRP_TOOLBAR, CFG_ATT_TOOLBAR_CHILDEXTRA, extraOffset)
	);
	
	oAction->action(HK_toggleToolbar)->setOn((bool)cm->item(CFG_GRP_TOOLBAR, CFG_ATT_TOOLBAR_MAIN, 1));
	oAction->action(HK_toggleStatusbar)->setOn((bool)cm->item(CFG_GRP_TOOLBAR, CFG_ATT_TOOLBAR_STATUS, 1));
	oAction->action(HK_toggleTaskbar)->setOn((bool)cm->item(CFG_GRP_TOOLBAR, CFG_ATT_TOOLBAR_CHILD, 1));
	
	// setting the tool bar does not emit any signals, so emulate these signals
	slotActionHandler(HK_toggleToolbar);
	slotActionHandler(HK_toggleStatusbar);
	slotActionHandler(HK_toggleTaskbar);
}

void
QLcrashApp::storeToolBars()
{
	ToolBarDock dock;
	bool nl;
	int index;
	int extraOffset;
	CConfigManager* cm = oDocument->configManager();
	
	// main toolbar
	getLocation(fileToolbar, dock, index, nl, extraOffset);
	cm->setItem(CFG_GRP_TOOLBAR, CFG_ATT_TOOLBAR_MAINEDGE, dock);
	cm->setItem(CFG_GRP_TOOLBAR, CFG_ATT_TOOLBAR_MAINNL, nl);
	cm->setItem(CFG_GRP_TOOLBAR, CFG_ATT_TOOLBAR_MAININDEX, index);
	cm->setItem(CFG_GRP_TOOLBAR, CFG_ATT_TOOLBAR_MAINEXTRA, extraOffset);
	
	// taskbar
	getLocation(childToolbar, dock, index, nl, extraOffset);
	cm->setItem(CFG_GRP_TOOLBAR, CFG_ATT_TOOLBAR_CHILDEDGE, dock);
	cm->setItem(CFG_GRP_TOOLBAR, CFG_ATT_TOOLBAR_CHILDNL, nl);
	cm->setItem(CFG_GRP_TOOLBAR, CFG_ATT_TOOLBAR_CHILDINDEX, index);
	cm->setItem(CFG_GRP_TOOLBAR, CFG_ATT_TOOLBAR_CHILDEXTRA, extraOffset);
}

void
QLcrashApp::slotActionHandler(int id)
{
	switch (id) {
		case HK_newConsole:
			oMainWin->slotNewConsole();
			break;
		case HK_taskList:
			slotTaskList();
			break;
		case HK_layout:
			slotViewLayout();
			break;
		case HK_graphPrint:
			slotGraphPrint();
			break;
		case HK_graphWalk:
			slotGraphWalk();
			break;
		case HK_graphTrace:
			oMainWin->graphTrace();
			break;
		case HK_symbolList:
			oMainWin->slotSymtabList();
			break;
		case HK_symbolListUpdate:
			oMainWin->slotSymtabListUpdate();
			break;
		case HK_switchMDI:
			slotViewNextWindow();
			break;
		case HK_switchMDIrev:
			slotViewPrevWindow();
			break;
		case HK_exit:
			slotFileQuit();
			break;
		case HK_newDumpView:
			slotFileNewDump();
			break;
		case HK_restart:
			oDocument->slotRestart();
			break;
		case HK_toggleToolbar:
			slotViewToolBar(oAction->action(HK_toggleToolbar)->isOn());
			break;
		case HK_toggleStatusbar:
			slotViewStatusBar(oAction->action(HK_toggleStatusbar)->isOn());
			break;
		case HK_toggleTaskbar:
			slotViewTaskBar(oAction->action(HK_toggleTaskbar)->isOn());
			break;
		case HK_findItem:
			oMainWin->findItem();
			break;
		case HK_options:
			slotOptions();
			break;
		case HK_keyBindings:
			slotKeyBindings();
			break;
		case HK_windowCascade:
			slotWindowCascade();
			break;
		case HK_windowTile:
			slotWindowTile();
			break;
		case HK_windowLower:
			slotViewLowerCurrent();
			break;
		case HK_helpAbout:
			slotHelpAbout();
			break;
		case HK_helpManual:
			slotHelpManual();
			break;
	}
}

void
QLcrashApp::slotBusy(bool w)
{
	if (w) {
		setCursor(waitCursor);
		oMainWin->setCursor(waitCursor);
		oBusy = true;
		
		// update actions
		oAction->action(HK_newConsole)->setEnabled(false);
		oAction->action(HK_taskList)->setEnabled(false);
		oAction->action(HK_layout)->setEnabled(false);
		oAction->action(HK_newDumpView)->setEnabled(false);
		oAction->action(HK_graphPrint)->setEnabled(false);
		oAction->action(HK_graphTrace)->setEnabled(false);
	}
	else {
		QTimer::singleShot(500, this, SLOT(slotBusyTimeout()));
	}
}

void
QLcrashApp::slotBusyTimeout()
{
	setCursor(arrowCursor);
	oMainWin->unsetCursor();
	oBusy = false;

	oAction->action(HK_newConsole)->setEnabled(true);
	oAction->action(HK_taskList)->setEnabled(true);
	oAction->action(HK_newDumpView)->setEnabled(true);
	oAction->action(HK_graphPrint)->setEnabled(true);
	oAction->action(HK_graphTrace)->setEnabled(true);
	
	// restore other actions depending on the active (MDI-) window
	slotWindowActivated(oMainWin->workspace()->activeWindow());
}

void
QLcrashApp::slotWindowActivated(QWidget* w)
{
	bool findItem = false;
	bool layout = false;
	
	if (w) {		
		if (w->isA("CGraphManager")) {
			findItem = true;
			layout = true;
		}
		else {
			findItem |= false;
			layout = false;
		}
		findItem |= w->isA("CListView");
	}
		
	oAction->action(HK_findItem)->setEnabled(findItem);
	oAction->action(HK_layout)->setEnabled(layout);
}
