/************************************************************
 *
 * COPYRIGHT (C) HITACHI,LTD. 2003-2004
 * WRITTEN BY HITACHI SYSTEMS DEVELOPMENT LABORATORY,
 *            HITACHI CENTRAL RESEARCH LABORATORY.
 *
 * Created by S.Oshima <oshima@sdl.hitachi.co.jp>
 * Updated by M.Hiramatsu <hiramatu@sdl.hitachi.co.jp>
 *
 ************************************************************/

#include "lkstdogkeeper.h"

void syslog_info(const char *format, ...);
void syslog_warning(const char *format, ...);
void syslog_error(const char *format, ...);
void stderr_info(const char *format, ...);
void stderr_warning(const char *format, ...);
void stderr_error(const char *format, ...);
void signal_handler(int signal_no);
int create_pid_file(const char *dname);
int delete_pid_file(void);
int daemonize(const char *dname);
int lkst_evhandler_getid(int fd, const char *name);
void cleanup(void);
void parse_options(int argc, char *argv[]);

void (*infolog)(const char *format, ...) = stderr_info;
void (*warnlog)(const char *format, ...) = stderr_warning;
void (*errlog)(const char *format, ...)  = stderr_error;

#define DEFAULT_SLEEP_SEC 5
#define DEFAULT_SLEEP_MAX 20

unsigned sleep_sec = DEFAULT_SLEEP_SEC;
unsigned sleep_max = DEFAULT_SLEEP_MAX;
char *dump_file = NULL;
int debug_mode = 0;

/* lkstdogkeeper [-dh] [-s sleep_time] [-x max_time] [-f dump_file]
 * or lkstdogkeeper dump_file
 */
void usage(void)
{
	printf("lkstdogkeeper - lkst heartbeat daemon\n");
	printf("Usage: lkstdogkeeper [-dh] [-s sleep_time] [-x max_time] [-f dump_file]\n");
	printf("       lkstdogkeeper taskdumpfile \n");
}

void parse_options(int argc, char *argv[])
{
	int c;
	long tmp;
	
	while ( (c=getopt(argc, argv, "dhs:x:f:")) != EOF) {
		switch (c) {
		case 'd':
			debug_mode = 1;
			break;
		case 's':
			tmp = strtol(optarg,NULL,10);
			if (tmp < 0) {
				printf("Error: sleep second must be positive.");
				exit(1);
			}
			sleep_sec = (unsigned)tmp;
			break;
		case 'x':
			tmp = strtol(optarg,NULL,10);
			if (tmp < 0) {
				printf("Error: max sleep second must be positive.");
				exit(1);
			}
			sleep_max = (unsigned)tmp;
			break;
		case 'f':
			if (dump_file) free(dump_file);
			dump_file = strdup(optarg);
			break;
		default:
			printf("Error: unknown options(%c)\n", optopt);
		case 'h':
			usage();
			exit(1);
		}
	}
}

int __inline__ open_dev(void)
{
	int fd;
	if ((fd = open("/dev/lkst", O_RDONLY)) < 0) {
		errlog("Cannot open /dev/lkst!; %s\n",strerror(errno));
		exit(1);
	}
	return fd;
}

int devfd;
lkst_evhandler_id dog_id;

int dump_tasks(char * fname)
{
	FILE *fp;
	struct taskparam param;
	int retval;
	struct lkst_evhandler_ctrl_param evhandler_ctrl_param;
	
	fp = fopen(fname,"w");
	if ( fp == NULL ){
		errlog("error: cannot open %s\n",fname);
		return -1;
	}

	retval = lkst_evhandler_getid(devfd, PUPPYNAME);
	if (retval < 0 || retval == LKST_EVHANDLER_ID_VOID){
		errlog("error: %s is not found\n",PUPPYNAME);
		return -1;
	}
	dog_id = (lkst_evhandler_id)retval;
	
	evhandler_ctrl_param.id = dog_id;
	param.size = 8192*256;
	param.addr = malloc(param.size);
	evhandler_ctrl_param.buf = &param;
	evhandler_ctrl_param.bufsize = sizeof(param);
	retval = ioctl(devfd, LKST_IOC_EVHANDLER_CTRL, &evhandler_ctrl_param);
	if (retval < 0) {
		errlog("ioctl(LKST_IOC_EVHANDLER_CTRL) error: %s\n", strerror(errno));
		return -1;
	}
	infolog("%d bytes wrote to %s\n", param.size, fname);
	fwrite(param.addr, 1, param.size, fp);
	fclose(fp);
	close(devfd);
	return 0;
}

int main(int argc, char *argv[])
{
	int retval;
	struct lkst_evhandler_ctrl_param evhandler_ctrl_param;
	char buf[16];
	size_t bufsize = 16;

	if (argc == 2 && argv[1][0]!='-' ) { /*TODO*/
		return dump_tasks(argv[1]);
	} else {
		parse_options(argc, argv);
	}

	if( daemonize(PROGNAME) ) return 1;

	devfd = open_dev();
	retval = lkst_evhandler_getid(devfd, DOGNAME);
	if (retval < 0 || retval == LKST_EVHANDLER_ID_VOID){
		errlog("error: %s is not found\n", DOGNAME);
		exit(1);
	}
	dog_id = (lkst_evhandler_id)retval;
	if (debug_mode)
		signal(SIGINT, signal_handler);
	signal(SIGTERM, signal_handler);
	signal(SIGPIPE, signal_handler);
	signal(SIGHUP, signal_handler);

// retry:
	snprintf(buf, 15, "%d\n", sleep_max);
	evhandler_ctrl_param.id = dog_id;
	evhandler_ctrl_param.buf = buf;
	evhandler_ctrl_param.bufsize = bufsize;

	retval = ioctl(devfd, LKST_IOC_EVHANDLER_CTRL, &evhandler_ctrl_param);
	if (retval<0) {
		errlog("ioctl(LKST_IOC_EVHANDLER_CTRL) error: %s\n", strerror(errno));
		exit(1);
	}

	snprintf(buf, 15, "%d\n", 0);
	evhandler_ctrl_param.id = dog_id;

	while(1){
		retval = ioctl(devfd, LKST_IOC_EVHANDLER_CTRL, &evhandler_ctrl_param);
		if (retval < 0) {
			errlog("ioctl(LKST_IOC_EVHANDLER_CTRL) error: %s\n", strerror(errno));
			exit(1);
		}else if (retval == 1) {
			infolog("watchdog slowdown detected\n");
			if (dump_file) {
				dump_tasks(dump_file);
				//goto retry;
			}
			break;
		}
 		sleep(sleep_sec);
 	}

	infolog("exit normally\n");
	close(devfd);

	return 1;
}

int daemonize(const char *dname)
{
	if ( getuid()!=0 ) {
		errlog("Must be root.\n Program aborted.\n");
		return 1;
	}
	if (!debug_mode) {
		infolog("daemonizing ... \n");
		/* open syslog */
		openlog(dname, LOG_PID, LOG_DAEMON);
		errlog = syslog_error;
		warnlog = syslog_warning;
		infolog = syslog_info;
		if ( daemon(0,0) < 0) {
			errlog("Failed to daemonize\n");
			return 1;
		}
	}

	if (atexit(cleanup)) {
		errlog("Failed to register exit function:%s\n", strerror(errno));
		return 1;
	}
	if (create_pid_file(dname) < 0) {
		errlog("Failed to create pid-file\n");
		return 1;
	}
	return 0;
}

/*write a log message to syslog*/
void syslog_info(const char *format, ...)
{
	va_list ap;
	va_start(ap,format);
	vsyslog(LOG_INFO, format, ap);
}
void syslog_warning(const char *format, ...)
{
	va_list ap;
	va_start(ap,format);
	vsyslog(LOG_WARNING, format, ap);
}
void syslog_error(const char *format, ...)
{
	va_list ap;
	va_start(ap,format);
	vsyslog(LOG_ERR, format, ap);
}
/* write a log message to stderr*/
void stderr_info(const char *format, ...)
{
	va_list ap;
	va_start(ap,format);
	fprintf(stderr, PROGNAME" info: ");
	vfprintf(stderr, format, ap);
}
void stderr_warning(const char *format, ...)
{
	va_list ap;
	va_start(ap,format);
	fprintf(stderr, PROGNAME" warning: ");
	vfprintf(stderr, format, ap);
}
void stderr_error(const char *format, ...)
{
	va_list ap;
	va_start(ap,format);
	fprintf(stderr, PROGNAME" error: ");
	vfprintf(stderr, format, ap);
}

int lkst_evhandler_getid(int fd, const char *name)
{
	int ret;
	struct lkst_evhandler_getparam egp;
	if(!name || name[0]=='\0') return -EINVAL;
	strncpy(egp.name, name, LKST_EVHANDLER_NAME_LEN-1);
	if( (ret = ioctl(fd, LKST_IOC_EVHANDLER_GETID, &egp)) < 0)
		return ret;
	return egp.id;
}

void signal_handler(int signal_no)
{
	int retval;
	struct lkst_evhandler_ctrl_param evhandler_ctrl_param;
	char buf3[] = {"-1"};	/* Watchdog timer stop command */
	size_t bufsize3 = sizeof(buf3);

	evhandler_ctrl_param.id = dog_id;
	evhandler_ctrl_param.buf = buf3;
	evhandler_ctrl_param.bufsize = bufsize3;

	retval = ioctl(devfd, LKST_IOC_EVHANDLER_CTRL, &evhandler_ctrl_param);
						/* evhandler_ctrl */
	if (retval) {
		errlog("ioctl(LKST_IOC_EVHANDLER_CTRL) error: %s\n", strerror(errno));
		exit(1);
	}
	
	exit(0);
}

void cleanup(void)
{
	delete_pid_file();
}
#define PID_PREFIX "/var/run/"

static char pid_file[256]="";

int create_pid_file(const char * dname)
{
	FILE *pfd;
	if (pid_file[0] != '\0') return 1;
	if (strlen(dname) >= 128) return -1; 
	snprintf(pid_file, 255, "%s%s.pid", PID_PREFIX, dname);

	if ( (pfd = fopen(pid_file, "r")) != NULL){
		errlog( "%s exist, another %s is already running.\n",
			pid_file, dname);
		infolog( "Please check if another lkstlogd is working.\n");
		pid_file[0] = '\0';
		return -1;
	}
	if ( (pfd = fopen(pid_file, "w")) == NULL){
		errlog( "Can't open %s : %s\n", pid_file,strerror(errno));
		infolog("Please check if you have write permission to %s.", 
			pid_file);
		pid_file[0] = '\0';
		return -2;
	}
	if (!fprintf(pfd,"%d\n", getpid())) {
		errlog( "Can't write %s : %s\n", pid_file,strerror(errno));
		fclose(pfd);
		return -3;
	}
	fclose(pfd);
	return 0;
}

int delete_pid_file(void)
{
	if(pid_file[0] == '\0') return 0;
	/* pid_file remove */
	if ( unlink((char *)pid_file) != 0){
		errlog("Can't remove %s.\n", pid_file);
		return -1;
	}
	pid_file[0]='\0';
	return 0;
}
