/*
 bio processing time analyzer

 Copyright (C) HITACHI,LTD. 2005
 WRITTEN BY HITACHI SYSTEMS DEVELOPMENT LABORATORY,
 Created by M.Hiramatsu <hiramatu@sdl.hitachi.co.jp>
 Updated by H.Kawai <h-kawai@sdl.hitachi.co.jp>

 The development of this program is partly supported by IPA
 (Information-Technology Promotion Agency, Japan).
  
 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

#include <stdio.h>
#include <errno.h>
#include <linux/lkst.h>
#include <linux/lkst_buffer.h>
#include <lkstla.h>

static slot_hkey info_key_biotime(struct lkst_log_record *rec_start, struct lkst_log_record *rec_end);
static slot_hkey session_key_biotime(struct lkst_log_record *rec);
static char * get_alias_biotime(slot_hkey key);
static ga_type_t get_type_biotime(struct lkst_log_record *);

static int bio_key=0;
extern struct gate_analyzer biotime_analyzer;

static int extra_opt_handler_bio(int c, char*v) {
	bio_key = 1;

	SET_COL(biotime_analyzer, COL_KEY, "address", ADDR_LEN, "%0*lx");
	SET_COL(biotime_analyzer, COL_ALIAS, "bio-addr", ADDR_LEN, "%*s");

	return 0;
}

static struct command_option bio_option = {
	.opt = "B",
	.format = "-B",
	.description = "show BIO addr instead of kdevice",
	.handler = extra_opt_handler_bio,
};


struct gate_analyzer biotime_analyzer = {
	.name = "biotime",
	.description = "BIO processing time analyzer",
	.cols = {
		[COL_KEY] = { "type", 8, "%0*x" },
		[COL_ALIAS] = { "kdevice-rw", 11, "%*s" },
		[COL_METRIC] = { "blockio-time" },
	},

	GA_INIT_SESSION_ANALYZER
	GA_INIT_1TO1_SESSION
	GA_INIT_RET_SESSION_TIME
	GA_INIT_RET_EVENT_PID

	.info_key = info_key_biotime,
	.session_key = session_key_biotime,
	.get_alias = get_alias_biotime,

	.get_type = get_type_biotime,

	.init = init_none,
	.nr_options = 1,
	.options = {&bio_option},
};

static slot_hkey info_key_biotime(struct lkst_log_record *rec_start,
				  struct lkst_log_record *rec_end)
{
	slot_hkey key;

	if (bio_key)
		key = ARG1UL(rec_start);
	else
		key = (rec_start->log_arg4 & 0xffffffff) |
			((rec_start->log_arg3 & 1) << 31);
	return key;
}

static slot_hkey session_key_biotime(struct lkst_log_record *rec)
{
	return ARG1UL(rec);
}

static char buf[256];

static char * get_alias_biotime(slot_hkey key)
{
	if (!bio_key)
		sprintf(buf,"%08x-%s", (unsigned int)key & 0x7fffffff,
			(key >> 31)? "WR": "RD");
	else
		sprintf(buf, "%0*lx", ADDR_LEN, key & ULMASK);
		
	return buf;
}

static ga_type_t get_type_biotime(struct lkst_log_record *rec)
{
	switch(rec->posix.log_event_type){
	case LKST_ETYPE_BIO_MAKE_REQ:
		return GAT_START;
	case LKST_ETYPE_BIO_END_IO:
		return GAT_END;
	default:
		return GAT_IGNORE;
	}
}

