/*
 LKST log formatter

 Copyright (C) HITACHI,LTD. 2004-2005
 WRITTEN BY HITACHI SYSTEMS DEVELOPMENT LABORATORY,
 Created by M.Hiramatsu <hiramatu@sdl.hitachi.co.jp>
 Updated by H.Kawai <h-kawai@sdl.hitachi.co.jp>
  
 The development of this program is partly supported by IPA
 (Information-Technology Promotion Agency, Japan).

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <errno.h>
#include <lkstla.h>

static int pre_analyze_log(struct gate_analyzer *ga);
static void callback_log(struct gate_analyzer *ga, slot_hkey key,
			 struct timespec *ets, double metric,int pid);
static void post_analyze_log(struct gate_analyzer *ga);
static int opt_handler_log(int c, char *opt);

struct analysis_formatter log_formatter = {
        .opt = { 
		.opt = "l::",
		.format = "-l[P][N][C][E][base-time]",
		.description = "show logs. P:show PID, N: show command name, "
		               "C:show as CSV, E: show elapsed time",
		.handler = opt_handler_log,
	},
	.active = 0,
	.pre_analyze = pre_analyze_log,
	.callback = callback_log,
	.post_analyze = post_analyze_log,
};

static int pid_show = 0;
static int comm_show = 0;
static int csv_char = ' ';
static int elapsed_show = 0;
static struct timespec start_ts = {0,0};
#include <ctype.h>

static int opt_handler_log(int c, char *opt)
{
	if (opt) {
		char * ptr;
		double start;
		while (*opt!='\0') {
			switch (*opt) {
			case 'C':
				csv_char=',';
				opt++;
				break;
			case 'E':
				elapsed_show = 1;
				opt++;
				break;
			case 'N':
				comm_show = 1;
				opt++;
				break;
			case 'P':
				pid_show = 1;
				opt++;
				break;
			default:
				start = strtod(opt,&ptr);
				if (ptr == opt) goto fmt_error;
				opt = ptr;
				start_ts.tv_sec = (long)start;
				start -= (double)start_ts.tv_sec;
				start *= (double)1000*1000*1000;
				start_ts.tv_nsec = (long)start;
				break;
			}
		}
	}
	if (elapsed_show && (start_ts.tv_sec != 0 || start_ts.tv_nsec != 0)) {
		goto fmt_error;
	}

	log_formatter.active = 1;
	return 0;
fmt_error:
	return -1;
}

static int pre_analyze_log(struct gate_analyzer *ga)
{
	int n;
	char pid_format[8];
	char comm_format[8];

	if (pid_show)
		sprintf(pid_format, "%%5d%c", csv_char);
	else
		pid_format[0] = '\0';

	if (comm_show)
		sprintf(comm_format, "%%-16s%c", csv_char);
	else
		comm_format[0] = '\0';

	n = snprintf(ga->format_data, sizeof(ga->format_data),
		     "%s%c%s%c%s%s%%s%c%s\n",
		     ga->cols[COL_KEY].format, csv_char,
		     ga->cols[COL_ALIAS].format, csv_char,
		     pid_format, comm_format,
		     csv_char,
		     "%15.9f"); // metric

	if (n >= sizeof(ga->format_data))
		return -1; 

	/* if quiet_mode is true, do not print title */
	if (quiet_mode)
		return 0;

	// print title
	printf("%s\n",ga->description);
	printf("%*s%c%*s%c", ga->cols[COL_KEY].width, ga->cols[COL_KEY].title,
	       csv_char, ga->cols[COL_ALIAS].width, ga->cols[COL_ALIAS].title,
	       csv_char);
	if (pid_show)
		printf("  pid%c", csv_char);
	if (comm_show)
		printf("comm            %c", csv_char);
	printf("%20s%c%15s\n", "start[sec]", csv_char,
	       ga->cols[COL_METRIC].title);

	return 0;
}

static void callback_log(struct gate_analyzer *ga, slot_hkey key,
			 struct timespec *ets, double metric, int pid)
{
	struct timespec ts;
	char buf[64];

	ts = *ets;
	if (elapsed_show && start_ts.tv_sec == 0 && start_ts.tv_nsec == 0)
		start_ts = first_event_ts;

	timespec_dec(ts,start_ts);
	if (ts.tv_sec == 0 && ts.tv_nsec < 0)
		sprintf(buf, "        -0.%09ld", -ts.tv_nsec);
	else
		sprintf(buf, "%10ld.%09ld", ts.tv_sec, ts.tv_nsec);

	if ((pid_show) && (comm_show)) {
		printf(ga->format_data, ga->cols[COL_KEY].width, key,
		       ga->cols[COL_ALIAS].width, ga->get_alias(key),
		       pid, pid_to_comm(pid), buf, metric);
	} else 	if (pid_show) {
		printf(ga->format_data, ga->cols[COL_KEY].width, key,
		       ga->cols[COL_ALIAS].width, ga->get_alias(key),
		       pid, buf, metric);
	} else if (comm_show) {
		printf(ga->format_data, ga->cols[COL_KEY].width, key,
		       ga->cols[COL_ALIAS].width, ga->get_alias(key),
		       pid_to_comm(pid), buf, metric);
	} else {
		printf(ga->format_data, ga->cols[COL_KEY].width, key,
		       ga->cols[COL_ALIAS].width, ga->get_alias(key),
		       buf, metric);
	}
}

static void post_analyze_log(struct gate_analyzer *ga)
{
	return ;
}

