/*
 lksteh_procname -- record process name
  
 Copyright (C) HITACHI,LTD. 2005
 WRITTEN BY HITACHI SYSTEMS DEVELOPMENT LABORATORY,
 Created by H.kawai <h-kawai@sdl.hitachi.co.jp>

 The development of this program is partly supported by IPA
 (Information-Technology Promotion Agency, Japan).
  
 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

#include <linux/version.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/lkst_private.h>
#include <linux/kernel.h>
#include <linux/smp.h>
#include "../include/extra_etypes.h"

/* Declaration of hook header */
LKST_ETYPE_DEF_MODULE(EID_PROCESS_NAME, INLINE, PROCESS_NAME,
		      "process name",
		      "ppid",
		      "tgid",
		      "comm1",
		      "comm2");

static lkst_buffer_id buffer_id[LKST_CPU_MAX] __cacheline_aligned = {
	[0 ... LKST_CPU_MAX-1] = LKST_BUFFER_ID_VOID
};
static int procname_buffer_id = LKST_BUFFER_ID_VOID;
static int buffer_nr_entries = 8192;

static ssize_t clear_procname_buffer(struct lkst_eh_device *,
				     const char *, size_t); // ctrl_store
static ssize_t show_info(struct lkst_eh_device *, char *); // ctrl_show
static void record_current_procs(void);

static int mod_init(void);
static void mod_cleanup(void);

static int create_procname_buffer(int cpu)
{
	cpumask_t allowed;
	int ret;

	if (!lkst_cpu_is_online(cpu)) {
		return 0;
	}
	
	lkst_cpu_migrate_save(&allowed, cpu);
	ret = lkst_buffer_create(procname_buffer_id,
				 buffer_nr_entries * LKST_EVENT_RECORD_BYTE,
				 LKST_BUFFER_ID_VOID);
	if (ret < 0) {
		printk("lksteh_procname: failed to create buffer\n");
		return ret;
	}
	buffer_id[cpu] = ret;
	lkst_cpu_migrate_restore(allowed);

	return ret;
}

static void delete_procname_buffer(int cpu)
{
	cpumask_t allowed;

	if (buffer_id[cpu] != LKST_BUFFER_ID_VOID) {
		lkst_cpu_migrate_save(&allowed, cpu);
		lkst_buffer_delete(buffer_id[cpu]);
		lkst_cpu_migrate_restore(allowed);
	}
}

static int __clear_procname_buffer(int cpu)
{
	int i;
	unsigned long flags;
	struct lkst_event_buffer *buffer;

	
	buffer = lkst_evhandlerprim_select_buffer_id(buffer_id[cpu]);
	if (!buffer)
		return -EINVAL;

	local_irq_save(flags);
 
	buffer->baseid = buffer->init_baseid = 0;
        buffer->write_offset = -LKST_BUFFER_INIT_POS;
 
	/* clear recid, cpu_id, and event_type */
        for (i = 0; i < buffer->size; i++) {
                LKST_BUFFER_BODY(buffer)[i].log_recid = 0;
                LKST_BUFFER_BODY(buffer)[i].log_event_type = LKST_ETYPE_VOID;
                LKST_BUFFER_BODY(buffer)[i].processor = cpu;
        }

	local_irq_restore(flags);

	return 0;
}

static ssize_t clear_procname_buffer(struct lkst_eh_device *ehdev,
				     const char *buf, size_t count)
{
	int i;
	int ret;
	cpumask_t allowed;

	for (i = 0; i < LKST_CPU_MAX; i++) {
		if (buffer_id[i] == LKST_BUFFER_ID_VOID)
			continue;
		lkst_cpu_migrate_save(&allowed, i);
		ret = __clear_procname_buffer(i);
		lkst_cpu_migrate_restore(allowed);
		if (ret < 0) {
			printk("lksteh_procname: "
			       "failed to clear buffer for cpu %d\n", i);
		}
	}

	record_current_procs();
	
	return count;
}

static ssize_t show_info(struct lkst_eh_device *ehdev, char *buf)
{
	int i;
	int n;
	char *p;

	p = buf;
	for (i = 0; i < LKST_CPU_MAX; i++) {
		if (buffer_id[i] == LKST_BUFFER_ID_VOID)
			continue;
		n = sprintf(p, "%scpu%d\n", (i)? "\n": "", i);
		if (n < 0)
			goto exit;
		p += n;
		n = sprintf(p, "buffer_id: %d\n", buffer_id[i]);
		if (n < 0)
			goto exit;
		p += n;
	}
 exit:
	return (ssize_t)(p - buf);
}

static void record_procname(struct task_struct *p)
{
	struct lkst_event_buffer *buffer;
	int pos;
	register struct lkst_event_record * event_entry_p;
	int baseid;
	unsigned long flags;
	
	buffer = lkst_evhandlerprim_select_buffer_id(
		buffer_id[smp_processor_id()]);
	if (!buffer)
		return;

	local_irq_save(flags);/* local irq disabling for atomicity */

        /* get a entry position */
	baseid = buffer->baseid++;

        /* adjust the entry position */
	pos = baseid - buffer->write_offset;
        if ( pos >= buffer->size )
                pos %= buffer->size;

        /* logging */
	event_entry_p = &(LKST_BUFFER_BODY(buffer)[pos]);
	event_entry_p->log_recid = baseid;
	event_entry_p->log_time = lkst_evhandlerprim_mc();
	event_entry_p->log_event_type = LKST_ETYPE_PROCESS_NAME;
	event_entry_p->log_pid = p->pid; /* don't use current->pid */ ;
	event_entry_p->log_arg1 = LKST_ARG(p->parent->pid);
	event_entry_p->log_arg2 = LKST_ARG(p->tgid);
	event_entry_p->log_arg3 = LKST_ARG64(*(u_int64_t*)p->comm);
	event_entry_p->log_arg4 = LKST_ARG64(*(u_int64_t*)(&p->comm[8]));

        /* Check buffer wrapping. */
        if ( pos == 0 )
                buffer->write_offset += buffer->size;
	
	local_irq_restore(flags);
}

static void lkst_evhandler_procname(void *phookrec, int event_type,
				  lkst_arg_t arg1, lkst_arg_t arg2,
				  lkst_arg_t arg3, lkst_arg_t arg4)
{
	preempt_disable();	/*IMPORTANT*/
	lkst_evhandlerprim_entry_log(event_type, arg1, arg2, arg3, arg4);

	switch (event_type) {
	case LKST_ETYPE_PROCESS_FORK:
		record_procname((void *)arg2);
		break;
	case LKST_ETYPE_PROCESS_EXEC:
		record_procname((void *)arg1);
		break;
	default:
		break;
	}
	preempt_enable_no_resched();	/*VERY IMPORTANT*/
}

static void record_current_procs(void)
{
	struct task_struct *p;

	record_procname(&init_task);
	read_lock(&tasklist_lock);
	for_each_process(p) {
		record_procname(p);
	}
	read_unlock(&tasklist_lock);
}

static LKST_EH_DEV_DEF(procname,
		       lkst_evhandler_procname,
		       NULL, clear_procname_buffer, show_info);

static int mod_init(void)
{
	int i;
	int ret;

	/* check module parameters */
	if (buffer_nr_entries <= 0) {
		printk("lksteh_procname: Error: "
		       "buffer_nr_entries is invalid\n");
		return -EINVAL;
	}

	/* Initialization of hook header */
	ret = lkst_hook_etype_register(&LKST_ETYPE_INFO(PROCESS_NAME));
	if (ret < 0) {
		printk("lksteh_procname: Error: "
		       "failed to register etype (%d)\n", ret);
		goto init_err;
	}
	
	/* register an event handler */
	ret = lkst_eh_device_register(&LKST_EH_DEV(procname));
	if (ret < 0) {
		printk("lksteh_procname: Error: "
		       "failed to register evhandler (%d)\n", ret);
		goto init_err;
	}

	/* create buffers for recording proc name */
	for (i = 0; i < LKST_CPU_MAX; i++) {
		ret = create_procname_buffer(i);
		if (ret < 0)
			goto init_err;
	}

	record_current_procs();

	return 0;
	
 init_err:
	mod_cleanup(); /*If occurs an error, invokes terminator.*/
	return ret;
}


static void mod_cleanup(void)
{
	int i;

	for (i = 0; i < LKST_CPU_MAX; i++)
		delete_procname_buffer(i);

	if (LKST_EH_DEV(procname).id != LKST_EVHANDLER_ID_VOID)
		lkst_eh_device_unregister(&LKST_EH_DEV(procname));

	lkst_hook_etype_unregister(&LKST_ETYPE_INFO(PROCESS_NAME));
}

module_init(mod_init);		/*define as initializer*/
module_exit(mod_cleanup);	/*define as terminator*/

MODULE_AUTHOR("H.Kawai <h-kawai@sdl.hitachi.co.jp>");
MODULE_DESCRIPTION("process name recording handler");
MODULE_LICENSE("GPL");

MODULE_PARM(procname_buffer_id, "i");
MODULE_PARM(buffer_nr_entries, "i");
MODULE_PARM_DESC(procname_buffer_id, "id of buffer where write log");
MODULE_PARM_DESC(buffer_nr_entries, "max entries per buffer");
