/*
 LKST logfile handling routines

 Copyright (C) HITACHI,LTD. 2004,2005
 WRITTEN BY HITACHI SYSTEMS DEVELOPMENT LABORATORY,
 Created by M.Hiramatsu <hiramatu@sdl.hitachi.co.jp>
 Updated by H.Kawai <h-kawai@sdl.hitachi.co.jp>
  
 The development of this program is partly supported by IPA
 (Information-Technology Promotion Agency, Japan).
  
 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <limits.h>
#include <logfile.h>
#include <timespec.h>

struct timespec first_event_ts = { LONG_MAX, LONG_MAX };

void logfile_tsc_to_timespec(lkstlogfile_t *lfile, 
			     struct timespec *ts, lkst_tsc_t tsc) 
{
	struct timespec *base = &lfile->header.sync_time.posix.log_time;
	lkst_tsc_t base_tsc = lfile->header.sync_time.log_arg1;
	u_int64_t cpu_hz = lfile->header.sync_time.log_arg2;
	lkst_tsc_t diff_tsc;
	if (!logfile_including_synctime(lfile)) return; //TODO return error-code

	if (base_tsc > tsc){
		diff_tsc = base_tsc - tsc;
		ts->tv_sec  = (unsigned long) (diff_tsc / cpu_hz);
		ts->tv_nsec = (unsigned long) ((diff_tsc % cpu_hz) *
					       1000000000 / cpu_hz);

		ts->tv_sec  = base->tv_sec - ts->tv_sec;
		ts->tv_nsec = base->tv_nsec - ts->tv_nsec;
		while (ts->tv_nsec < 0 && ts->tv_sec > 0) {
			ts->tv_sec --;
			ts->tv_nsec += NSEC_PER_SEC;
		}
	} else {
		diff_tsc = tsc - base_tsc;
		ts->tv_sec  = (unsigned long) (diff_tsc / cpu_hz);
		ts->tv_nsec = (unsigned long) ((diff_tsc % cpu_hz) *
					       1000000000 / cpu_hz);
		timespec_inc((*ts),(*base));
	}
}

int logfile_read_one(lkstlogfile_t * lfile) 
{
	if (lfile==NULL) return -1;
	if (feof(lfile->fp) || 
	    fread(&lfile->entry, sizeof(struct lkst_log_record),1,lfile->fp)<1) {
		lfile->entry.posix.log_time.tv_sec = LONG_MAX;//0x7fffffff;
		lfile->entry.posix.log_time.tv_nsec = LONG_MAX;//0x7fffffff;
		return 0;
	}
	return 1;
}

static int read_header_part(FILE *fp, struct lkst_file_header *lfh)
{
	int err=0;
	rewind(fp);
	err += fread(&lfh->head, sizeof(log_header_t), 1, fp);
	err += fread(&lfh->entry, sizeof(struct posix_log_entry), 1, fp);
	err += fread(&lfh->endian, sizeof(int), 1, fp);
	err += fread(&lfh->version, sizeof(int), 1, fp);
	err += fread(&lfh->arch, sizeof(char), LKST_ARCH_NAME_LEN, fp);
	if (err < LKST_ARCH_NAME_LEN+4 ) return -1;
	if (lfh->head.log_version != 0 || lfh->head.log_magic != LOGFILE_MAGIC){
		fprintf(stderr, "error: logfile version or magic is mismatched \n"
			"version %ld:%ld / magic 0x%x:0x%x \n",
			lfh->head.log_version, (long)0, 
			lfh->head.log_magic,LOGFILE_MAGIC);
		return -2;
	}
	return 0;
}

/* this is very informal mannor ... X-(*/
lkstlogfile_t *new_logfile(char *fname)
{
	lkstlogfile_t * lfile;
	int ret;
	lfile = calloc(sizeof(lkstlogfile_t),1);
	if (lfile == NULL) {
		fprintf(stderr, "error: failed to allocate memory\n");
		return NULL;
	}
	lfile->fp = fopen(fname, "r");
	if (lfile->fp == NULL) {
		fprintf(stderr, "error: failed to open (%s): %s\n", 
		       fname, strerror(errno));
		goto errout;
	}
	if ((ret = read_header_part(lfile->fp, &lfile->header)) < 0) {
		fprintf(stderr, "error: failed to read header (%s)(%d)\n",
			fname,ret);
		goto errout;
	}
	if (logfile_read_one(lfile) < 1) {
		fprintf(stderr, "warning: file is empty (%s)\n",fname);
	}
	if (lfile->entry.posix.log_event_type !=
	    LKST_ETYPE_LKST_SYNC_TIME) {
		fprintf(stderr, "error: the file format is old (%s)\n",fname);
		goto errout;
	}
	lfile->header.sync_time = lfile->entry;
	do {
		if (logfile_read_one(lfile) < 1) {
			fprintf(stderr, "warning: file is empty (%s)\n",fname);
		}
	} while (lfile->entry.posix.log_event_type ==
		 LKST_ETYPE_LKST_OVWRTN_REC);

	if (timespec_lt(lfile->entry.posix.log_time, first_event_ts))
		first_event_ts = lfile->entry.posix.log_time;

	return lfile;
errout:
	if (lfile->fp!=NULL) fclose(lfile->fp);
	free(lfile);
	return NULL;
}

void close_logfile(lkstlogfile_t *lfile) 
{
	if (lfile!=NULL) {
		if (lfile->fp!=NULL) fclose(lfile->fp);
		free(lfile);
	}
}

int add_loglist(struct lkstloglist * llist, char *fname) 
{
	lkstlogfile_t *lfile;
	if (llist == NULL) return -EINVAL;
	if (llist->nr_logfile == LOGFILE_MAX) return -ENOSPC;
	lfile = new_logfile(fname);
	if (lfile != NULL) {
		llist->logfiles[llist->nr_logfile++] = lfile;
		return 0;
	}
	return -EINVAL;
}

void cleanup_loglist(struct lkstloglist *llist)
{
	int i;
	if (llist) {
		for ( i=0;i<LOGFILE_MAX;i++) {
			close_logfile(llist->logfiles[i]);
			llist->logfiles[i] = NULL;
		}
	}
}

struct lkst_log_record * loglist_earliest_entry(struct lkstloglist *llist,
					        struct lkst_log_record *reload)
{
	int i;
	struct lkst_log_record *ret = NULL,*next;
	struct timespec ts;
	if (llist && llist->nr_logfile>0) {
		for (i=0;i<llist->nr_logfile;i++) {
			next = &llist->logfiles[i]->entry;
			if (next == reload) {
				ts = reload->posix.log_time;
				if (logfile_read_one(llist->logfiles[i])<1) {
					continue;
				}
				if (next->posix.log_event_type == 
				    LKST_ETYPE_LKST_OVWRTN_REC) {
					fprintf(stderr, "warning: some logs were overwritten, you may get the incorrect results.\n");
					next->posix.log_time = ts;
				}
			}
			if (ret == NULL || 
			    timespec_lt(next->posix.log_time,
					ret->posix.log_time)) {
			    ret = next;
			}
		}
		if (ret!=NULL && ret->posix.log_time.tv_nsec == LONG_MAX)
			ret = NULL;
	}
	return ret;
}
