/*
 process run-time analyzer

 Copyright (C) HITACHI,LTD. 2005
 WRITTEN BY HITACHI SYSTEMS DEVELOPMENT LABORATORY,
 Created by H.Kawai <h-kawai@sdl.hitachi.co.jp>
  
 The development of this program is partly supported by IPA
 (Information-Technology Promotion Agency, Japan).
  
 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <linux/lkst.h>
#include <linux/lkst_buffer.h>
#include <lkstla.h>
#include <string.h>
#include <extra_etypes.h>

static slot_hkey session_key_procrun(struct lkst_log_record *rec);
static slot_hkey info_key_procrun(struct lkst_log_record *rec_start, struct lkst_log_record *rec_end);
static char * get_alias_procrun(slot_hkey key);
static ga_type_t get_type_procrun(struct lkst_log_record *rec);
static int get_pid_procrun(struct lkst_log_record *rec,
			    struct lkst_log_record *dummy);
static int init_procrun(void);

static int pid_show=0;
struct gate_analyzer procrun_analyzer;

static int extra_opt_handler_pid(int c, char*v) {
	pid_show = 1;
	SET_COL(procrun_analyzer, COL_ALIAS, "pid", 5, "%*s");
	
	return 0;
}

static struct command_option cmd_options[1] = {
	{
		.opt = "P",
		.format = "-P",
		.description = "show PID instead of task_name",
		.handler = extra_opt_handler_pid,
	}
};

struct gate_analyzer procrun_analyzer = {
	.name = "procrun",
	.description = "process run-time analyzer",
	.cols = {
		[COL_KEY] = { "pid", 6, "%*d" },
		[COL_ALIAS] = { "task_name", 16, "%*s" },
		[COL_METRIC] = { "run-time" },
	},

	/* .analyze_logs, .get_cpu */
	GA_INIT_SESSION_ANALYZER

	/* .multi_session */
	GA_INIT_1TO1_SESSION

	/* .get_time, .get_metric, .flag, .ranks, .min, .max */
	GA_INIT_RET_SESSION_TIME

	.info_key = info_key_procrun,
	.session_key = session_key_procrun,
	.get_alias = get_alias_procrun,
	.get_pid = get_pid_procrun,
	.get_type = get_type_procrun,

	.init = init_procrun,
	.nr_options = 1,
	.options = { cmd_options },
};

static generic_slots_t *gsl_info = NULL;

static int init_procrun(void)
{
	gsl_info = new_generic_slots(5,20);
	if (!gsl_info)
		return -ENOMEM;

	return 0;
}

static slot_hkey session_key_procrun(struct lkst_log_record *rec)
{
	static int start_session = 0;
	slot_hkey key;

	if (start_session)
		key = ARG1UL(rec);
	else
		key = (slot_hkey)rec->posix.log_pid;
	
	start_session ^= 1;

	return key;
}

static void register_alias(struct lkst_log_record *rec) 
{
	slot_hkey key = info_key_procrun(rec,rec);
	slot_t *s;

	if (pid_show)
		return ;

	s = find_slot(gsl_info, key);
	if (!s)
		s = get_free_slot(gsl_info,key);

	if (s) {
		memcpy( (char*)slot_data(s), &rec->log_arg3, 8);
		memcpy( ((char*)slot_data(s))+8, &rec->log_arg4, 8);
		((char*)slot_data(s))[16]='\0';
	}
}

static char str_pid[16];
static char * get_alias_procrun(slot_hkey key)
{
	if (pid_show) {
		snprintf(str_pid, 10, "%ld", key);
		return str_pid;
	} else {
		slot_t *s = find_slot(gsl_info, key);
		if (s)
			return (char *)slot_data(s);
		else
			return "(unknown)";
	}
}

static int get_pid_procrun(struct lkst_log_record *rec,
			    struct lkst_log_record *dummy)
{
	int pid = -1;

	switch (rec->posix.log_event_type) { 
	case LKST_ETYPE_PROCESS_SCHED_ENTER:
		pid = (int)rec->posix.log_pid;
		break;
	case LKST_ETYPE_PROCESS_CONTEXTSW2:
		pid = (int)rec->log_arg1;
		break;
	}
	return pid;
}

static ga_type_t get_type_procrun(struct lkst_log_record *rec)
{
	switch (rec->posix.log_event_type) {

	case LKST_ETYPE_PROCESS_CONTEXTSW2:
		return GAT_EN_ST;
	case LKST_ETYPE_PROCESS_SCHED_ENTER:
		register_alias(rec);
		break;
	default:
		break;
	}

	return GAT_IGNORE;
}

/*return pid*/
static slot_hkey info_key_procrun(struct lkst_log_record *rec,
				   struct lkst_log_record *rec_end)
{
	return (slot_hkey)get_pid_procrun(rec, rec_end);
}
