/*
 hashed list

 Copyright (C) HITACHI,LTD. 2004-2005
 WRITTEN BY HITACHI SYSTEMS DEVELOPMENT LABORATORY,
 Created by M.Hiramatsu <hiramatu@sdl.hitachi.co.jp>
 Updated by H.Kawai <h-kawai@sdl.hitachi.co.jp>
  
 The development of this program is partly supported by IPA
 (Information-Technology Promotion Agency, Japan).
  
 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#define __KERNEL__
#include <asm/types.h>
#undef __KERNEL__
#include <linux/hash.h>
#include <errno.h>
#include <slot.h>

static inline slot_t * __new_slot(size_t data_size) 
{
	slot_t * ret;
	ret = malloc(sizeof(slot_t)+data_size);
	if (ret) {
		memset(ret, 0, sizeof(slot_t)+data_size);
		ret->key = HKEY_UNUSED;
	}
	return (slot_t*)ret;
}

/*from 0th */
static inline slot_t * __nth_entry(slot_t *l, int n)
{
	int i;
	for (i=0;l!=NULL && i<n;i++)l=l->next;
	return l;
}

static inline int __list_len(slot_t * l) 
{
	int i;
	for (i=0;l!=NULL;i++)l=l->next;
	return i;
}

static inline slot_t * __tail_entry(slot_t *l) 
{
	while (l->next != NULL)l = l->next;
	return l;
}

static inline void __insert_slots(slot_t **p, slot_t *s) 
{
	slot_t *e;
	if (s == NULL)return ;
	while (*p != NULL && (*p)->key < s->key) {
		p = &((*p)->next);
	}
	e = __tail_entry(s);
	e->next = *p;
	*p = s;
}

static inline void __release_slots(slot_t **p, slot_t *s) 
{
	slot_t *e;
	if (s == NULL)return ;
	while (*p != NULL) {
		p = &((*p)->next);
	}
	for (e = s; e->next != NULL; e = e->next){
		e->key = HKEY_UNUSED;
	}
	e->next = *p;
	*p = s;
}

static int __refill_pool(generic_slots_t * gslot) 
{
	int i;
	slot_t *s;
	if (gslot->pool) return 0;
	s = gslot->pool =  __new_slot(gslot->data_size);
	for (i=1;s !=NULL && i < HASH_TABLE_SIZE(gslot); i++) {
		s->next = __new_slot(gslot->data_size);
		s = s->next;
	}
	gslot->nr_slots += i;
	return i;
}

static int __rebalance_table(generic_slots_t *gslot)
{
	int i, nr, nr_avg, nr_mod;
	slot_t *p;
	if (gslot->flag == GSF_HASHED) {
		return -EINVAL;
	}
	
	nr_avg = gslot->nr_used_slots / HASH_TABLE_SIZE(gslot);
	nr_mod = gslot->nr_used_slots % HASH_TABLE_SIZE(gslot);

	/*join to one list*/
	p = gslot->h_table[0];
	for (i=1;i<HASH_TABLE_SIZE(gslot);i++) {
		nr = nr_avg;
		if (nr_mod) {nr ++; nr_mod --;}
		if (nr != 0) {
			p = __nth_entry(p, nr);
			gslot->h_table[i] = p;
		}else
			gslot->h_table[i] = NULL;
	}
	return 0;
}

static long __get_index(generic_slots_t * gslot, slot_hkey key)
{
	if (gslot->flag == GSF_HASHED) {
		return hash_long(key,gslot->hash_bits);
	} else {
		int i;
		slot_t *s,*e;
		for (i = 0;i < HASH_TABLE_SIZE(gslot)-1;i++) {
			s = gslot->h_table[i];
			e = gslot->h_table[i+1];
			if ((s == NULL) || (e == NULL) || 
			    (e != NULL && e->key > key))
				break;
		}
		return i;
	}
}

/*
 * function: new_generic_slots
 * desc: create a new generic_slots.
 * args:  hash_bits  -- bit width of the size of hash table 
 *        data_size  -- the size of data which each slot has.
 * retval:  !NULL    -- success
 *           NULL    -- hash_bits is out of range, or
 *                      data_size is invalid, or
 *                      memory allocation failure.
 */

generic_slots_t * new_generic_slots(int hash_bits, size_t data_size)
{
	struct generic_slots * ret;

	if (hash_bits < HASH_BITS_MIN || hash_bits > HASH_BITS_MAX || 
	    data_size < 0 ) {
		return NULL;
	}

	ret = malloc(sizeof(struct generic_slots));
	if (ret) {
		ret->flag = 0;
		ret->data_size = data_size;
		ret->hash_bits = hash_bits;
		ret->nr_slots = 0;
		ret->nr_used_slots = 0;
		ret->h_table = calloc(sizeof(slot_t *), HASH_TABLE_SIZE(ret));
		ret->pool = NULL;
		if (!ret->h_table) {
			free(ret);
			ret = NULL;
		}
	}
	return ret;
}

int release_all_slots(generic_slots_t *gslot)
{
	int i;
	if (gslot == NULL || gslot->h_table == NULL) return -EINVAL;
	
	if (gslot->flag == GSF_HASHED) {
		i = HASH_TABLE_SIZE(gslot)-1;
	}else {
		i = 0;
	}
	for (; i >= 0 ; i--) {
		__release_slots(&gslot->pool,gslot->h_table[i]);
	}
	for (i=0; i<HASH_TABLE_SIZE(gslot); i++) {
		gslot->h_table[i] = NULL;
	}
	gslot->nr_used_slots = 0;
	return 0;
}

/*
 * function: free_generic_slots
 * desc: release the specified generic_slots.
 * args:    gslot  -- a pointer of generic_slots to free.
 * retval:    0    -- success
 *         -EINVAL -- gslot is invalid
 */

int free_generic_slots(generic_slots_t * gslot)
{
	int i;
	slot_t *s, *n;
	if (gslot == NULL ) return -EINVAL;
	if (gslot->h_table ) {
		if (gslot->flag == GSF_HASHED) {
			i = HASH_TABLE_SIZE(gslot)-1;
		}else {
			i = 0;
		}
		for (; i >= 0 ; i--) {
			s = gslot->h_table[i];
			while (s!=NULL) {
				n = s->next;
				free(s);
				s = n;
			}
		}
		free(gslot->h_table);
	}
	free(gslot);
	return 0;
}


/*
 * function: get_free_slot
 * desc: get an unused (or new) slot.
 * args:    gslot  -- a pointer of generic_slots.
 *           key   -- the key of the slot
 * retval:  !NULL  -- success
 *           NULL  -- gslot is invalid
 */

slot_t * get_free_slot(generic_slots_t * gslot, slot_hkey key)
{
	slot_t * s, **l;
	if (gslot == NULL) return NULL;
	if (key == HKEY_UNUSED) key = HKEY_DUMMY;
	
	if(!gslot->pool) {
		if (__refill_pool(gslot) == 0)return NULL;
		__rebalance_table(gslot);
	}
	s = gslot->pool;
	gslot->pool = s->next;
	s->key = key; 
	s->next = NULL;
	l = &gslot->h_table[__get_index(gslot, key)];
	__insert_slots(l,s);
	gslot->nr_used_slots++; 
	return s;
}

/*
 * function: find_slot
 * desc: find the slot that has the specified key.
 * args:    gslot  -- a pointer of generic_slots.
 *           key   -- the key of the slot
 * retval:  !NULL  -- success
 *           NULL  -- gslot is invalid, or
 *                    could not find the slot.
 */

slot_t * find_slot(generic_slots_t * gslot, slot_hkey key)
{
	slot_t * slot;
	if (gslot == NULL) return NULL;
	if (key == HKEY_UNUSED) key = HKEY_DUMMY;

	slot = gslot->h_table[__get_index(gslot,key)];
	while (slot != NULL) {
		if (slot->key == key) return slot;
		slot = slot->next;
	}
	return NULL;
}

/*
 * function: release_slot
 * desc: release the slot that has the specified key.
 * args:    gslot  -- a pointer of generic_slots.
 *           key   -- the key of the slot
 * retval:   0  -- success
 *           NULL  -- gslot is invalid, or
 *                    could not find the slot.
 */

int release_slot(generic_slots_t * gslot, slot_hkey key)
{
	slot_t **s, *p;
	if (gslot == NULL) return -EINVAL;

	s = &gslot->h_table[__get_index(gslot,key)];
	while (*s!=NULL) {
		if ((*s)->key == key) {
			p = *s;
			*s = p->next;
			p->next = gslot->pool;
			p->key = HKEY_UNUSED;
			gslot->pool = p;
			gslot->nr_used_slots--;
			return 0;
		}
		s = &((*s)->next);
	}
	return -EINVAL;
}

slot_t * find_nearest_slot(generic_slots_t *gslot, slot_hkey key, int backward)
{
	slot_t *s, *p = NULL;

	if (gslot->flag != GSF_SORTED)
		return NULL;
	p = s = gslot->h_table[__get_index(gslot,key)];
	while (s!=NULL && key > s->key) {
		p = s;
		s = s->next;
	}
	if (!s)
		return p;
	if (key == s->key)
		return s;

	return (backward)?p:s;
}
