/*
 socketcall time distribution analyzer

 Copyright (C) HITACHI,LTD. 2005
 WRITTEN BY HITACHI SYSTEMS DEVELOPMENT LABORATORY,
 Created by H.Kawai <h-kawai@sdl.hitachi.co.jp>
  
 The development of this program is partly supported by IPA
 (Information-Technology Promotion Agency, Japan).
  
 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <linux/lkst.h>
#include <linux/lkst_buffer.h>
#include <linux/net.h>
#include <lkstla.h>
#define _WITHOUT_SYSCALL_NAME_DEFINITION_
#include <syscalls.h>
#include <extra_etypes.h>

#define LKST_ETYPE_SYSCALL_SYSENTER 0x32
#define LKST_ETYPE_SYSCALL_SYSEXIT 0x33
static int socketcall_no;

static int init_socketcall(void);
static slot_hkey info_key_socketcall(struct lkst_log_record *rec_start, struct lkst_log_record *rec_end);
static slot_hkey session_key_socketcall(struct lkst_log_record *rec);
static char * get_alias_socketcall(slot_hkey key);
static ga_type_t get_type_socketcall(struct lkst_log_record *rec);

extern struct gate_analyzer socketcall_analyzer;
static int show_systime=0;
static int extra_opt_handler_systime(int c, char*v) {
	show_systime = 1;
	SET_COL(socketcall_analyzer, COL_METRIC, "system-time", 0, "");
	return 0;
}

static char *socketcall_name[] = {
	[SYS_SOCKET] = "socket",
	[SYS_BIND] = "bind",
	[SYS_CONNECT] = "connect",
	[SYS_LISTEN] = "listen",
	[SYS_ACCEPT] = "accept",
	[SYS_GETSOCKNAME] = "getsockname",
	[SYS_GETPEERNAME] = "getpeername",
	[SYS_SOCKETPAIR] = "socketpair",
	[SYS_SEND] = "send",
	[SYS_RECV] = "recv",
	[SYS_SENDTO] = "sendto",
	[SYS_RECVFROM] = "recvfrom",
	[SYS_SHUTDOWN] = "shutdown",
	[SYS_SETSOCKOPT] = "setsockopt",
	[SYS_GETSOCKOPT] = "getsockopt",
	[SYS_SENDMSG] = "sendmsg",
	[SYS_RECVMSG] = "recvmsg",
};

#define NR_socketcalls (sizeof(socketcall_name) / sizeof(char *))

static struct command_option systime_option = {
	.opt = "T",
	.format = "-T",
	.description = "show system time instead of real time",
	.handler = extra_opt_handler_systime,
};

struct gate_analyzer socketcall_analyzer = {
	.name = "socketcall",
	.description = "socketcall analyzer",
	.cols = {
		[COL_KEY] = { "callno", 6, "%*d" },
		[COL_ALIAS] = { "socketcall_name", 18, "%*s" },
		[COL_METRIC] = { "processing-time" },
	},

	GA_INIT_SESSION_ANALYZER
	GA_INIT_1TON_SESSION
	GA_INIT_RET_SESSION_TIME
	GA_INIT_RET_EVENT_PID

	.info_key = info_key_socketcall,
	.session_key = session_key_socketcall,
	.get_alias = get_alias_socketcall,
	.get_type = get_type_socketcall,

	.init = init_socketcall,
	.nr_options = 1,
	.options = {&systime_option},
};

static generic_slots_t *gsl_info = NULL;
static int init_socketcall(void)
{
	int i;
	
	socketcall_no = -1;

	gsl_info = new_generic_slots(5,8);
	if (!gsl_info)
		return -ENOMEM;

	for (i = 0; i < NR_syscalls; i++) {
		if (!syscall_name[i])
			continue;
		if (strcmp(syscall_name[i], "socketcall") == 0) {
			socketcall_no = i;
			break;
		}
	}

	return (socketcall_no < 0)? -ENOSYS: 0;
}

static slot_hkey info_key_socketcall(struct lkst_log_record *rec_start,
				     struct lkst_log_record *rec)
{
	slot_hkey key = session_key_socketcall(rec_start);
	slot_t *s;

	s = find_slot(gsl_info, key);
	if (!s)
		return (slot_hkey)-1;

	return (slot_hkey)*((unsigned long *)slot_data(s));
}

static slot_hkey session_key_socketcall(struct lkst_log_record *rec)
{
	if (rec->posix.log_event_type == LKST_ETYPE_PROCESS_CONTEXTSW2)
		return (slot_hkey)(rec->log_arg1);
	else
		return (slot_hkey)(rec->posix.log_pid);
}

static char *get_alias_socketcall(slot_hkey key)
{
	if (key >= NR_socketcalls || socketcall_name[key] == NULL)
		return "(unknown)";
	else 
		return socketcall_name[key];
}

static void save_socketcall_type(struct lkst_log_record *rec)
{
	slot_hkey key = session_key_socketcall(rec);
	slot_t *s;

	s = find_slot(gsl_info, key);
	if (!s)
		s = get_free_slot(gsl_info,key);

	if (s) {
		*((unsigned long *)slot_data(s)) = ARG1UL(rec);
	}
}

static ga_type_t get_type_socketcall(struct lkst_log_record *rec)
{
	switch(rec->posix.log_event_type) {
	case LKST_ETYPE_SYSCALL_ENTRY:
	case LKST_ETYPE_SYSCALL_SYSENTER:
		if (ARG1UL(rec) != socketcall_no)
			return GAT_IGNORE;
		return GAT_START;
	case LKST_ETYPE_SYSCALL_EXIT:
	case LKST_ETYPE_SYSCALL_SYSEXIT:
		if (ARG1UL(rec) != socketcall_no)
			return GAT_IGNORE;
		return GAT_END;
	case LKST_ETYPE_NET_SOCKETIF:
		save_socketcall_type(rec);
		return GAT_IGNORE;
	case LKST_ETYPE_PROCESS_CONTEXTSWITCH:
		if(show_systime)
			return GAT_PAUSE;
	case LKST_ETYPE_PROCESS_CONTEXTSW2:
		if(show_systime)
			return GAT_RESUME;
	default:
		return GAT_IGNORE;
	}
}
