/*
 wait cpu analyzer

 Copyright (C) HITACHI,LTD. 2005
 WRITTEN BY HITACHI SYSTEMS DEVELOPMENT LABORATORY,
 Created by H.Kawai <h-kawai@sdl.hitachi.co.jp>
  
 The development of this program is partly supported by IPA
 (Information-Technology Promotion Agency, Japan).
  
 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <linux/lkst.h>
#include <linux/lkst_buffer.h>
#include <lkstla.h>
#include <string.h>
#include <extra_etypes.h>


static slot_hkey session_key_waitcpu(struct lkst_log_record *rec);
static slot_hkey info_key_waitcpu(struct lkst_log_record *rec_start, struct lkst_log_record *rec_end);
static char * get_alias_waitcpu(slot_hkey key);
static ga_type_t get_type_waitcpu(struct lkst_log_record *rec);
static int get_pid_waitcpu(struct lkst_log_record *rec,
			    struct lkst_log_record *dummy);
static int init_waitcpu(void);

static int pid_show=0;
struct gate_analyzer waitcpu_analyzer;

static int extra_opt_handler_pid(int c, char*v) {
	pid_show = 1;
	SET_COL(waitcpu_analyzer, COL_ALIAS, "pid", 5, "%*s");
	
	return 0;
}

static struct command_option cmd_options[1] = {
	{
		.opt = "P",
		.format = "-P",
		.description = "show PID instead of task_name",
		.handler = extra_opt_handler_pid,
	}
};

struct gate_analyzer waitcpu_analyzer = {
	.name = "waitcpu",
	.description = "Wait time for cpu analyzer",
	.cols = {
		[COL_KEY] = { "pid", 6, "%*d" },
		[COL_ALIAS] = { "task_name", 16, "%*s" },
		[COL_METRIC] = { "wait-time" },
	},

	/* .analyze_logs, .get_cpu */
	GA_INIT_SESSION_ANALYZER

	/* .multi_session */
	GA_INIT_FIRSTTO1_SESSION

	/* .get_time, .get_metric, .flag, .ranks, .min, .max */
	GA_INIT_RET_SESSION_TIME

	.info_key = info_key_waitcpu,
	.session_key = session_key_waitcpu,
	.get_alias = get_alias_waitcpu,
	.get_pid = get_pid_waitcpu,
	.get_type = get_type_waitcpu,

	.init = init_waitcpu,
	.nr_options = 1,
	.options = { cmd_options },
};

static generic_slots_t *gsl_info = NULL;

static int init_waitcpu(void)
{
	gsl_info = new_generic_slots(5,20);
	if (!gsl_info)
		return -ENOMEM;

	return 0;
}

static slot_hkey session_key_waitcpu(struct lkst_log_record *rec)
{
	switch (rec->posix.log_event_type) {
	case LKST_ETYPE_PROCESS_WAKEUP2:
	case LKST_ETYPE_PROCESS_CONTEXTSW2:
		return ARG1UL(rec);
	default:
		// LKST_ETYPE_PROCESS_CONTEXTSWITCH:
		return (slot_hkey)rec->posix.log_pid;
	}
}

static void register_alias(struct lkst_log_record *rec) 
{
	slot_hkey key = info_key_waitcpu(rec,rec);
	slot_t *s;

	if (pid_show)
		return ;

	s = find_slot(gsl_info, key);
	if (!s)
		s = get_free_slot(gsl_info,key);

	if (s) {
		memcpy( (char*)slot_data(s), &rec->log_arg3, 8);
		memcpy( ((char*)slot_data(s))+8, &rec->log_arg4, 8);
		((char*)slot_data(s))[16]='\0';
	}
}

static char str_pid[16];
static char * get_alias_waitcpu(slot_hkey key)
{
	if (pid_show) {
		snprintf(str_pid, 10, "%ld", key);
		return str_pid;
	} else {
		slot_t *s = find_slot(gsl_info, key);
		if (s)
			return (char *)slot_data(s);
		else
			return "(unknown)";
	}
}

static int get_pid_waitcpu(struct lkst_log_record *rec,
			    struct lkst_log_record *dummy)
{
	int pid = -1;

	switch (rec->posix.log_event_type) { 
	case LKST_ETYPE_PROCESS_CONTEXTSWITCH:
	case LKST_ETYPE_PROCESS_SCHED_ENTER:
		pid = (int)rec->posix.log_pid;
		break;
	case LKST_ETYPE_PROCESS_WAKEUP2:
	case LKST_ETYPE_PROCESS_CONTEXTSW2:
		pid = (int)rec->log_arg1;
		break;
	}
	return pid;
}

static ga_type_t get_type_waitcpu(struct lkst_log_record *rec)
{
	switch (rec->posix.log_event_type) {
	case LKST_ETYPE_PROCESS_WAKEUP2:
		return GAT_START;
	case LKST_ETYPE_PROCESS_CONTEXTSWITCH:
		/* A process can wake up itself, when such event occurs,
		   we should ignore it */
		if (ARG3UL(rec) == 0)
			return GAT_REMOVE_START;
		else 
			return GAT_REMOVE;
	case LKST_ETYPE_PROCESS_CONTEXTSW2:
		return GAT_END;
	case LKST_ETYPE_PROCESS_SCHED_ENTER:
		register_alias(rec);
		break;
	default:
		break;
	}

	return GAT_IGNORE;
}

/*return pid*/
static slot_hkey info_key_waitcpu(struct lkst_log_record *rec,
				   struct lkst_log_record *rec_end)
{
	return (slot_hkey)get_pid_waitcpu(rec, rec_end);
}
