#
#       file_type.py
#
#       2003.11.25
#
#   Copyright (C) Hidetoshi Nakano
#
#   Please use this program at your own risk.
#   Without any warranty.
# 
#
#############################################
import os
import gzip
import zipfile

import Lpy.def_parm
import Lpy.lpy_programs
import Lpy.utils.file_utils
import Lpy.utils.get_env

if Lpy.utils.get_env.get_pythonVersion() >= 23:
    bzip2_support = 1
    try:
        import bz2
    except:
        bzip2_support = 0
else:
    bzip2_support = 0

F_ps  = "POSTSCRIPT"
F_pcl = "PCL"
F_HPGL  = "HPGL"
F_Lips  = "LIPS"
F_Escpage = "ESC/Page"
F_Escp    = "ESCP"
F_eps   = 'EPS'

F_pjl  = 'PJL'
F_pdf  = 'PDF'
F_text = 'TEXT'
F_texinfo = 'Texinfo'
F_latex = 'TEX'
F_dvi   = 'DVI'
F_troff = 'TROFF'
F_image = 'IMAGE'
F_tiff  = 'TIFF'
F_binary = "BINARY"
#####
F_gzip = 'GZIP'
F_comp = 'COMPRESSION'
F_bzip = 'BZIP2'
F_zip  = 'ZIP'
F_tar  = 'TAR'

## PDL:Page Description Language file
PDL_type = (
    (F_pjl,  '\033%-12345X@PJL'),
    (F_pcl,  '\033E\033'),
    (F_Lips, '\033%@'),
    (F_Escpage,'\033@\033'),
)

## fileType
fileTypes = (
    (F_ps,   ('postscript',)),
    (F_pdf,  ('pdf',)),
    (F_troff,('troff',)),
    (F_texinfo,('texinfo',)),
    (F_dvi,  ('tex dvi',)),
    (F_latex,('latex',)),
    (F_image,('jpeg','png','mng','tiff',
    'pc bitmap data','xwd','netpbm','fits',
    'sun raster','targa','sgi','kodak photo cd')),
)

## compression file
compressType = (
    (F_gzip,'gzip compressed'),  # .gz
    (F_bzip,'bzip2 compressed'), # .bz2
    (F_comp,"compress'd"),    # .Z
    (F_zip, 'zip'),           # .zip
    (F_tar, "tar archive"),   # .tar
)

NO_convertType = []
compressTypeKeys = []

for key,dat in PDL_type:
    NO_convertType.append(key)
for key,dat in compressType:
    compressTypeKeys.append(key)

ESC = "\033" # 0x1b
FileCheck = Lpy.lpy_programs.get_prg('file')

def check_PDL_type(filename):
        dat = Lpy.utils.file_utils.get_filedata(filename,14)
        if dat == None or len(dat) < 3:
            return None
        if dat[0] == ESC: # ESC
            for key,keydata in PDL_type:
                if dat.startswith(keydata):
                    return key
        return None

def check_compressType(data):
        for key,dat in compressType:
            if key == F_tar:
                if data.find(dat) >= 0:
                    return F_tar
            else:
                if data.startswith(dat):
                    return key
        return None

def check_ftype(data):
        for key,dat in fileTypes:
            for name in dat:
                if data.startswith(name):
                    if key == F_ps:
                        if data.find('eps') >= 0:
                            return F_eps
                    return key
        ## image
        if data.find('image') >= 0:
            return F_image

        ## Text
        for name in ('text','ascii','english','script'):
            if data.find(name) >= 0:
                return F_text
        ## Binary
        for name in ('elf','video','audio'):
           if data.find(name) >= 0:
                return F_binary
        return None

def get_ftype(filename):
        if not filename:
            return None

        data = Lpy.utils.file_utils.get_command_data("%s -L -b %s" % (FileCheck,filename))
        if data is None:
            return Lpy.def_parm.Error_no_support
        dat = data[0].lower()
        ret = check_compressType(dat)
        if ret:
            return ret
        ret = check_ftype(dat)
        if ret:
            return ret
        return Lpy.def_parm.Error_no_support

"""
# file 4.0.6 and above
import magic

ms = magic.open(magic.MAGIC_NONE)
ms.load()
#type =  ms.file(filename)
#print type
--------------------------
f = file(filename, "r")
buffer = f.read(4096)
f.close()

type = ms.buffer(buffer)
print type
ms.close()
"""

def gzip_decompress(filename):
    try:
        fp = gzip.GzipFile(filename,mode = 'rb')
    except:
        print "GzipFile open Error.(%s)" % filename
        return None
    try:
        data = fp.read()
    except:
        print "GzipFile read Error.(%s)" % filename
        data = None
    fp.close()

    if data:
        tmpfile = Lpy.utils.file_utils.make_tmpfile(filename)
        if Lpy.utils.file_utils.write_tmpfile(tmpfile,data,"w"):
            return tmpfile
    return None

def zip_decompress(filename):
    try:
        fp = zipfile.ZipFile(filename,mode = 'r')
    except:
        print "ZipFile open Error.(%s)" % filename
        return None
    try:
        data = fp.read()
    except:
        print "ZipFile read Error.(%s)" % filename
        data = None
    fp.close()

    if data:
        tmpfile = Lpy.utils.file_utils.make_tmpfile(filename)
        if Lpy.utils.file_utils.write_tmpfile(tmpfile,data,"w"):
            return tmpfile
    return None

def bzip2_decompress(filename):
    if bzip2_support:
        try:
            fp = bz2.BZ2File(filename,mode = 'r')
        except:
            print "ZipFile open Error.(%s)" % filename
            return None
        try:
            data = fp.read()
        except:
            print "ZipFile read Error.(%s)" % filename
            data = None
        fp.close()

        if data:
            tmpfile = Lpy.utils.file_utils.make_tmpfile(filename)
            if Lpy.utils.file_utils.write_tmpfile(tmpfile,data,"w"):
                return tmpfile
        return None
    else:
        cmds = Lpy.lpy_programs.get_prg('bunzip2')
        if cmds is None:
            return Lpy.def_parm.Error_no_support

        runit = '%s -c %s > %s' % (cmds,filename,tmpfile) # -q
        msg = Lpy.utils.file_utils.sys_command(runit)
        if msg:
            return None
        return tmpfile

def decompress(filename): # .Z
        tmpfile = Lpy.utils.file_utils.make_tmpfile(filename)
        cmds = Lpy.lpy_programs.get_prg('gunzip')
        if cmds is None:
            return Lpy.def_parm.Error_no_support
        runit = '%s -c %s > %s' % (cmds,filename,tmpfile) # -q
        msg = Lpy.utils.file_utils.sys_command(runit)
        if msg:
            return None
        return tmpfile

def tar_decompress(filename):
    return Lpy.def_parm.Error_no_support

def check_compress(ftype,filename):
        if ftype == F_gzip:    # .gz
            tmpfile = gzip_decompress(filename)
        elif ftype == F_bzip: # .bz2
            tmpfile = bzip2_decompress(filename)
        elif ftype == F_comp: # .Z
            tmpfile = decompress(filename)
        elif ftype == F_zip:   # .zip
            tmpfile = zip_decompress(filename)
        elif ftype == F_tar:  # .tar
            tmpfile = tar_decompress(filename)

        if tmpfile is None:
            return '-1'
        elif tmpfile == Lpy.def_parm.Error_no_support:
            return Lpy.def_parm.Error_no_support

        if Lpy.utils.file_utils.check_file(tmpfile) in Lpy.utils.file_utils.Error_lists:
            Lpy.utils.file_utils.remove_file(tmpfile)
            return '-1'
        return tmpfile

####
class FileAttrib:
    def __init__(self,filename,ftype = None,end_delete = None):
        self.filename = filename
        self.err_flag = None
        self.psfile   = None
        self.title    = None
        self.ftype    = None
        self.no_convert = None
        self.delete   = None
        self.orgfile  = None
        self.lang     = None

        if ftype and (ftype == "binary" or ftype == "BINARY"):
            self.ftype = F_binary
        if end_delete:
            self.delete = Lpy.utils.file_utils.OK

    def set_filetype(self, ftype, mode=None):
            self.ftype = ftype
            self.lang = self.ftype
            self.no_convert = mode

    def set_ftype(self):
        filename = Lpy.utils.file_utils.check_file(self.filename)
        if filename in Lpy.utils.file_utils.Error_lists:
            self.err_flag = _("I can't read a file (%s).") % self.filename
            return self.err_flag
        self.filename = filename
        self.title = os.path.basename(self.filename)

        ## PDL file
        ftype = check_PDL_type(self.filename)
        if ftype:
            self.set_filetype(ftype,1)
            return None
        ## Not installed 'file' program
        if FileCheck is None:
            self.set_filetype(F_text)
            return None

        ## check filetype
        ftype = get_ftype(self.filename)
        if ftype is None or ftype == Lpy.def_parm.Error_no_support:
            if ftype == Lpy.def_parm.Error_no_support and self.ftype:
                self.set_filetype(self.ftype)
                return None
            else:
                self.err_flag = _("I can't support a filetype (%s).") % self.filename
                return self.err_flag

        if ftype in compressTypeKeys:
            return self.set_compression(ftype,self.filename)

        self.set_filetype(ftype)
        return None

    # decompression
    def set_compression(self,ftype,filename):
        tmpfile = check_compress(ftype,filename)
        if tmpfile == Lpy.def_parm.Error_no_support:
            self.err_flag = _("I can't support a filetype (%s).") % filename
            return self.err_flag

        elif tmpfile == '-1':
            self.err_flag = _("I can't decompress a file (%s).") % filename
            return self.err_flag

        ftype = check_PDL_type(tmpfile)
        if ftype in NO_convertType:
                self.set_tmpfile(tmpfile)
                self.set_filetype(ftype,1)
                return None

        ftype = get_ftype(tmpfile)
        if (ftype == Lpy.def_parm.Error_no_support or
            ftype in compressTypeKeys):
            # no support ::double compression:: (*.tar.gz .. )
            Lpy.utils.file_utils.remove_file(tmpfile)
            self.err_flag = _("I can't support a filetype (%s).") % filename
            return self.err_flag

        self.set_tmpfile(tmpfile)
        self.set_filetype(ftype)
        return None

    def get_output_file(self):
        if self.psfile:
            return self.psfile
        if self.filename:
            return self.filename
        else:
            return self.orgfile

    def get_filename(self):
        if self.delete == Lpy.utils.file_utils.OK and self.orgfile:
            return self.orgfile
        return self.filename

    def set_tmpfile(self,outfile):
        if self.orgfile is None:
            self.orgfile  = self.filename
        else:
            self.multi_remove(self.orgfile,self.filename)

        self.filename = outfile
        self.delete = Lpy.utils.file_utils.OK

    def clean_tmpfile(self):
        if self.psfile:
            Lpy.utils.file_utils.remove_file(self.psfile)

        if self.delete == Lpy.utils.file_utils.OK and self.orgfile:
            self.multi_remove(self.orgfile,self.filename)
            self.filename = self.orgfile
            self.orgfile  = None
        self.delete = None

    def multi_remove(self,orgfile,filename):
        orgins = orgfile.strip().split()
        names  = filename.strip().split()
        for i in range(len(names)):
            #print "clean",names[i],orgins[i]
            if names[i] != orgins[i]:
                Lpy.utils.file_utils.remove_file(names[i])
