/*
  MeCab -- Yet Another Part-of-Speech and Morphological Analyzer

  $Id: libmecab.cpp,v 1.19 2006/07/09 15:18:41 taku-ku Exp $;

  Copyright (C) 2001-2006 Taku Kudo <taku@chasen.org>
  Copyright (C) 2004-2006 Nippon Telegraph and Telephone Corporation

*/
#include "mecab.h"
#include "mutex.h"
#include "utils.h"
#include "tokenizer.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

static const int LIBMECAB_ID = 77718;

std::string &getStaticErrorString()
{
  static std::string errorStr;
  return errorStr;
}

const char *getGlobalError()
{
  return getStaticErrorString().c_str();
}

void setGlobalError(const char *str)
{
  MeCab::Mutex m;
  m.lock();
  std::string &error = getStaticErrorString();
  error = str;
}

struct mecab_t {
  int allocated;
  MeCab::Tagger* ptr;
};

#if defined  (_WIN32) && ! defined (__CYGWIN__)
#include <windows.h>
HINSTANCE DllInstance = 0;

BOOL __stdcall DllMain( HINSTANCE hinst, DWORD dwReason, void* )
{
  if (! DllInstance) DllInstance = hinst;
  return TRUE;
}
#endif

mecab_t* mecab_new(int argc, char **argv)
{
  mecab_t *c = new mecab_t;
  MeCab::Tagger *ptr = MeCab::createTagger(argc, argv);
  if (! c || ! ptr) {
    delete c;
    setGlobalError(MeCab::getTaggerError());
    return 0;
  }
  c->ptr = ptr;
  c->allocated = LIBMECAB_ID;
  return c;
}

mecab_t* mecab_new2(const char *arg)
{
  mecab_t *c = new mecab_t;
  MeCab::Tagger *ptr = MeCab::createTagger(arg);
  if (! c || ! ptr) {
    delete c;
    setGlobalError(MeCab::getTaggerError());
    return 0;
  }
  c->ptr = ptr;
  c->allocated = LIBMECAB_ID;
  return c;
}

const char *mecab_version()
{
  return MeCab::Tagger::version();
}

const char* mecab_strerror(mecab_t *c)
{
  if (! c || ! c->allocated)
    return const_cast<char *>(getGlobalError());
  return c->ptr->what();
}

void mecab_destroy(mecab_t *c)
{
  if (c && c->allocated) {
    delete c->ptr;
    delete c;
  }
  c = 0;
}

#define MECAB_CHECK_FIRST_ARG(c,t)  \
 if (! (c) || (c)->allocated != LIBMECAB_ID) { \
    setGlobalError("first argment seems invalid"); \
    return 0; \
 } MeCab::Tagger *(t) = (c)->ptr;

const char* mecab_sparse_tostr(mecab_t *c, const char *str)
{
  MECAB_CHECK_FIRST_ARG(c,t);
  return t->parse(str);
}

const char* mecab_sparse_tostr2(mecab_t *c, const char *str, size_t len)
{
  MECAB_CHECK_FIRST_ARG(c,t);
  return t->parse(str, len);
}

char* mecab_sparse_tostr3(mecab_t *c, const char *str, size_t len,
                          char *out, size_t len2)
{
  MECAB_CHECK_FIRST_ARG(c,t);
  return const_cast<char *>(t->parse(str, len, out, len2));
}

mecab_node_t* mecab_sparse_tonode(mecab_t *c, const char *str)
{
  MECAB_CHECK_FIRST_ARG(c,t);
  return reinterpret_cast<mecab_node_t *>(t->parseToNode(str));
}

mecab_node_t* mecab_sparse_tonode2(mecab_t *c, const char *str, size_t len)
{
  MECAB_CHECK_FIRST_ARG(c,t);
  return reinterpret_cast<mecab_node_t *>(t->parseToNode(str,len));
}

const char* mecab_nbest_sparse_tostr(mecab_t* c, size_t N, const char *str)
{
  MECAB_CHECK_FIRST_ARG(c,t);
  return t->parseNBest(N, str);
}

const char* mecab_nbest_sparse_tostr2(mecab_t* c, size_t N, const char* str, size_t len)
{
  MECAB_CHECK_FIRST_ARG(c,t);
  return t->parseNBest(N, str, len);
}

char* mecab_nbest_sparse_tostr3(mecab_t* c, size_t N, const char *str, size_t len,
                                char *out, size_t len2)
{
  MECAB_CHECK_FIRST_ARG(c,t);
  return const_cast<char *>(t->parseNBest(N, str, len, out, len2));
}

int mecab_nbest_init(mecab_t *c, const char *str)
{
  MECAB_CHECK_FIRST_ARG(c,t);
  return t->parseNBestInit(str);
}

int mecab_nbest_init2(mecab_t *c, const char *str, size_t len)
{
  MECAB_CHECK_FIRST_ARG(c,t);
  return t->parseNBestInit(str, len);
}

const char* mecab_nbest_next_tostr(mecab_t* c)
{
  MECAB_CHECK_FIRST_ARG(c,t);
  return t->next();
}

char* mecab_nbest_next_tostr2(mecab_t* c, char *out, size_t len2)
{
  MECAB_CHECK_FIRST_ARG(c,t);
  return const_cast<char *>(t->next(out, len2));
}

mecab_node_t* mecab_nbest_next_tonode(mecab_t* c)
{
  MECAB_CHECK_FIRST_ARG(c,t);
  return reinterpret_cast<mecab_node_t *>(t->nextNode());
}

const char* mecab_format_node(mecab_t *c, mecab_node_t* n)
{
  MECAB_CHECK_FIRST_ARG(c,t);
  return t->formatNode(n);
}
