/* 
 * Copyright (c) 2006-2007 NTT DATA CORPORATION.
 * All rights reserved.
 */

#include <dirent.h>
#include <sys/stat.h>
#include <unistd.h>
#include "postgres.h"
#include "pgsenna2.h"
#include "miscadmin.h"
#include "funcapi.h"

PG_FUNCTION_INFO_V1(pgs2destroy);
PG_FUNCTION_INFO_V1(pgs2indexinfo);
PG_FUNCTION_INFO_V1(pgs2getlexicon);
PG_FUNCTION_INFO_V1(pgs2version);


typedef struct {
  char *filename;
  int dead_flag;   // 0 == alive, 1 == dead
  int key_size;
  int flags;
  int initial_n_segments;
  sen_encoding encoding;
  unsigned nrecords_keys;
  unsigned file_size_keys;
  unsigned nrecords_lexicon;
  unsigned file_size_lexicon;
  unsigned inv_seg_size;
  unsigned inv_chunk_size;
  void *next_list;
} sen_index_list;

typedef struct {
  int id;
  char *lexicon_array;
  int flag;
} sym_array;

inline char *
text2cstr(text *t)
{
  int32 size = VARSIZE(t) - VARHDRSZ;
  char *c = palloc(size + 1);
  memcpy(c, t->vl_dat, size);
  c[size] = '\0';
  return c;
}

static void
do_dir(const char *dirname, int *n)
{
  DIR *dir = opendir(dirname);
  if (dir) {
    int fd = dirfd(dir);
    struct dirent *ent;
    fchdir(fd);
    while ((ent = readdir(dir))) {
      const char *name = ent->d_name;
      if (!strcmp(name, ".") || !strcmp(name, "..")) {
        continue;
      }
      switch (ent->d_type) {
      case DT_REG :
        {
          int len = strlen(name) - 4;
          if (len > 0 &&
              name[len] == '.' &&
              name[len + 1] == 'S' &&
              name[len + 2] == 'E' &&
              name[len + 3] == 'N') {
            struct stat s;
            char *idxname = strdup(name);
            idxname[len] = '\0';
            if (stat(idxname, &s) == -1 && errno == ENOENT) {
              sen_index_remove(idxname);
              (*n)++;
            }
          }
        }
        break;
      case DT_DIR :
      case DT_LNK :
        do_dir(ent->d_name, n);
        fchdir(fd);
        break;
      default :
        break;
      }
    }
    closedir(dir);
  }
}

Datum
pgs2destroy(PG_FUNCTION_ARGS)
{
  int n = 0;
  DIR *dir = opendir(".");
  if (dir) {
    int fd = dirfd(dir);
    do_dir(DataDir, &n);
    fchdir(fd);
    closedir(dir);
  }
  PG_RETURN_INT32(n);
}

static void
do_dir_indexinfo(const char *dirname, int *n, sen_index_list *si_list)
{
  DIR *dir = opendir(dirname);
  if (dir) {
    int fd = dirfd(dir);
    struct dirent *ent;
    fchdir(fd);
    while ((ent = readdir(dir))) {
      const char *name = ent->d_name;
      if (!strcmp(name, ".") || !strcmp(name, "..")) {
        continue;
      }
      switch (ent->d_type) {
      case DT_REG :
        {
          int len = strlen(name) - 4;
          if (len > 0 &&
              name[len] == '.' &&
              name[len + 1] == 'S' &&
              name[len + 2] == 'E' &&
              name[len + 3] == 'N') {
            char *idxname = strdup(name);
            struct stat s;
            sen_index_list *si_list_next;
            sen_index_list *si_list_tmp;
            sen_index *sindex;

            idxname[len] = '\0';
            sindex = sen_index_open(idxname);
            si_list_next = (sen_index_list *)palloc(sizeof(*si_list_next));
            sen_index_info(sindex,
                           &(si_list_next->key_size),
                           &(si_list_next->flags),
                           &(si_list_next->initial_n_segments),
                           &(si_list_next->encoding),
                           &(si_list_next->nrecords_keys),
                           &(si_list_next->file_size_keys),
                           &(si_list_next->nrecords_lexicon),
                           &(si_list_next->file_size_lexicon),
                           &(si_list_next->inv_seg_size),
                           &(si_list_next->inv_chunk_size));
            si_list_tmp = si_list;
            while(si_list_tmp->next_list != NULL) {
              si_list_tmp = si_list_tmp->next_list;
            }
            si_list_tmp->next_list = si_list_next;
            si_list_next->filename = idxname;
            if (stat(idxname, &s) == -1 && errno == ENOENT) {
              si_list_next->dead_flag = 1;
            } else {
              si_list_next->dead_flag = 0;
            }
            si_list_next->next_list = NULL;
            (*n)++;
          }
        }
        break;
      case DT_DIR :
      case DT_LNK :
        do_dir_indexinfo(ent->d_name, n, si_list);
        fchdir(fd);
        break;
      default :
        break;
      }
    }
    closedir(dir);
  }
}

Datum
pgs2indexinfo(PG_FUNCTION_ARGS)
{
  int n = 0;

  FuncCallContext     *funcctx;
  int                  call_cntr;
  int                  max_calls;
  TupleDesc            tupdesc;
  AttInMetadata       *attinmeta;
  sen_index_list *si_list;

  if (SRF_IS_FIRSTCALL()) {
    DIR *dir = opendir(".");
    MemoryContext   oldcontext;
    funcctx = SRF_FIRSTCALL_INIT();
    oldcontext = MemoryContextSwitchTo(funcctx->multi_call_memory_ctx);
    if (get_call_result_type(fcinfo, NULL, &tupdesc) != TYPEFUNC_COMPOSITE)
      ereport(ERROR,
              (errcode(ERRCODE_FEATURE_NOT_SUPPORTED),
               errmsg("function returning record called in context "
                      "that cannot accept type record")));
    attinmeta = TupleDescGetAttInMetadata(tupdesc);
    funcctx->attinmeta = attinmeta;

    si_list = (sen_index_list*) palloc(sizeof(*si_list));
    si_list->next_list = NULL;
    funcctx->user_fctx = (void *)si_list;
    if (dir) {
      int fd = dirfd(dir);
      do_dir_indexinfo(DataDir, &n, si_list);
      fchdir(fd);
      closedir(dir);
    }
    funcctx->max_calls = n;
    MemoryContextSwitchTo(oldcontext);
  }

  funcctx = SRF_PERCALL_SETUP();

  si_list = (sen_index_list *) funcctx->user_fctx;

  call_cntr = funcctx->call_cntr;
  max_calls = funcctx->max_calls;
  attinmeta = funcctx->attinmeta;
 
  if (call_cntr < max_calls) {
    int i;
    char       **values;
    HeapTuple    tuple;
    Datum        result;

    si_list = si_list->next_list;
    for (i = 0; i < call_cntr; i++) {
      si_list = si_list->next_list;
    }
    values = (char **) palloc(12 * sizeof(char *));
    for(i = 0; i < 12; i++) {
      values[i] = (char *) palloc(24 * sizeof(char));
    }
    snprintf(values[0], 24, "%s", si_list->filename);
    snprintf(values[1], 24, "%d", si_list->dead_flag);
    snprintf(values[2], 24, "%d", si_list->key_size);
    snprintf(values[3], 24, "%d", si_list->flags);
    snprintf(values[4], 24, "%d", si_list->initial_n_segments);
    snprintf(values[5], 24, "%d", si_list->encoding);
    snprintf(values[6], 24, "%u", si_list->nrecords_keys);
    snprintf(values[7], 24, "%u", si_list->file_size_keys);
    snprintf(values[8], 24, "%u", si_list->nrecords_lexicon);
    snprintf(values[9], 24, "%u", si_list->file_size_lexicon);
    snprintf(values[10], 24, "%u", si_list->inv_seg_size);
    snprintf(values[11], 24, "%u", si_list->inv_chunk_size);
    tuple = BuildTupleFromCStrings(attinmeta, values);
    result = HeapTupleGetDatum(tuple);
    for(i = 0; i < 12; i++) {
      pfree(values[i]);
    }
    pfree(values);
    SRF_RETURN_NEXT(funcctx, result);
  } else {
    SRF_RETURN_DONE(funcctx);
  }
}

void *sen_inv_cursor_open(sen_inv *inv, unsigned key, int with_pos);

static sym_array **
do_dir_getlexicon(const char *dirname, int *n, sym_array **s_array, char *filename)
{
  DIR *dir = opendir(dirname);
  if (dir) {
    int fd = dirfd(dir);
    struct dirent *ent;
    fchdir(fd);
    while ((ent = readdir(dir))) {
      const char *name = ent->d_name;
      if (!strcmp(name, ".") || !strcmp(name, "..")) {
        continue;
      }
      switch (ent->d_type) {
      case DT_REG :
        {
          int len = strlen(name) - 4;
          if (len > 0 &&
              name[len] == '.' &&
              name[len + 1] == 'S' &&
              name[len + 2] == 'E' &&
              name[len + 3] == 'N') {
            char *idxname = strdup(name);
            idxname[len] = '\0';
            if (strcmp(idxname, filename) == 0) {
              sen_index *sindex;
              char *keybuf;
              int sid = 1;
              keybuf = (char *) palloc(sizeof(char *) * SEN_SYM_MAX_KEY_SIZE);
              sindex = sen_index_open(idxname);
              (*n) = 0;
              s_array = (sym_array **) palloc(sen_sym_size(sindex->lexicon) *
                                              sizeof(**s_array));
              while (true) {
                keybuf[0] = '\0';
                sen_sym_key(sindex->lexicon, sid, keybuf, SEN_SYM_MAX_KEY_SIZE);
                if (sid == SEN_SYM_NIL) {
                  break;
                }
                s_array[*n] = (sym_array *) palloc(sizeof(s_array[*n]));
                s_array[*n]->id = sid;
                s_array[*n]->lexicon_array = (char *) palloc(strlen(keybuf) *
                                                             sizeof(char));
                if(sen_inv_cursor_open(sindex->inv , sid, 1)) {
                  s_array[*n]->flag = 1;
                } else {
                  s_array[*n]->flag = 0;
                }
                strcpy(s_array[*n]->lexicon_array, keybuf);
                sid = sen_sym_next(sindex->lexicon, sid);
                (*n)++;
              }
              sen_index_close(sindex);
              closedir(dir);
              return s_array;
            }
          }
        }
        break;
      case DT_DIR :
      case DT_LNK :
        {
          sym_array **tmp_s_array;
          tmp_s_array = do_dir_getlexicon(ent->d_name, n, s_array, filename);
          if (tmp_s_array != NULL) {
            closedir(dir);
            return tmp_s_array;
          }
          fchdir(fd);
        }
        break;
      default :
        break;
      }
    }
    closedir(dir);
  }
  return NULL;
}

Datum
pgs2getlexicon(PG_FUNCTION_ARGS)
{
  int n = 0;

  text *filename_a = (text*)PG_GETARG_TEXT_P(0);
  char *filename = NULL;
  FuncCallContext     *funcctx;
  int                  call_cntr;
  int                  max_calls;
  TupleDesc            tupdesc;
  AttInMetadata       *attinmeta;
  sym_array           **s_array = NULL;

  if (SRF_IS_FIRSTCALL()) {
    DIR *dir = opendir(".");
    MemoryContext   oldcontext;
    funcctx = SRF_FIRSTCALL_INIT();
    oldcontext = MemoryContextSwitchTo(funcctx->multi_call_memory_ctx);
    if (get_call_result_type(fcinfo, NULL, &tupdesc) != TYPEFUNC_COMPOSITE)
      ereport(ERROR,
              (errcode(ERRCODE_FEATURE_NOT_SUPPORTED),
               errmsg("function returning record called in context "
                      "that cannot accept type record")));
    attinmeta = TupleDescGetAttInMetadata(tupdesc);
    funcctx->attinmeta = attinmeta;

    if (dir) {
      int fd = dirfd(dir);
      filename = text2cstr(filename_a);
      s_array = do_dir_getlexicon(DataDir, &n, s_array, filename);
      fchdir(fd);
      closedir(dir);
    }
    funcctx->user_fctx = s_array;
    funcctx->max_calls = n;
    MemoryContextSwitchTo(oldcontext);
  }

  funcctx = SRF_PERCALL_SETUP();
  s_array = (sym_array **) funcctx->user_fctx;

  call_cntr = funcctx->call_cntr;
  max_calls = funcctx->max_calls;
  attinmeta = funcctx->attinmeta;
 
  if (call_cntr < max_calls) {
    int i;
    char       **values;
    HeapTuple    tuple;
    Datum        result;
    int len;

    len = strlen(s_array[call_cntr]->lexicon_array) + 1;
    values = (char **) palloc(3 * sizeof(char *));
    values[0] = (char *) palloc(16 * sizeof(char));
    values[1] = (char *) palloc(len * sizeof(char));
    values[2] = (char *) palloc(16 * sizeof(char));
    snprintf(values[0], 16, "%d", s_array[call_cntr]->id);
    snprintf(values[1], len, "%s", s_array[call_cntr]->lexicon_array);
    snprintf(values[2], 16, "%d", s_array[call_cntr]->flag);
    tuple = BuildTupleFromCStrings(attinmeta, values);
    result = HeapTupleGetDatum(tuple);
    for(i = 0; i < 2; i++) {
      pfree(values[i]);
    }
    pfree(values);
    SRF_RETURN_NEXT(funcctx, result);
  } else {
    SRF_RETURN_DONE(funcctx);
  }
}

Datum
pgs2version(PG_FUNCTION_ARGS)
{
  text *buffer;

  buffer = (text *) palloc(VARHDRSZ + strlen(PGS2_PACKAGE_STRING));
  VARATT_SIZEP(buffer) = VARHDRSZ + strlen(PGS2_PACKAGE_STRING);
  memcpy((char *)(VARDATA(buffer)), PGS2_PACKAGE_STRING,
         strlen(PGS2_PACKAGE_STRING));
  PG_RETURN_TEXT_P(buffer);
}

