/*
# LXCF - LXC Facility
# copyright (C) 2013-2014 FUJITSU LIMITED All Rights Reserved

# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; version 2
# of the License.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  
# 02110-1301, USA.
*/

/*
 * lxcf batch scheduler
 *	copyright (C) 2014 Fujitsu Limited.
 *
 */
 
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <time.h>
#include <sys/file.h>

#define	HQUEUE		"/var/tmp/lxcf/hqueue"
#define	QQUEUE		"/var/tmp/lxcf/qqueue"
#define	LQUEUE		"/var/tmp/lxcf/lqueue"

#define	LXCF_LOG	"/var/log/lxcf/lxcf-messages"
#define MAXBUF		4096

char	*buf;
char	jobbuf[MAXBUF];
char	cmdbuf[MAXBUF+5];

void cmd_exec(char *cmd)
{
	system(cmd);
}

char* date_time()
{
	time_t	timer;
	struct tm *timep;
	char	*s;

	time(&timer);
	timep = localtime(&timer);
	s = asctime(timep);
	if (s[strlen(s)-1] == '\n') {
		s[strlen(s)-1] = 0;
	}
	return s;
}


char* get_job_queue(char* filequeue)
{
	FILE *fdh;
	int  fh;
	int  i, j;
	int  maxqueue, c;

	// get job from QUEUE	
	fdh = fopen(filequeue, "r");
	if (fdh == NULL) {
		fprintf(stderr, "error: lxcf-sched : can't open %s\n", filequeue);
		exit(-1);
	}

	// lock QUEUE
	fh = fileno(fdh);
	flock(fh, LOCK_EX);

	// count jobfile lines
	for (maxqueue = 0;;) {
		if ((c = fgetc(fdh)) == '\n') {
			maxqueue++;
		} else if (c == EOF) {
			break;
		}
	}
	rewind(fdh);
	
	// The buffer where the job is put is allocated. 
	buf = (char*)malloc(maxqueue*MAXBUF);
	
	// get all QUEUE
	for (i = 0; i < maxqueue; i++) {
		if (fgets(&buf[i*MAXBUF], MAXBUF-1, fdh) == NULL) {
			buf[maxqueue*MAXBUF-1] = 0;
			break;
		}
		buf[(i+1)*MAXBUF-1] = 0;
	}
	
	// find job in QUEUE
	if (i != 0) {
		fdh = freopen(filequeue, "w", fdh);
		if (fdh == NULL) {
			fprintf(stderr, "error: lxcf-sched : can't re-open %s\n", filequeue);
			exit(-1);
		}
		
		for (j = 1; j < i; j++) {
			fprintf(fdh, "%s", &buf[j*MAXBUF]);
		}

		// unlock QUEUE
		flock(fh, LOCK_UN);
		fclose(fdh);

		// The first job is put in jobbuf. 
		strncpy(jobbuf, buf, MAXBUF-1);
		jobbuf[strlen(jobbuf)-1] = 0;
		
		// free buf
		free(buf);
		buf = NULL;
		
		return jobbuf;
	}
	
	// unlock QUEUE
	flock(fh, LOCK_UN);
	fclose(fdh);

	// free buf
	free(buf);

	// Because the job is not found, NULL is returned. 
	return NULL;
}

char *get_job()
{
	char *s;
	
	s = get_job_queue(HQUEUE);
	if (s != NULL) {
		printf("### %s H-QUEUE : %s ###\n", date_time(), s);
		fflush(stdout);
		fflush(stderr);
		return s;
	}

	s = get_job_queue(QQUEUE);
	if (s != NULL) {
		printf("### %s Q-QUEUE : %s ###\n", date_time(), s);
		fflush(stdout);
		fflush(stderr);
		return s;
	}

	s = get_job_queue(LQUEUE);
	if (s != NULL) {
		printf("### %s L-QUEUE : %s ###\n", date_time(), s);
		fflush(stdout);
		fflush(stderr);
		return s;
	}

	return NULL;
}

void launch_job()
{
	char 	*s;
	int	i;
	
	while(1) {
		sleep (1);

		// get a job from all queue
		s = get_job();
		if (s == NULL) {
			continue;
		}

		for (i = 0; i < strlen(s); i++) {
			if (s[i] == ' ') {
				snprintf(cmdbuf, MAXBUF+5-1, "lxcf %s", &s[i]);
				cmdbuf[MAXBUF+5-1] = 0;
				cmd_exec(cmdbuf);
				break;
			}
		}

		printf("### %s JOB END ###\n", date_time());
		fflush(stdout);
		fflush(stderr);
	}
}

void usage()
{
	fprintf(stderr, "usage : lxcf-sched\n");
	exit(-1);
}


int main(int argc, char **argv)
{
	// euid check
	if (geteuid() != 0) {
		fprintf(stderr, "error: Because you are not root, you cannot execute this command. \n");
		exit(-1);
	}

	// args check
	if (argc != 1) {
		usage();
	}

	// reopen stdout
	if (freopen(LXCF_LOG, "a", stdout) == NULL) {
		fprintf(stderr, "error: can't open %s\n",LXCF_LOG);
		exit(-1);
	}

	// reopen stderr
	if (freopen(LXCF_LOG, "a", stderr) == NULL) {
		fprintf(stderr, "error: can't open %s\n",LXCF_LOG);
		exit(-1);
	}
	
	launch_job();

	// close stdout & stderr
	fclose(stdout);
	fclose(stderr);
	
	exit(0);			
}

